//
// Copyright (c) 1997-2001 Tony Givargis. Permission to copy is
// granted provided that this header remains intact. This software
// is provided with no warranties.
//

//----------------------------------------------------------------------------

#ifndef mips_h
#define mips_h

//----------------------------------------------------------------------------

#include "cachesim.h"
#include "bussim.h"

//----------------------------------------------------------------------------

enum MipsReg {
    
    REG_ZR, REG_AT, REG_V0, REG_V1, REG_A0, REG_A1, REG_A2, REG_A3, REG_T0,
    REG_T1, REG_T2, REG_T3, REG_T4, REG_T5, REG_T6, REG_T7, REG_S0, REG_S1,
    REG_S2, REG_S3, REG_S4, REG_S5, REG_S6, REG_S7, REG_T8, REG_T9, REG_K0,
    REG_K1, REG_GP, REG_SP, REG_FP, REG_RA, REG_LO, REG_HI, REG_LAST
};

//----------------------------------------------------------------------------

enum MipsOpCode {
    
    OPC_ADD     ,  OPC_ADDI    ,  OPC_ADDIU   ,  OPC_ADDU    ,
    OPC_AND     ,  OPC_ANDI    ,  OPC_BC1T    ,  OPC_BC1F    ,
    OPC_BEQ     ,  OPC_BGEZ    ,  OPC_BGEZAL  ,  OPC_BGTZ    ,
    OPC_BLEZ    ,  OPC_BLTZ    ,  OPC_BLTZAL  ,  OPC_BNE     ,
    OPC_DIV     ,  OPC_DIVU    ,  OPC_J       ,  OPC_JAL     ,
    OPC_JALR    ,  OPC_JR      ,  OPC_LB      ,  OPC_LBU     ,
    OPC_LH      ,  OPC_LHU     ,  OPC_LW      ,  OPC_LUI     ,
    OPC_MFHI    ,  OPC_MFLO    ,  OPC_MTHI    ,  OPC_MTLO    ,
    OPC_MFC1    ,  OPC_MTC1    ,  OPC_MULT    ,  OPC_MULTU   ,
    OPC_NOOP    ,  OPC_NOR     ,  OPC_OR      ,  OPC_ORI     ,
    OPC_SB      ,  OPC_SH      ,  OPC_SLL     ,  OPC_SLLV    ,
    OPC_SLT     ,  OPC_SLTI    ,  OPC_SLTU    ,  OPC_SLTIU   ,
    OPC_SRA     ,  OPC_SRAV    ,  OPC_SRL     ,  OPC_SRLV    ,
    OPC_SUB     ,  OPC_SUBU    ,  OPC_SW      ,  OPC_XOR     ,
    OPC_XORI    ,  OPC_ABS_D   ,  OPC_ABS_S   ,  OPC_ADD_D   ,
    OPC_ADD_S   ,  OPC_C_EQ_D  ,  OPC_C_EQ_S  ,  OPC_C_LE_D  ,
    OPC_C_LE_S  ,  OPC_C_LT_D  ,  OPC_C_LT_S  ,  OPC_CVT_D_S ,
    OPC_CVT_D_W ,  OPC_CVT_S_D ,  OPC_CVT_S_W ,  OPC_CVT_W_D ,
    OPC_CVT_W_S ,  OPC_DIV_D   ,  OPC_DIV_S   ,  OPC_L_D     ,
    OPC_L_S     ,  OPC_MOV_D   ,  OPC_MOV_S   ,  OPC_MUL_D   ,
    OPC_MUL_S   ,  OPC_NEG_D   ,  OPC_NEG_S   ,  OPC_S_D     ,
    OPC_S_S     ,  OPC_SUB_D   ,  OPC_SUB_S   ,  OPC_LAST    
};

//----------------------------------------------------------------------------

struct MipsInstruction {
    
    MipsOpCode opCode;
    unsigned long im;
    unsigned char r1;
    unsigned char r2;
    unsigned char r3;
};

//----------------------------------------------------------------------------

class MipsVirtualMachine {
    
public:
    double cpuPower;
    double iCachePower;
    double dCachePower;
    double cpuICacheBusPower;
    double cpuDCacheBusPower;
    double cacheMemBusPower;
    double memPower;
    
public:
    double power;
    double exeTime;
    
public:
    unsigned long instExecFreq[OPC_LAST];
    
public:
    CacheSimulator iCache;
    CacheSimulator dCache;
    BusSimulator cpuICacheBus;
    BusSimulator cpuDCacheBus;
    BusSimulator cacheMemBus;
    double voltage;
    
private:
    unsigned szDataSeg;
    unsigned szCodeSeg;
    unsigned char *dataSeg;
    unsigned char *copyOfDataSeg;
    MipsInstruction *codeSeg;
    
private:
    void *loader;
    
private:
    unsigned szMemory;
    unsigned char *memory;
    unsigned long pc;
    unsigned long regI[34];
    float regF[32];
    bool floatCondFlag;
    MipsInstruction *ip;
    
private:
    unsigned timer;
    
public:
    MipsVirtualMachine(unsigned _szDataSeg,
                       unsigned _szCodeSeg,
                       unsigned char *_dataSeg,
                       unsigned char *_copyOfDatSeg,
                       MipsInstruction *_codeSeg,
                       void *_loader);
    ~MipsVirtualMachine();
    
public:
    bool Reset(int is, int il, int ia,
               int ds, int dl, int da,
               int ciaw, Encoding ciae,
               int cidw, Encoding cide,
               int cdaw, Encoding cdae,
               int cddw, Encoding cdde,
               int cmaw, Encoding cmae,
               int cmdw, Encoding cmde,
               double _voltage);
    
public:
    double ComputePower();
    
public:
    void Execute();
    
private:
    void FetchDecode();
    
private:
    unsigned char ReadByte(unsigned long a);
    unsigned short ReadHalf(unsigned long a);
    unsigned long ReadWord(unsigned long a);
    void WriteByte(unsigned long a, unsigned char d);
    void WriteHalf(unsigned long a, unsigned short d);
    void WriteWord(unsigned long a, unsigned long d);
    
private:
    void Execute_ADD();
    void Execute_ADDI();
    void Execute_ADDIU();
    void Execute_ADDU();
    void Execute_AND();
    void Execute_ANDI();
    void Execute_BC1T();
    void Execute_BC1F();
    void Execute_BEQ();
    void Execute_BGEZ();
    void Execute_BGEZAL();
    void Execute_BGTZ();
    void Execute_BLEZ();
    void Execute_BLTZ();
    void Execute_BLTZAL();
    void Execute_BNE();
    void Execute_DIV();
    void Execute_DIVU();
    void Execute_J();
    void Execute_JAL();
    void Execute_JALR();
    void Execute_JR();
    void Execute_LB();
    void Execute_LBU();
    void Execute_LH();
    void Execute_LHU();
    void Execute_LW();
    void Execute_LUI();
    void Execute_MFHI();
    void Execute_MFLO();
    void Execute_MTHI();
    void Execute_MTLO();
    void Execute_MFC1();
    void Execute_MTC1();
    void Execute_MULT();
    void Execute_MULTU();
    void Execute_NOOP();
    void Execute_NOR();
    void Execute_OR();
    void Execute_ORI();
    void Execute_SB();
    void Execute_SH();
    void Execute_SLL();
    void Execute_SLLV();
    void Execute_SLT();
    void Execute_SLTI();
    void Execute_SLTU();
    void Execute_SLTIU();
    void Execute_SRA();
    void Execute_SRAV();
    void Execute_SRL();
    void Execute_SRLV();
    void Execute_SUB();
    void Execute_SUBU();
    void Execute_SW();
    void Execute_XOR();
    void Execute_XORI();
    
private:
    void Execute_ABS_D();
    void Execute_ABS_S();
    void Execute_ADD_D();
    void Execute_ADD_S();
    void Execute_C_EQ_D();
    void Execute_C_EQ_S();
    void Execute_C_LE_D();
    void Execute_C_LE_S();
    void Execute_C_LT_D();
    void Execute_C_LT_S();
    void Execute_CVT_D_S();
    void Execute_CVT_D_W();
    void Execute_CVT_S_D();
    void Execute_CVT_S_W();
    void Execute_CVT_W_D();
    void Execute_CVT_W_S();
    void Execute_DIV_D();
    void Execute_DIV_S();
    void Execute_L_D();
    void Execute_L_S();
    void Execute_MOV_D();
    void Execute_MOV_S();
    void Execute_MUL_D();
    void Execute_MUL_S();
    void Execute_NEG_D();
    void Execute_NEG_S();
    void Execute_S_D();
    void Execute_S_S();
    void Execute_SUB_D();
    void Execute_SUB_S();
};

//----------------------------------------------------------------------------

#endif