//
// Copyright (c) 1997-2001 Tony Givargis. Permission to copy is
// granted provided that this header remains intact. This software
// is provided with no warranties.
//

//----------------------------------------------------------------------------

#include <cassert>
#include <cstring>
#include <cstdlib>
#include <cmath>
#include "tech.h"
#include "loader.h"
#include "mipsvm.h"

//----------------------------------------------------------------------------

#define BYTE(m, a)       *reinterpret_cast< char*           >((m) + (a))
#define UBYTE(m, a)      *reinterpret_cast< unsigned char*  >((m) + (a))
#define HALF(m, a)       *reinterpret_cast< short*          >((m) + (a))
#define UHALF(m, a)      *reinterpret_cast< unsigned short* >((m) + (a))
#define WORD(m, a)       *reinterpret_cast< long*           >((m) + (a))

//----------------------------------------------------------------------------

MipsVirtualMachine::MipsVirtualMachine(unsigned _szDataSeg,
                                       unsigned _szCodeSeg,
                                       unsigned char *_dataSeg,
                                       unsigned char *_copyOfDataSeg,
                                       MipsInstruction *_codeSeg,
                                       void *_loader) {
    szDataSeg = _szDataSeg;
    szCodeSeg = _szCodeSeg;
    
    dataSeg = _dataSeg;
    copyOfDataSeg = _copyOfDataSeg;
    codeSeg = _codeSeg;
    
    loader = _loader;
    
    assert( szDataSeg && szCodeSeg );
    assert( dataSeg && codeSeg && copyOfDataSeg );
    assert( loader );
    
    Reset(
        16384, 16, 1, 
        16384, 16, 1, 
        32, Binary, 32, Binary, 
        32, Binary, 32, Binary,
        32, Binary, 32, Binary,
        3.3);
}

//----------------------------------------------------------------------------

MipsVirtualMachine::~MipsVirtualMachine() {
}

//----------------------------------------------------------------------------

bool MipsVirtualMachine::Reset(int is, int il, int ia,
                               int ds, int dl, int da,
                               int ciaw, Encoding ciae,
                               int cidw, Encoding cide,
                               int cdaw, Encoding cdae,
                               int cddw, Encoding cdde,
                               int cmaw, Encoding cmae,
                               int cmdw, Encoding cmde,
                               double _voltage) {
    
    if( is >= (il * ia) && ds >= (dl * da) ) {
        
        memset(instExecFreq, 0, sizeof(instExecFreq));
        srand(0);
        iCache.Reset(is, il, ia, _voltage);
        dCache.Reset(ds, dl, da, _voltage);
        cpuICacheBus.Reset(ciaw, ciae, cidw, cide, _voltage);
        cpuDCacheBus.Reset(cdaw, cdae, cddw, cdde, _voltage);
        cacheMemBus.Reset(cmaw, cmae, cmdw, cmde, _voltage);
        voltage = _voltage;
        
        memcpy(dataSeg, copyOfDataSeg, sizeof(unsigned char) * szDataSeg);
        return true;
    }
    return false;
}

//----------------------------------------------------------------------------

double MipsVirtualMachine::ComputePower() {
    
    double p;
    
    // per instruction power
    p = 0.0;
    for(int i=0; i<OPC_LAST; i++) {
        
        switch( static_cast<MipsOpCode>(i) ) {
            
        case OPC_LB:  case OPC_LBU:
        case OPC_LH:  case OPC_LHU:
        case OPC_LW:  case OPC_LUI:
        case OPC_SB:  case OPC_SH:
        case OPC_SW:  case OPC_L_D:
        case OPC_L_S: case OPC_S_D:
        case OPC_S_S: 
            p += instExecFreq[i] * 3;
            break;
            
        default:
            p += instExecFreq[i] * 2;
            break;
        }
    }
    
    // account for stalls
    p += iCache.numReadMiss * 0.5;
    p += dCache.numReadMiss * 0.5;
    p += dCache.numWriteMiss * 0.5;
    
    p = p * 1024.0 * CapTran;
    return p * voltage * voltage * .5;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute() {
    
    unsigned long mr, mw;
    
    regI[0] = 0;
    regF[0] = 0;
    pc = 0;
    timer = 0;
    
    do {
        
        FetchDecode();
        instExecFreq[ip->opCode]++;
        timer++;
        
        switch( ip->opCode ) {
            
        case OPC_ADD:     Execute_ADD();     break;
        case OPC_ADDI:    Execute_ADDI();    break;
        case OPC_ADDIU:   Execute_ADDIU();   break;
        case OPC_ADDU:    Execute_ADDU();    break;
        case OPC_AND:     Execute_AND();     break;
        case OPC_ANDI:    Execute_ANDI();    break;
        case OPC_BC1T:    Execute_BC1T();    break;
        case OPC_BC1F:    Execute_BC1F();    break;
        case OPC_BEQ:     Execute_BEQ();     break;
        case OPC_BGEZ:    Execute_BGEZ();    break;
        case OPC_BGEZAL:  Execute_BGEZAL();  break;
        case OPC_BGTZ:    Execute_BGTZ();    break;
        case OPC_BLEZ:    Execute_BLEZ();    break;
        case OPC_BLTZ:    Execute_BLTZ();    break;
        case OPC_BLTZAL:  Execute_BLTZAL();  break;
        case OPC_BNE:     Execute_BNE();     break;
        case OPC_DIV:     Execute_DIV();     break;
        case OPC_DIVU:    Execute_DIVU();    break;
        case OPC_J:       Execute_J();       break;
        case OPC_JAL:     Execute_JAL();     break;
        case OPC_JALR:    Execute_JALR();    break;
        case OPC_JR:      Execute_JR();      break;
        case OPC_LB:      Execute_LB();      break;
        case OPC_LBU:     Execute_LBU();     break;
        case OPC_LH:      Execute_LH();      break;
        case OPC_LHU:     Execute_LHU();     break;
        case OPC_LW:      Execute_LW();      break;
        case OPC_LUI:     Execute_LUI();     break;
        case OPC_MFHI:    Execute_MFHI();    break;
        case OPC_MFLO:    Execute_MFLO();    break;
        case OPC_MTHI:    Execute_MTHI();    break;
        case OPC_MTLO:    Execute_MTLO();    break;
        case OPC_MFC1:    Execute_MFC1();    break;
        case OPC_MTC1:    Execute_MTC1();    break;
        case OPC_MULT:    Execute_MULT();    break;
        case OPC_MULTU:   Execute_MULTU();   break;
        case OPC_NOOP:    Execute_NOOP();    break;
        case OPC_NOR:     Execute_NOR();     break;
        case OPC_OR:      Execute_OR();      break;
        case OPC_ORI:     Execute_ORI();     break;
        case OPC_SB:      Execute_SB();      break;
        case OPC_SH:      Execute_SH();      break;
        case OPC_SLL:     Execute_SLL();     break;
        case OPC_SLLV:    Execute_SLLV();    break;
        case OPC_SLT:     Execute_SLT();     break;
        case OPC_SLTI:    Execute_SLTI();    break;
        case OPC_SLTU:    Execute_SLTU();    break;
        case OPC_SLTIU:   Execute_SLTIU();   break;
        case OPC_SRA:     Execute_SRA();     break;
        case OPC_SRAV:    Execute_SRAV();    break;
        case OPC_SRL:     Execute_SRL();     break;
        case OPC_SRLV:    Execute_SRLV();    break;
        case OPC_SUB:     Execute_SUB();     break;
        case OPC_SUBU:    Execute_SUBU();    break;
        case OPC_SW:      Execute_SW();      break;
        case OPC_XOR:     Execute_XOR();     break;
        case OPC_XORI:    Execute_XORI();    break;
        case OPC_ABS_S:   Execute_ABS_S();   break;
        case OPC_ABS_D:   Execute_ABS_D();   break;
        case OPC_ADD_D:   Execute_ADD_D();   break;
        case OPC_ADD_S:   Execute_ADD_S();   break;
        case OPC_C_EQ_D:  Execute_C_EQ_D();  break;
        case OPC_C_EQ_S:  Execute_C_EQ_S();  break;
        case OPC_C_LE_D:  Execute_C_LE_D();  break;
        case OPC_C_LE_S:  Execute_C_LE_S();  break;
        case OPC_C_LT_D:  Execute_C_LT_D();  break;
        case OPC_C_LT_S:  Execute_C_LT_S();  break;
        case OPC_CVT_D_S: Execute_CVT_D_S(); break;
        case OPC_CVT_D_W: Execute_CVT_D_W(); break;
        case OPC_CVT_S_D: Execute_CVT_S_D(); break;
        case OPC_CVT_S_W: Execute_CVT_S_W(); break;
        case OPC_CVT_W_D: Execute_CVT_W_D(); break;
        case OPC_CVT_W_S: Execute_CVT_W_S(); break;
        case OPC_DIV_D:   Execute_DIV_D();   break;
        case OPC_DIV_S:   Execute_DIV_S();   break;
        case OPC_L_D:     Execute_L_D();     break;
        case OPC_L_S:     Execute_L_S();     break;
        case OPC_MOV_D:   Execute_MOV_D();   break;
        case OPC_MOV_S:   Execute_MOV_S();   break;
        case OPC_MUL_D:   Execute_MUL_D();   break;
        case OPC_MUL_S:   Execute_MUL_S();   break;
        case OPC_NEG_D:   Execute_NEG_D();   break;
        case OPC_NEG_S:   Execute_NEG_S();   break;
        case OPC_S_D:     Execute_S_D();     break;
        case OPC_S_S:     Execute_S_S();     break;
        case OPC_SUB_D:   Execute_SUB_D();   break;
        case OPC_SUB_S:   Execute_SUB_S();   break;
        default: assert( false );
        }
    }
    while( pc != 0 );
    
    // memory traffic
    mr  = (iCache.numReadMiss * (iCache.line >> 2));
    mr += (dCache.numReadMiss * (dCache.line >> 2));
    mw  = (dCache.numWriteBack * (dCache.line >> 2));
    
    // compute energy
    cpuPower = ComputePower();
    iCachePower = iCache.ComputePower();
    dCachePower = dCache.ComputePower();
    cpuICacheBusPower = cpuICacheBus.ComputePower();
    cpuDCacheBusPower = cpuDCacheBus.ComputePower();
    cacheMemBusPower = cacheMemBus.ComputePower();
    memPower  = ComputeMemoryPower(mr + mw, voltage);
    
    // compute execution time
    exeTime  = iCache.numRead;
    exeTime += iCache.numReadMiss * 20;
    exeTime += dCache.numReadMiss * 20;
    exeTime += dCache.numWriteMiss * 20;
    exeTime += (32 / cpuICacheBus.addrState.width) * iCache.numRead;
    exeTime += (32 / cpuICacheBus.dataState.width) * iCache.numRead;
    exeTime += (32 / cpuDCacheBus.addrState.width) * dCache.numRead;
    exeTime += (32 / cpuDCacheBus.dataState.width) * dCache.numRead;
    exeTime += (32 / cpuDCacheBus.addrState.width) * dCache.numWrite;
    exeTime += (32 / cpuDCacheBus.dataState.width) * dCache.numWrite;
    exeTime += (32 / cacheMemBus.addrState.width) * iCache.numReadMiss;
    exeTime += (32 / cacheMemBus.dataState.width) * iCache.numReadMiss;
    exeTime += (32 / cacheMemBus.addrState.width) * dCache.numReadMiss;
    exeTime += (32 / cacheMemBus.dataState.width) * dCache.numReadMiss;
    exeTime += (32 / cacheMemBus.addrState.width) * dCache.numWriteBack;
    exeTime += (32 / cacheMemBus.dataState.width) * dCache.numWriteBack;
    exeTime /= ComputeClockSpeed(voltage);
    
    // compute power
    power  = (cpuPower          /= exeTime);
    power += (iCachePower       /= exeTime);
    power += (dCachePower       /= exeTime);
    power += (cpuICacheBusPower /= exeTime);
    power += (cpuDCacheBusPower /= exeTime);
    power += (cacheMemBusPower  /= exeTime);
    power += (memPower          /= exeTime);
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::FetchDecode() {
    
    unsigned i, j, n;
    
    cpuICacheBus.Transfer(pc);
    n = iCache.Read(pc << 2);
    for(j=0; j<n; j++) {
        
        for(i=0; i<(iCache.line>>2); i++) {
            
            cacheMemBus.Transfer(pc + (i << 2));
        }
    }
    ip = &codeSeg[pc++];
}

//----------------------------------------------------------------------------

unsigned char MipsVirtualMachine::ReadByte(unsigned long a) {
    
    unsigned i, j, n;
    unsigned short d = 0;
    
    if( a < szDataSeg ) {
        
        d = BYTE(dataSeg, a);
    }
    else {
        
        assert( false );
    }
    
    cpuDCacheBus.Transfer(a);
    n = dCache.Read(a);
    for(j=0; j<n; j++) {
        
        for(i=0; i<(dCache.line>>2); i++) {
            
            cacheMemBus.Transfer(a + (i << 2));
        }
    }
    return d;
}

//----------------------------------------------------------------------------

unsigned short MipsVirtualMachine::ReadHalf(unsigned long a) {
    
    unsigned i, j, n;
    unsigned short d = 0;
    
    if( a < szDataSeg ) {
        
        d = HALF(dataSeg, a);
    }
    else {
        
        assert( false );
    }
    
    cpuDCacheBus.Transfer(a);
    n = dCache.Read(a);
    for(j=0; j<n; j++) {
        
        for(i=0; i<(dCache.line>>2); i++) {
            
            cacheMemBus.Transfer(a + (i << 2));
        }
    }
    return d;
}

//----------------------------------------------------------------------------

unsigned long MipsVirtualMachine::ReadWord(unsigned long a) {
    
    unsigned i, j, n;
    unsigned long d = 0;
    
    if( a < szDataSeg ) {
        
        d = WORD(dataSeg, a);
    }
    else {
        
        assert( false );
    }
    
    cpuDCacheBus.Transfer(a);       
    n = dCache.Read(a);
    for(j=0; j<n; j++) {
        
        for(i=0; i<(dCache.line>>2); i++) {
            
            cacheMemBus.Transfer(a + (i << 2));
        }
    }
    return d;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::WriteByte(unsigned long a, unsigned char d) {
    
    unsigned i, j, n;
    
    if( a < szDataSeg ) {
        
        BYTE(dataSeg, a) = d;
    }
    else {
        
        static_cast<Loader*>(loader)->Output(d);
    }
    
    cpuDCacheBus.Transfer(a);
    n = dCache.Write(a);
    for(j=0; j<n; j++) {
        
        for(i=0; i<(dCache.line>>2); i++) {
            
            cacheMemBus.Transfer(a + (i << 2));
        }
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::WriteHalf(unsigned long a, unsigned short d) {
    
    unsigned i, j, n;
    
    if( a < szDataSeg ) {
        
        HALF(dataSeg, a) = d;
    }
    else {
        
        assert( false );
    }
    
    cpuDCacheBus.Transfer(a);       
    n = dCache.Write(a);
    for(j=0; j<n; j++) {
        
        for(i=0; i<(dCache.line>>2); i++) {
            
            cacheMemBus.Transfer(a + (i << 2));
        }
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::WriteWord(unsigned long a, unsigned long d) {
    
    unsigned i, j, n;
    
    if( a < szDataSeg ) {
        
        WORD(dataSeg, a) = d;
    }
    else {
        
        assert( false );
    }
    
    cpuDCacheBus.Transfer(a);       
    n = dCache.Write(a);
    for(j=0; j<n; j++) {
        
        for(i=0; i<(dCache.line>>2); i++) {
            
            cacheMemBus.Transfer(a + (i << 2));
        }
    }
}

//----------------------------------------------------------------------------
// Integer instructions follow...
//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_ADD() {
    
    regI[ip->r1] = regI[ip->r2] + regI[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_ADDI() {
    
    regI[ip->r1] = regI[ip->r2] + ip->im;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_ADDIU() {
    
    regI[ip->r1] = regI[ip->r2] + ip->im;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_ADDU() {
    
    regI[ip->r1] = regI[ip->r2] + regI[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_AND() {
    
    regI[ip->r1] = regI[ip->r2] & regI[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_ANDI() {
    
    regI[ip->r1] = regI[ip->r2] & ip->im;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_BC1T() {
    
    if( floatCondFlag ) {
        
        pc = ip->im;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_BC1F() {
    
    if( !floatCondFlag ) {
        
        pc = ip->im;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_BEQ() {
    
    if( regI[ip->r1] == regI[ip->r2] ) {
        
        pc += ip->im;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_BGEZ() {
    
    if( static_cast<long>(regI[ip->r1]) >= 0 ) {
        
        pc += ip->im;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_BGEZAL() {
    
    if( static_cast<long>(regI[ip->r1]) >= 0 ) {
        
        regI[REG_RA] = pc;
        pc += ip->im;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_BGTZ() {
    
    if( static_cast<long>(regI[ip->r1]) > 0 ) {
        
        pc += ip->im;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_BLEZ() {
    
    if( static_cast<long>(regI[ip->r1]) <= 0 ) {
        
        pc += ip->im;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_BLTZ() {
    
    if( static_cast<long>(regI[ip->r1]) < 0 ) {
        
        pc += ip->im;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_BLTZAL() {
    
    if( static_cast<long>(regI[ip->r1]) < 0 ) {
        
        regI[REG_RA] = pc;
        pc += ip->im;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_BNE() {
    
    if( regI[ip->r1] != regI[ip->r2] ) {
        
        pc += ip->im;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_DIV() {
    
    regI[REG_LO] = regI[ip->r1] / regI[ip->r2];
    regI[REG_HI] = regI[ip->r1] % regI[ip->r2];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_DIVU() {
    
    regI[REG_LO] = regI[ip->r1] / regI[ip->r2];
    regI[REG_HI] = regI[ip->r1] % regI[ip->r2];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_J() {
    
    pc = ip->im;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_JAL() {
    
    regI[REG_RA] = pc;
    pc = ip->im;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_JALR() {
    
    regI[REG_RA] = pc;
    pc = regI[ip->r1];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_JR() {
    
    pc = regI[ip->r1];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_LB() {
    
    unsigned long a;
    
    a = regI[ip->r2] + ip->im;
    regI[ip->r1] = static_cast<char>(ReadByte(a));
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_LBU() {
    
    unsigned long a;
    
    a = regI[ip->r2] + ip->im;
    regI[ip->r1] = ReadByte(a);
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_LH() {
    
    unsigned long a;
    
    a = regI[ip->r2] + ip->im;
    regI[ip->r1] = static_cast<short>(ReadHalf(a));
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_LHU() {
    
    unsigned long a;
    
    a = regI[ip->r2] + ip->im;
    regI[ip->r1] = ReadHalf(a);
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_LW() {
    
    unsigned long a;
    
    a = regI[ip->r2] + ip->im;
    regI[ip->r1] = static_cast<long>(ReadWord(a));
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_LUI() {
    
    regI[ip->r1] = (ip->im & 0x0000ffff) << 16;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MFHI() {
    
    regI[ip->r1] = regI[REG_HI];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MFLO() {
    
    regI[ip->r1] = regI[REG_LO];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MTHI() {
    
    regI[REG_HI] = regI[ip->r1];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MTLO() {
    
    regI[REG_LO] = regI[ip->r1];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MFC1() {
    
    union { float v1; unsigned long v2; } v;
    
    v.v1 = regF[ip->r2];
    regI[ip->r1] = v.v2;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MTC1() {
    
    union { unsigned long v1; float v2; } v;
    
    v.v1 = regI[ip->r1];
    regF[ip->r2] = v.v2;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MULT() {
    
    regI[REG_LO] = regI[ip->r1] * regI[ip->r2];
    regI[REG_HI] = 0;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MULTU() {
    
    regI[REG_LO] = regI[ip->r1] * regI[ip->r2];
    regI[REG_HI] = 0;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_NOOP() {
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_NOR() {
    
    regI[ip->r1] = !(regI[ip->r2] | regI[ip->r3]);
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_OR() {
    
    regI[ip->r1] = regI[ip->r2] | regI[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_ORI() {
    
    regI[ip->r1] = regI[ip->r2] | ip->im;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SB() {
    
    unsigned long a;
    unsigned char d;
    
    a = regI[ip->r2] + ip->im;
    d = static_cast<char>(regI[ip->r1]);
    WriteByte(a, static_cast<char>(d));
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SH() {
    
    unsigned long a;
    unsigned short d;
    
    a = regI[ip->r2] + ip->im;
    d = static_cast<short>(regI[ip->r1]);
    WriteHalf(a, static_cast<short>(d));
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SLL() {
    
    regI[ip->r1] = regI[ip->r2] << ip->im;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SLLV() {
    
    regI[ip->r1] = regI[ip->r2] << regI[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SLT() {
    
    if( static_cast<long>(regI[ip->r2]) < 
        static_cast<long>(regI[ip->r3]) ) {
        
        regI[ip->r1] = 1;
    }
    else {
        
        regI[ip->r1] = 0;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SLTI() {
    
    if( static_cast<long>(regI[ip->r2]) < 
        static_cast<long>(ip->im) ) {
        
        regI[ip->r1] = 1;
    }
    else {
        
        regI[ip->r1] = 0;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SLTU() {
    
    if( static_cast<unsigned long>(regI[ip->r2]) < 
        static_cast<unsigned long>(regI[ip->r3]) ) {
        
        regI[ip->r1] = 1;
    }
    else {
        
        regI[ip->r1] = 0;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SLTIU() {
    
    if( static_cast<unsigned long>(regI[ip->r2]) < 
        static_cast<unsigned long>(ip->im) ) {
        
        regI[ip->r1] = 1;
    }
    else {
        
        regI[ip->r1] = 0;
    }
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SRA() {
    
    regI[ip->r1] = static_cast<long>(regI[ip->r2]) >> ip->im;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SRAV() {
    
    regI[ip->r1] = static_cast<long>(regI[ip->r2]) >> regI[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SRL() {
    
    regI[ip->r1] = regI[ip->r2] >> ip->im;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SRLV() {
    
    regI[ip->r1] = regI[ip->r2] >> regI[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SUB() {
    
    regI[ip->r1] = regI[ip->r2] - regI[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SUBU() {
    
    regI[ip->r1] = regI[ip->r2] - regI[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SW() {
    
    unsigned long a;
    unsigned long d;
    
    a = regI[ip->r2] + ip->im;
    d = static_cast<long>(regI[ip->r1]);
    WriteWord(a, static_cast<long>(d));
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_XOR() {
    
    regI[ip->r1] = regI[ip->r2] ^ regI[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_XORI() {
    
    regI[ip->r1] = regI[ip->r2] ^ ip->im;
}

//----------------------------------------------------------------------------
// Floating point instructions follow...
//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_ABS_D() {
    
    union { struct { float l, h; } v1; double v2; } v;
    
    v.v1.l = regF[ip->r2    ];
    v.v1.h = regF[ip->r2 + 1];
    v.v2 = fabs(v.v2);
    regF[ip->r1    ] = v.v1.l;
    regF[ip->r1 + 1] = v.v1.h;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_ABS_S() {
    
    regF[ip->r1] = static_cast<float>(fabs(regF[ip->r2]));
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_ADD_D() {
    
    union { struct { float l, h; } v1; double v2; } v, v1, v2;
    
    v1.v1.l = regF[ip->r2    ];
    v1.v1.h = regF[ip->r2 + 1];
    
    v2.v1.l = regF[ip->r3    ];
    v2.v1.h = regF[ip->r3 + 1];
    
    v.v2 = v1.v2 + v2.v2;
    
    regF[ip->r1    ] = v.v1.l;
    regF[ip->r1 + 1] = v.v1.h;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_ADD_S() {
    
    regF[ip->r1] = regF[ip->r2] + regF[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_C_EQ_D() {
    
    union { struct { float l, h; } v1; double v2; } v1, v2;
    
    v1.v1.l = regF[ip->r1    ];
    v1.v1.h = regF[ip->r1 + 1];
    
    v2.v1.l = regF[ip->r2    ];
    v2.v1.h = regF[ip->r2 + 1];
    
    floatCondFlag = v1.v2 == v2.v2;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_C_EQ_S() {
    
    floatCondFlag = regF[ip->r1] == regF[ip->r2];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_C_LE_D() {
    
    union { struct { float l, h; } v1; double v2; } v1, v2;
    
    v1.v1.l = regF[ip->r1    ];
    v1.v1.h = regF[ip->r1 + 1];
    
    v2.v1.l = regF[ip->r2    ];
    v2.v1.h = regF[ip->r2 + 1];
    
    floatCondFlag = v1.v2 <= v2.v2;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_C_LE_S() {
    
    floatCondFlag = regF[ip->r1] <= regF[ip->r2];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_C_LT_D() {
    
    union { struct { float l, h; } v1; double v2; } v1, v2;
    
    v1.v1.l = regF[ip->r1    ];
    v1.v1.h = regF[ip->r1 + 1];
    
    v2.v1.l = regF[ip->r2    ];
    v2.v1.h = regF[ip->r2 + 1];
    
    floatCondFlag = v1.v2 < v2.v2;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_C_LT_S() {
    
    floatCondFlag = regF[ip->r1] < regF[ip->r2];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_CVT_D_S() {
    
    union { struct { float l, h; } v1; double v2; } v;
    
    v.v2 = regF[ip->r2];
    
    regF[ip->r1    ] = v.v1.l;
    regF[ip->r1 + 1] = v.v1.h;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_CVT_D_W() {
    
    union { struct { float l, h; } v1; double v2; } v;
    
    v.v2 = *(reinterpret_cast<int*>(&regF[ip->r2]));
    
    regF[ip->r1    ] = v.v1.l;
    regF[ip->r1 + 1] = v.v1.h;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_CVT_S_D() {
    
    union { struct { float l, h; } v1; double v2; } v;
    
    v.v1.l = regF[ip->r2    ];
    v.v1.h = regF[ip->r2 + 1];
    
    regF[ip->r1] = static_cast<float>(v.v2);
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_CVT_S_W() {
    
    int t;
    
    t = *(reinterpret_cast<int*>(&regF[ip->r2]));
    regF[ip->r1] = static_cast<float>(t);
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_CVT_W_D() {
    
    union { struct { float l, h; } v1; double v2; } v;
    int t;
    
    v.v1.l = regF[ip->r2    ];
    v.v1.h = regF[ip->r2 + 1];
    t = static_cast<int>(v.v2);
    
    *(reinterpret_cast<int*>(&regF[ip->r1])) = t;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_CVT_W_S() {
    
    int t;
    
    t = static_cast<int>(regF[ip->r2]);
    *(reinterpret_cast<int*>(&regF[ip->r1])) = t;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_DIV_D() {
    
    union { struct { float l, h; } v1; double v2; } v, v1, v2;
    
    v1.v1.l = regF[ip->r2    ];
    v1.v1.h = regF[ip->r2 + 1];
    
    v2.v1.l = regF[ip->r3    ];
    v2.v1.h = regF[ip->r3 + 1];
    
    v.v2 = v1.v2 / v2.v2;
    
    regF[ip->r1    ] = v.v1.l;
    regF[ip->r1 + 1] = v.v1.h;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_DIV_S() {
    
    regF[ip->r1] = regF[ip->r2] / regF[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_L_D() {
    
    union { struct { unsigned long l, h; } v1; struct { float l, h; } v2; } v;
    
    v.v1.l = ReadWord(regI[ip->r2] + ip->im);
    v.v1.h = ReadWord(regI[ip->r2] + ip->im + 4);
    
    regF[ip->r1    ] = v.v2.l;
    regF[ip->r1 + 1] = v.v2.h;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_L_S() {
    
    union { float v1; unsigned long v2; } v;
    
    v.v2 = ReadWord(regI[ip->r2] + ip->im);
    regF[ip->r1] = v.v1;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MOV_D() {
    
    regF[ip->r1    ] = regF[ip->r2    ];
    regF[ip->r1 + 1] = regF[ip->r2 + 1];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MOV_S() {
    
    regF[ip->r1] = regF[ip->r2];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MUL_D() {
    
    union { struct { float l, h; } v1; double v2; } v, v1, v2;
    
    v1.v1.l = regF[ip->r2    ];
    v1.v1.h = regF[ip->r2 + 1];
    
    v2.v1.l = regF[ip->r3    ];
    v2.v1.h = regF[ip->r3 + 1];
    
    v.v2 = v1.v2 * v2.v2;
    
    regF[ip->r1    ] = v.v1.l;
    regF[ip->r1 + 1] = v.v1.h;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_MUL_S() {
    
    regF[ip->r1] = regF[ip->r2] * regF[ip->r3];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_NEG_D() {
    
    union { struct { float l, h; } v1; double v2; } v, v1;
    
    v1.v1.l = regF[ip->r2    ];
    v1.v1.h = regF[ip->r2 + 1];
    
    v.v2 = -v1.v2;
    
    regF[ip->r1    ] = v.v1.l;
    regF[ip->r1 + 1] = v.v1.h;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_NEG_S() {
    
    regF[ip->r1] = -regF[ip->r2];
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_S_D() {
    
    union { struct { float l, h; } v1; struct { unsigned long l, h; } v2; } v;
    
    v.v1.l = regF[ip->r1    ];
    v.v1.h = regF[ip->r1 + 1];
    WriteWord(regI[ip->r2] + ip->im    , v.v2.l);
    WriteWord(regI[ip->r2] + ip->im + 4, v.v2.h);
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_S_S() {
    
    union { float v1; unsigned long v2; } v;
    
    v.v1 = regF[ip->r1];
    WriteWord(regI[ip->r2] + ip->im, v.v2);
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SUB_D() {
    
    union { struct { float l, h; } v1; double v2; } v, v1, v2;
    
    v1.v1.l = regF[ip->r2    ];
    v1.v1.h = regF[ip->r2 + 1];
    
    v2.v1.l = regF[ip->r3    ];
    v2.v1.h = regF[ip->r3 + 1];
    
    v.v2 = v1.v2 - v2.v2;
    
    regF[ip->r1    ] = v.v1.l;
    regF[ip->r1 + 1] = v.v1.h;
}

//----------------------------------------------------------------------------

void MipsVirtualMachine::Execute_SUB_S() {
    
    regF[ip->r1] = regF[ip->r2] - regF[ip->r3];
}