//
// Copyright (c) 1997-2001 Tony Givargis. Permission to copy is
// granted provided that this header remains intact. This software
// is provided with no warranties.
//

//----------------------------------------------------------------------------

#include <iostream>
#include <fstream>
#include <cstdio>
#include <cassert>
#include <cstring>
#include "mipsasm.h"
using namespace std;

//----------------------------------------------------------------------------

#define BYTE(m, a)       *reinterpret_cast< char*           >((m) + (a))
#define UBYTE(m, a)      *reinterpret_cast< unsigned char*  >((m) + (a))
#define HALF(m, a)       *reinterpret_cast< short*          >((m) + (a))
#define UHALF(m, a)      *reinterpret_cast< unsigned short* >((m) + (a))
#define WORD(m, a)       *reinterpret_cast< long*           >((m) + (a))

//----------------------------------------------------------------------------

MipsAssembler::MipsAssembler(unsigned _szDataSeg, unsigned _szCodeSeg) {
    
    szDataSeg = _szDataSeg;
    szCodeSeg = _szCodeSeg;
    dataSeg = new unsigned char[szDataSeg + 4];
    copyOfDataSeg = new unsigned char[szDataSeg + 4];
    codeSeg = new MipsInstruction[szCodeSeg + 1];
    
    assert( dataSeg && copyOfDataSeg && codeSeg );
}

//----------------------------------------------------------------------------

MipsAssembler::~MipsAssembler() {
    
    delete[] dataSeg;
    delete[] codeSeg;
}

//----------------------------------------------------------------------------

bool MipsAssembler::Assemble(const char *srcFileName, 
                             const char *errFileName) {
    
    assert( srcFileName && errFileName );
    
    errStr.open(errFileName);
    
    assert( errStr.good() );
    
    Initialize();
    if( Read(srcFileName) && BackAnotate() ) {
        
        Cleanup();
        return true;
    }
    Cleanup();
    return false;
}

//----------------------------------------------------------------------------

void MipsAssembler::Initialize() {
    
    memset(dataSeg, 0, sizeof(unsigned char) * szDataSeg);
    memset(copyOfDataSeg, 0, sizeof(unsigned char) * szDataSeg);
    memset(codeSeg, 0, sizeof(MipsInstruction) * szCodeSeg);
    
    codeAddress = 5;
    dataAddress = 4;
    isText = false;
}

//----------------------------------------------------------------------------

bool MipsAssembler::Read(const char *srcFileName) {
    
    ifstream ifs(srcFileName);
    char buf[256], remainBuf[256];
    list<char*> tokens;
    bool actionTaken;
    bool skip = false;
    
    assert( ifs.good() );
    
    while( !ifs.eof() ) {
        
        tokens.clear();
        
        // get a line of text
        if( !skip ) {
            
            ifs.getline(buf, sizeof(buf));
        }
        else {
            
            skip = false;
            strcpy(buf, remainBuf);
        }
        for(int i=0; i<strlen(buf); i++) {
            
            if( buf[i] == ';' ) {
                
                skip = true;
                buf[i] = 0;
                strcpy(remainBuf, &buf[i + 2]);
                break;
            }
        }
        
        // process
        if( buf[0] == ' ' || buf[0] == 0 ) continue;
        Tokenize(tokens, buf);
        
        actionTaken = Directive(tokens) || Instruction(tokens) || Label(tokens);
        
        assert( actionTaken /* not supported */ );
        
        // check memory overflow
        if( codeAddress > (szCodeSeg - 4) ) {
            
            errStr << "\n\nMIPS Assembler error: code space overflow!!!\n";
            return false;
        }
        if( dataAddress > (szDataSeg - 4) ) {
            
            errStr << "\n\nMIPS Assembler error: data space overflow!!!\n";
            return false;
        }
    }
    return true;
}

//----------------------------------------------------------------------------

bool MipsAssembler::BackAnotate() {
    
    bool mainFound, initializeFound;
    Fix *f;
    Symbol *s;
    list<Fix*>::iterator i;
    list<Symbol*>::iterator j;
    
    for(i=fixl.begin(); i!=fixl.end(); ++i) {
        
        f = *i;
        s = f->symbol;
        
        if( f->type != 0 ) {
            
            switch( f->type ) {
                
            case 1: 
                BYTE(dataSeg, f->dataAddress) = (char)s->address; 
                break;
                
            case 2: 
                HALF(dataSeg, f->dataAddress) = (short)s->address; 
                break;
                
            case 3: WORD(dataSeg, f->dataAddress) = (long)s->address; 
                break;
                
            default: 
                assert( false );
            }
        }
        else {
            
            if( s->address == 0xffffffff ) {
                
                errStr << "\n\nMIPS Assembler error: undefined reference";
                errStr << " (" << s->name << ")!!!\n";
                return false;
            }
            else {
                
                codeSeg[f->codeAddress].im += s->address + f->offset;
            }
        }
    }
    
    mainFound = false;
    initializeFound = false;
    for(j=symb.begin(); j!=symb.end(); ++j) {
        
        s = (*j);
        
        assert( s->address != -1 );
        
        if( strcmp(s->name, "main") == 0 ) {
            
            mainFound = true;
        }
        if( strcmp(s->name, "__initialize__") == 0 ) {
            
            initializeFound = true;
        }
    }
    if( !mainFound ) {
        
        errStr << "\n\nMIPS Assembler error: (main) is undefined!!!\n";
        return false;
    }
    if( !initializeFound ) {
        
        errStr << "\n\nMIPS Assembler error: possibly \'Platune.h\' ";
        errStr << "is not included!!!\n";
        return false;
    }
    
    codeSeg[0].opCode = OPC_ADDIU;
    codeSeg[0].r1 = REG_SP;
    codeSeg[0].r2 = REG_ZR;
    codeSeg[0].im = szDataSeg - 16;
    codeSeg[1].opCode = OPC_ADDIU;
    codeSeg[1].r1 = REG_RA;
    codeSeg[1].r2 = REG_ZR;
    codeSeg[1].im = 3;
    codeSeg[2].opCode = OPC_J;
    codeSeg[2].im = InsertSymbol("__initialize__")->address;
    codeSeg[3].opCode = OPC_ADDIU;
    codeSeg[3].r1 = REG_RA;
    codeSeg[3].r2 = REG_ZR;
    codeSeg[3].im = 0;
    codeSeg[4].opCode = OPC_J;
    codeSeg[4].im = InsertSymbol("main")->address;
    return true;
}

//----------------------------------------------------------------------------

void MipsAssembler::Cleanup() {
    
    list<Symbol*>::iterator i;
    list<Fix*>::iterator j;
    
    for(i=symb.begin(); i!=symb.end(); ++i) {
        
        delete[] (*i)->name;
        delete *i;
    }
    
    for(j=fixl.begin(); j!=fixl.end(); ++j) {
        
        delete *j;
    }
    
    symb.clear();
    fixl.clear();
    
    memcpy(copyOfDataSeg, dataSeg, sizeof(unsigned char) * szDataSeg);
    errStr.close();
}

//----------------------------------------------------------------------------

bool MipsAssembler::Directive(list<char*> &tokens) {
    
    char *firstToken = *tokens.begin();
    
    if(      !strcmp(firstToken, ".fmask")     ) Translate_fmask(tokens);
    else if( !strcmp(firstToken, ".align")     ) Translate_align(tokens);
    else if( !strcmp(firstToken, ".byte")      ) Translate_byte(tokens);
    else if( !strcmp(firstToken, ".comm")      ) Translate_comm(tokens);
    else if( !strcmp(firstToken, ".cpload")    ) Translate_cpload(tokens);
    else if( !strcmp(firstToken, ".cprestore") ) Translate_cprestore(tokens);
    else if( !strcmp(firstToken, ".data")      ) Translate_data(tokens);
    else if( !strcmp(firstToken, ".ent")       ) Translate_ent(tokens);
    else if( !strcmp(firstToken, ".end")       ) Translate_end(tokens);
    else if( !strcmp(firstToken, ".extern")    ) Translate_extern(tokens);
    else if( !strcmp(firstToken, ".file")      ) Translate_file(tokens);
    else if( !strcmp(firstToken, ".frame")     ) Translate_frame(tokens);
    else if( !strcmp(firstToken, ".globl")     ) Translate_global(tokens);
    else if( !strcmp(firstToken, ".gpword")    ) Translate_gpword(tokens);
    else if( !strcmp(firstToken, ".half")      ) Translate_half(tokens);
    else if( !strcmp(firstToken, ".lcomm")     ) Translate_lcomm(tokens);
    else if( !strcmp(firstToken, ".loc")       ) Translate_loc(tokens);
    else if( !strcmp(firstToken, ".mask")      ) Translate_mask(tokens);
    else if( !strcmp(firstToken, ".rdata")     ) Translate_rdata(tokens);
    else if( !strcmp(firstToken, ".sdata")     ) Translate_sdata(tokens);
    else if( !strcmp(firstToken, ".set")       ) Translate_set(tokens);
    else if( !strcmp(firstToken, ".space")     ) Translate_space(tokens);
    else if( !strcmp(firstToken, ".text")      ) Translate_text(tokens);
    else if( !strcmp(firstToken, ".word")      ) Translate_word(tokens);
    else if( !strcmp(firstToken, ".asciiz")    ) Translate_asciiz(tokens);
    else 
        return false;
    return true;
}

//----------------------------------------------------------------------------

bool MipsAssembler::Instruction(list<char*> &tokens) {
    
    char *firstToken = *tokens.begin();
    
    if(      !strcmp(firstToken, "sb")        ) Translate_sb(tokens); 
    else if( !strcmp(firstToken, "sh")        ) Translate_sh(tokens); 
    else if( !strcmp(firstToken, "sw")        ) Translate_sw(tokens); 
    else if( !strcmp(firstToken, "usw")       ) Translate_sw(tokens); 
    else if( !strcmp(firstToken, "lb")        ) Translate_lb(tokens);
    else if( !strcmp(firstToken, "lh")        ) Translate_lh(tokens); 
    else if( !strcmp(firstToken, "lw")        ) Translate_lw(tokens); 
    else if( !strcmp(firstToken, "ulw")       ) Translate_lw(tokens);
    else if( !strcmp(firstToken, "la")        ) Translate_la(tokens); 
    else if( !strcmp(firstToken, "lbu")       ) Translate_lbu(tokens); 
    else if( !strcmp(firstToken, "lhu")       ) Translate_lhu(tokens); 
    else if( !strcmp(firstToken, "div")       ) Translate_div(tokens);
    else if( !strcmp(firstToken, "divu")      ) Translate_divu(tokens); 
    else if( !strcmp(firstToken, "rem")       ) Translate_rem(tokens); 
    else if( !strcmp(firstToken, "remu")      ) Translate_remu(tokens); 
    else if( !strcmp(firstToken, "mul")       ) Translate_mul(tokens); 
    else if( !strcmp(firstToken, "addu")      ) Translate_addu(tokens);
    else if( !strcmp(firstToken, "and")       ) Translate_and(tokens); 
    else if( !strcmp(firstToken, "or")        ) Translate_or(tokens); 
    else if( !strcmp(firstToken, "xor")       ) Translate_xor(tokens); 
    else if( !strcmp(firstToken, "subu")      ) Translate_subu(tokens);
    else if( !strcmp(firstToken, "sll")       ) Translate_sll(tokens); 
    else if( !strcmp(firstToken, "sra")       ) Translate_sra(tokens);
    else if( !strcmp(firstToken, "srl")       ) Translate_srl(tokens);
    else if( !strcmp(firstToken, "not")       ) Translate_not(tokens); 
    else if( !strcmp(firstToken, "negu")      ) Translate_negu(tokens);
    else if( !strcmp(firstToken, "move")      ) Translate_move(tokens); 
    else if( !strcmp(firstToken, "b")         ) Translate_b(tokens);
    else if( !strcmp(firstToken, "j")         ) Translate_j(tokens); 
    else if( !strcmp(firstToken, "beq")       ) Translate_beq(tokens);
    else if( !strcmp(firstToken, "bge")       ) Translate_bge(tokens);
    else if( !strcmp(firstToken, "bgeu")      ) Translate_bgeu(tokens);
    else if( !strcmp(firstToken, "bgt")       ) Translate_bgt(tokens);
    else if( !strcmp(firstToken, "bgtu")      ) Translate_bgtu(tokens);
    else if( !strcmp(firstToken, "ble")       ) Translate_ble(tokens);
    else if( !strcmp(firstToken, "bleu")      ) Translate_bleu(tokens); 
    else if( !strcmp(firstToken, "blt")       ) Translate_blt(tokens); 
    else if( !strcmp(firstToken, "bltu")      ) Translate_bltu(tokens);
    else if( !strcmp(firstToken, "bne")       ) Translate_bne(tokens); 
    else if( !strcmp(firstToken, "jal")       ) Translate_jal(tokens);
    else if( !strcmp(firstToken, "l.d")       ) Translate_l_d(tokens);
    else if( !strcmp(firstToken, "l.s")       ) Translate_l_s(tokens);
    else if( !strcmp(firstToken, "s.d")       ) Translate_s_d(tokens);
    else if( !strcmp(firstToken, "s.s")       ) Translate_s_s(tokens);
    else if( !strcmp(firstToken, "trunc.w.d") ) Translate_trunc_w_d(tokens);
    else if( !strcmp(firstToken, "mfc1")      ) Translate_mfc1(tokens);
    else if( !strcmp(firstToken, "mtc1")      ) Translate_mtc1(tokens);
    else if( !strcmp(firstToken, "cvt.d.w")   ) Translate_cvt_d_w(tokens);
    else if( !strcmp(firstToken, "cvt.d.s")   ) Translate_cvt_d_s(tokens);
    else if( !strcmp(firstToken, "cvt.s.d")   ) Translate_cvt_s_d(tokens);
    else if( !strcmp(firstToken, "add.d")     ) Translate_add_d(tokens);
    else if( !strcmp(firstToken, "add.s")     ) Translate_add_s(tokens);
    else if( !strcmp(firstToken, "div.d")     ) Translate_div_d(tokens);
    else if( !strcmp(firstToken, "div.s")     ) Translate_div_s(tokens);
    else if( !strcmp(firstToken, "mul.d")     ) Translate_mul_d(tokens);
    else if( !strcmp(firstToken, "mul.s")     ) Translate_mul_s(tokens);
    else if( !strcmp(firstToken, "sub.d")     ) Translate_sub_d(tokens);
    else if( !strcmp(firstToken, "sub.s")     ) Translate_sub_s(tokens);
    else if( !strcmp(firstToken, "mov.d")     ) Translate_mov_d(tokens);
    else if( !strcmp(firstToken, "mov.s")     ) Translate_mov_s(tokens);
    else if( !strcmp(firstToken, "neg.d")     ) Translate_neg_d(tokens);
    else if( !strcmp(firstToken, "neg.s")     ) Translate_neg_s(tokens);
    else if( !strcmp(firstToken, "c.eq.d")    ) Translate_c_eq_d(tokens);
    else if( !strcmp(firstToken, "c.eq.s")    ) Translate_c_eq_s(tokens);
    else if( !strcmp(firstToken, "c.le.d")    ) Translate_c_le_d(tokens);
    else if( !strcmp(firstToken, "c.le.s")    ) Translate_c_le_s(tokens);
    else if( !strcmp(firstToken, "c.lt.d")    ) Translate_c_lt_d(tokens);
    else if( !strcmp(firstToken, "c.lt.s")    ) Translate_c_lt_s(tokens);
    else if( !strcmp(firstToken, "bc1t")      ) Translate_bc1t(tokens);
    else if( !strcmp(firstToken, "bc1f")      ) Translate_bc1f(tokens);
    else 
        return false;
    return true;
}

//----------------------------------------------------------------------------

bool MipsAssembler::Label(list<char*> &tokens) {
    
    char *firstToken = *tokens.begin();
    
    if( firstToken[strlen(firstToken)-1] == ':' ) {
        
        firstToken[strlen(firstToken)-1] = 0;
        InsertSymbol(firstToken, isText ? codeAddress : dataAddress);
        return true;
    }
    return false;
}

//----------------------------------------------------------------------------
// directives...
//----------------------------------------------------------------------------

void MipsAssembler::Translate_fmask(list<char*> &tokens) {
    
    assert( true /* ignored */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_align(list<char*> &tokens) {
    
    assert( true /* ignored */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_byte(list<char*> &tokens) {
    
    tokens.pop_front();
    BYTE(dataSeg, dataAddress) = 
        (char)GetValue(*tokens.begin(), dataAddress, 1);
    dataAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_comm(list<char*> &tokens) {
    
    tokens.pop_front();
    InsertSymbol(*tokens.begin(), dataAddress);
    tokens.pop_front();
    dataAddress += GetValue(*tokens.begin());
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_cpload(list<char*> &tokens) {
    
    assert( false /* not supported */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_cprestore(list<char*> &tokens) {
    
    assert( false /* not supported */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_data(list<char*> &tokens) {
    
    isText = false;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_ent(list<char*> &tokens) {
    
    assert( true /* ignored */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_end(list<char*> &tokens) {
    
    assert( true /* ignored */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_extern(list<char*> &tokens) {
    
    assert( true /* ignored */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_file(list<char*> &tokens) {
    
    assert( false /* not supported */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_frame(list<char*> &tokens) {
    
    assert( true /* ignored */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_global(list<char*> &tokens) {
    
    assert( true /* ignored */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_gpword(list<char*> &tokens) {
    
    assert( false /* not supported */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_half(list<char*> &tokens) {
    
    tokens.pop_front();
    HALF(dataSeg, dataAddress) = 
        (short)GetValue(*tokens.begin(), dataAddress, 2);
    dataAddress += 2;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_lcomm(list<char*> &tokens) {
    
    tokens.pop_front();
    InsertSymbol(*tokens.begin(), dataAddress);
    tokens.pop_front();
    dataAddress += GetValue(*tokens.begin());
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_loc(list<char*> &tokens) {
    
    assert( false /* not supported */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_mask(list<char*> &tokens) {
    
    assert( true /* ignored */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_rdata(list<char*> &tokens) {
    
    isText = false;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_sdata(list<char*> &tokens) {
    
    isText = false;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_set(list<char*> &tokens) {
    
    assert( true /* ignored */ );
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_space(list<char*> &tokens) {
    
    tokens.pop_front();
    dataAddress += atoi(*tokens.begin());
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_text(list<char*> &tokens) {
    
    isText = true;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_word(list<char*> &tokens) {
    
    tokens.pop_front();
    WORD(dataSeg, dataAddress) = 
        (long)GetValue(*tokens.begin(), dataAddress, 3);
    dataAddress += 4;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_asciiz(list<char*> &tokens) {
    
    assert( false /* not supported */ );
}

//----------------------------------------------------------------------------
// instructions...
//----------------------------------------------------------------------------

void MipsAssembler::Translate_sb(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_SB;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_SB;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_SB;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_sh(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_SH;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_SH;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_SH;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_sw(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_SW;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_SW;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_SW;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_lb(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_LB;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_LB;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_LB;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_lh(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_LH;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_LH;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_LH;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_lw(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_LW;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_LW;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_LW;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_la(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_ADDIU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_ADDIU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_lbu(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_LBU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_LBU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_LBU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_lhu(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_LHU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_LHU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_LHU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_div(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    codeSeg[codeAddress].opCode = OPC_DIV;
    codeSeg[codeAddress].r1 = r2;
    codeSeg[codeAddress].r2 = r3;
    codeAddress += 1;
    codeSeg[codeAddress].opCode = OPC_MFLO;
    codeSeg[codeAddress].r1 = r1;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_divu(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    codeSeg[codeAddress].opCode = OPC_DIVU;
    codeSeg[codeAddress].r1 = r2;
    codeSeg[codeAddress].r2 = r3;
    codeAddress += 1;
    codeSeg[codeAddress].opCode = OPC_MFLO;
    codeSeg[codeAddress].r1 = r1;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_rem(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    codeSeg[codeAddress].opCode = OPC_DIV;
    codeSeg[codeAddress].r1 = r2;
    codeSeg[codeAddress].r2 = r3;
    codeAddress += 1;
    codeSeg[codeAddress].opCode = OPC_MFHI;
    codeSeg[codeAddress].r1 = r1;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_remu(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    codeSeg[codeAddress].opCode = OPC_DIVU;
    codeSeg[codeAddress].r1 = r2;
    codeSeg[codeAddress].r2 = r3;
    codeAddress += 1;
    codeSeg[codeAddress].opCode = OPC_MFHI;
    codeSeg[codeAddress].r1 = r1;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_mul(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    codeSeg[codeAddress].opCode = OPC_MULT;
    codeSeg[codeAddress].r1 = r2;
    codeSeg[codeAddress].r2 = r3;
    codeAddress += 1;
    codeSeg[codeAddress].opCode = OPC_MFLO;
    codeSeg[codeAddress].r1 = r1;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_addu(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0://value
        codeSeg[codeAddress].opCode = OPC_ADDIU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_ADDU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].r3 = r3;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_and(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_ANDI;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_AND;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].r3 = r3;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_or(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_ORI;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_OR;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].r3 = r3;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_xor(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_XORI;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_XOR;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].r3 = r3;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_subu(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_ADDIU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = -im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_SUBU;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].r3 = r3;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_sll(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_SLL;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_SLLV;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].r3 = r3;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_sra(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_SRA;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_SRAV;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].r3 = r3;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_srl(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_SRL;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_SRLV;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].r3 = r3;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_not(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    codeSeg[codeAddress].opCode = OPC_XORI;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeSeg[codeAddress].im = 0xffffffff;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_negu(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    codeSeg[codeAddress].opCode = OPC_XORI;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeSeg[codeAddress].im = 0xffffffff;
    codeAddress += 1;
    codeSeg[codeAddress].opCode = OPC_ADDIU;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r1;
    codeSeg[codeAddress].im = 0x00000001;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_move(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    codeSeg[codeAddress].opCode = OPC_ADDU;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeSeg[codeAddress].r3 = REG_ZR;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_b(list<char*> &tokens) {
    
    unsigned char r1;
    long im;
    
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r1, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_J;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_j(list<char*> &tokens) {
    
    tokens.pop_front();
    codeSeg[codeAddress].opCode = OPC_JR;
    codeSeg[codeAddress].r1 = GetRegister(*tokens.begin());
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_beq(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_BEQ;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im - codeAddress - 1;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_bge(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    codeAddress += 1;
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeAddress -= 1;
        codeSeg[codeAddress].opCode = OPC_SUB;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].r2 = r1;
        codeSeg[codeAddress].r3 = r2;
        codeAddress += 1;
        codeSeg[codeAddress].opCode = OPC_BGEZ;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].im = im - codeAddress - 1;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_bgeu(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    codeAddress += 1;
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeAddress -= 1;
        codeSeg[codeAddress].opCode = OPC_SUBU;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].r2 = r1;
        codeSeg[codeAddress].r3 = r2;
        codeAddress += 1;
        codeSeg[codeAddress].opCode = OPC_BGEZ;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].im = im - codeAddress - 1;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_bgt(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    codeAddress += 1;
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeAddress -= 1;
        codeSeg[codeAddress].opCode = OPC_SUB;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].r2 = r1;
        codeSeg[codeAddress].r3 = r2;
        codeAddress += 1;
        codeSeg[codeAddress].opCode = OPC_BGTZ;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].im = im - codeAddress - 1;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_bgtu(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    codeAddress += 1;
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeAddress -= 1;
        codeSeg[codeAddress].opCode = OPC_SUBU;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].r2 = r1;
        codeSeg[codeAddress].r3 = r2;
        codeAddress += 1;
        codeSeg[codeAddress].opCode = OPC_BGTZ;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].im = im - codeAddress - 1;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_ble(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    codeAddress += 1;
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeAddress -= 1;
        codeSeg[codeAddress].opCode = OPC_SUB;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].r2 = r1;
        codeSeg[codeAddress].r3 = r2;
        codeAddress += 1;
        codeSeg[codeAddress].opCode = OPC_BLEZ;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].im = im - codeAddress - 1;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_bleu(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    codeAddress += 1;
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeAddress -= 1;
        codeSeg[codeAddress].opCode = OPC_SUBU;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].r2 = r1;
        codeSeg[codeAddress].r3 = r2;
        codeAddress += 1;
        codeSeg[codeAddress].opCode = OPC_BLEZ;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].im = im - codeAddress - 1;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_blt(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    codeAddress += 1;
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeAddress -= 1;
        codeSeg[codeAddress].opCode = OPC_SUB;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].r2 = r1;
        codeSeg[codeAddress].r3 = r2;
        codeAddress += 1;
        codeSeg[codeAddress].opCode = OPC_BLTZ;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].im = im - codeAddress - 1;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_bltu(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    codeAddress += 1;
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeAddress -= 1;
        codeSeg[codeAddress].opCode = OPC_SUBU;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].r2 = r1;
        codeSeg[codeAddress].r3 = r2;
        codeAddress += 1;
        codeSeg[codeAddress].opCode = OPC_BLTZ;
        codeSeg[codeAddress].r1 = REG_AT;
        codeSeg[codeAddress].im = im - codeAddress - 1;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_bne(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_BNE;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im - codeAddress - 1;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_jal(list<char*> &tokens) {
    
    unsigned char r1;
    long im;
    
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r1, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_JAL;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_JALR;
        codeSeg[codeAddress].r1 = r1;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_l_d(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_L_D;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_L_D;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_L_D;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_l_s(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_L_S;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_L_S;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_L_S;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_s_d(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_S_D;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_S_D;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_S_D;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_s_s(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_S_S;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = REG_ZR;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_S_S;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = 0;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        codeSeg[codeAddress].opCode = OPC_S_S;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_trunc_w_d(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r3, im) ) {
        
    case 0:// value
        assert( false );
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_CVT_W_D;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_mfc1(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        assert( false );
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_MFC1;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_mtc1(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        assert( false );
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_MTC1;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_cvt_d_w(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        assert( false );
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_CVT_D_W;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_cvt_s_d(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        assert( false );
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_CVT_S_D;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_cvt_d_s(list<char*> &tokens) {
    
    unsigned char r1, r2;
    long im;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r2, im) ) {
        
    case 0:// value
        assert( false );
        break;
        
    case 1:// register
        codeSeg[codeAddress].opCode = OPC_CVT_D_S;
        codeSeg[codeAddress].r1 = r1;
        codeSeg[codeAddress].r2 = r2;
        codeAddress += 1;
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_add_d(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_ADD_D;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeSeg[codeAddress].r3 = r3;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_add_s(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_ADD_S;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeSeg[codeAddress].r3 = r3;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_div_d(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_DIV_D;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeSeg[codeAddress].r3 = r3;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_div_s(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_DIV_S;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeSeg[codeAddress].r3 = r3;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_mul_d(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_MUL_D;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeSeg[codeAddress].r3 = r3;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_mul_s(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_MUL_S;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeSeg[codeAddress].r3 = r3;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_sub_d(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_SUB_D;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeSeg[codeAddress].r3 = r3;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_sub_s(list<char*> &tokens) {
    
    unsigned char r1, r2, r3;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r3 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_SUB_S;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeSeg[codeAddress].r3 = r3;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_mov_d(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_MOV_D;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_mov_s(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_MOV_S;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_neg_d(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_NEG_D;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_neg_s(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_NEG_S;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_c_eq_d(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_C_EQ_D;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_c_eq_s(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_C_EQ_S;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_c_le_d(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_C_LE_D;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_c_le_s(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_C_LE_S;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_c_lt_d(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_C_LT_D;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_c_lt_s(list<char*> &tokens) {
    
    unsigned char r1, r2;
    
    tokens.pop_front();
    r1 = GetRegister(*tokens.begin());
    tokens.pop_front();
    r2 = GetRegister(*tokens.begin());
    
    codeSeg[codeAddress].opCode = OPC_C_LT_S;
    codeSeg[codeAddress].r1 = r1;
    codeSeg[codeAddress].r2 = r2;
    codeAddress += 1;
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_bc1t(list<char*> &tokens) {
    
    unsigned char r1;
    long im;
    
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r1, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_BC1T;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

void MipsAssembler::Translate_bc1f(list<char*> &tokens) {
    
    unsigned char r1;
    long im;
    
    tokens.pop_front();
    switch( CrackArg(*tokens.begin(), r1, im) ) {
        
    case 0:// value
        codeSeg[codeAddress].opCode = OPC_BC1F;
        codeSeg[codeAddress].im = im;
        codeAddress += 1;
        break;
        
    case 1:// register
        assert( false );
        break;
        
    case 2:// value and register
        assert( false );
        break;
    }
}

//----------------------------------------------------------------------------

long MipsAssembler::CrackArg(char *token, unsigned char &reg, long &value) {
    
    Fix* f;
    Symbol* s;
    long o1, o2;
    char op;
    char *part1;
    char *part2;
    
    // register only
    if( token[0] == '(' ) {
        
        reg = GetRegister(token + 1);
        return 1;
    }
    
    part1 = strtok(token, "(");
    part2 = strtok(NULL, ")");
    
    // value only
    if( !part2 && token[0] != '$' ) {
        
        if( sscanf(token, "%li", &o1) == 1 ) {
            
            value = o1;
        }
        else {
            
            value = 0;
            for(part2=token; *part2!=0; part2++) {
                
                if( *part2 == '+' ) {
                    
                    value += atoi(part2 + 1);
                    *part2 = 0;
                    break;
                }
            }
            s = InsertSymbol(part1);
            value += s->address == 0xffffffff ? 0 : s->address;
            if( s->address == 0xffffffff ) {
                
                f = new Fix;
                f->symbol = s;
                f->codeAddress = codeAddress;
                f->dataAddress = 0;
                f->offset = value;
                f->type = 0;
                fixl.push_back(f);
                value = 0;
            }
        }
        return 0;
    }
    
    // register only
    if( !part2 && token[0] == '$' ) {
        
        reg = GetRegister(part1);
        return 1;
    }
    
    // register and value/register
    reg = GetRegister(part2);
    if( sscanf(part1, "%li%c%li", &o1, &op, &o2) == 3 ) {
        
        if( op == '+' ) {
            
            value = o1 + o2;
        }
        else if( op == '-' ) {
            
            value = o1 = o2;
        }
        else {
            
            assert( false );
        }
    }
    else if( sscanf(part1, "%li", &o1) == 1 ) {
        
        value = o1;
    }
    else {
        
        value = 0;
        for(char *sp=part1; *sp; sp++) {
            
            if( *sp == '+' || *sp == '-' ) {
                
                assert( sscanf(sp, "%d", &value) == 1);
                *sp = 0;
                break;
            }
        }
        
        s = InsertSymbol(part1);
        value += s->address == 0xffffffff ? 0 : s->address;
        if( s->address == 0xffffffff ) {
            
            f = new Fix;
            f->symbol = s;
            f->codeAddress = codeAddress;
            f->dataAddress = 0;
            f->offset = value;
            f->type = 0;
            fixl.push_back(f);
            value = 0;
        }
    }
    return 2;
}

//----------------------------------------------------------------------------

long MipsAssembler::GetRegister(char *token) {
    
    long v;
    
    if( token[0] == '$' && sscanf(token + 1, "%li", &v) == 1 ) {
        
        assert( v < 32 );
        return v;
    }
    else if( token[0] == '$' && token[1] == 'f' && 
        sscanf(token + 2,"%li", &v) == 1 ) {
        
        assert( v < 32 );
        return v;
    }
    else if( token[0] == '$' && token[1] == 's' && token[2] == 'p' ) {
        
        return REG_SP;
    }
    else if( token[0] == '$' && token[1] == 'f' && token[2] == 'p' ) {
        
        return REG_FP;
    }
    assert( false );
    return 0;
}

//----------------------------------------------------------------------------

long MipsAssembler::GetValue(char *token, int address, int type) {
    
    Symbol *s;
    Fix *f;
    long v;
    
    if( token[1] == 'x' && sscanf(token, "%lx", &v) == 1 ) {
        
        return v;
    }
    else if( sscanf(token, "%li", &v) == 1 ) {
        
        return v;
    }
    else {
        
        s = InsertSymbol(token);        
        if( s->address == 0xffffffff ) {
            
            assert( type != 0 );
            
            f = new Fix;
            f->symbol = s;
            f->codeAddress = 0;
            f->dataAddress = address;
            f->offset = 0;
            f->type = type;
            fixl.push_back(f);
            return 0;
        }
    }
    return s->address;
}

//----------------------------------------------------------------------------

Symbol *MipsAssembler::InsertSymbol(const char *name, unsigned address) {
    
    list<Symbol*>::iterator i;
    Symbol *s = 0;
    
    // find
    for(i=symb.begin(); i!=symb.end(); ++i) {
        
        if( strcmp((*i)->name, name) == 0 ) {
            
            s = (*i);
            break;
        }
    }
    
    // fix
    if( s && s->address == 0xffffffff ) {
        
        s->address = address;
    }
    
    // create
    if( !s ) {
        
        s = new Symbol;
        s->name = new char[strlen(name) + 1];
        strcpy(s->name, name);
        s->address = address;
        symb.push_back(s);
    }
    return s;
}

//----------------------------------------------------------------------------

list<char*> &MipsAssembler::Tokenize(list<char*> &l, char *buf) {
    
    char *s;
    
    for(s=buf; *s; ++s) {
        
        *s = *s == ',' ? ' ' : *s;
    }
    for(s=strtok(buf, " "); s; s=strtok(0, " ")) {
        
        l.push_back(s);
    }
    return l;
}