//
// Copyright (c) 1997-2001 Tony Givargis. Permission to copy is
// granted provided that this header remains intact. This software
// is provided with no warranties.
//

//----------------------------------------------------------------------------

#include <iomanip>
#include <cassert>
#include <cstring>
#include <cmath>
#include <process.h>
#include "tech.h"
#include "loader.h"
using namespace std;

//----------------------------------------------------------------------------

Loader::Loader(const char *_baseDir) {
    
    strcpy(baseDir, _baseDir);
    
    sprintf(cppFileName, "%s\\cpp.o", baseDir);
    sprintf(lccFileName, "%s\\lcc.o", baseDir);
    sprintf(errFileName, "%s\\error.txt", baseDir);
    sprintf(outFileName, "");
    sprintf(repFileName, "");
    sprintf(parFileName, "");
    
    ma = new MipsAssembler(0xfffff, 0x7fff);
    mvm = new MipsVirtualMachine(
        ma->szDataSeg, 
        ma->szCodeSeg, 
        ma->dataSeg, 
        ma->copyOfDataSeg,
        ma->codeSeg,
        this);
    psData = new ParameterSpaceData();
}

//----------------------------------------------------------------------------

Loader::~Loader() {
    
    if( outStr.is_open() ) {
        
        outStr.close();
    }
    
    if( repStr.is_open() ) {
        
        repStr.close();
    }
    
    if( parStr.is_open() ) {
        
        parStr.close();
    }
    
    delete ma;
    delete mvm;
    delete psData;
}

//----------------------------------------------------------------------------

bool Loader::Load(const char *fileName, char *msgBuf) {
    
    char a0[4][256];
    char a1[6][256];
    
    sprintf(a0[0], "%s\\cpp.exe", baseDir);
    sprintf(a0[1], "cpp");
    sprintf(a0[2], fileName);
    sprintf(a0[3], cppFileName);
    sprintf(a1[0], "%s\\lcc.exe", baseDir);
    sprintf(a1[1], "lcc");
    sprintf(a1[2], "-target=mips-ultrix");
    sprintf(a1[3], "-errout=%s", errFileName);
    sprintf(a1[4], cppFileName);
    sprintf(a1[5], lccFileName);
    
    sprintf(outFileName, "%s\\%s.out.txt", baseDir, fileName);
    sprintf(repFileName, "%s\\%s.rep.txt", baseDir, fileName);
    sprintf(parFileName, "%s\\%s.par.txt", baseDir, fileName);
    
    if( _spawnl(_P_WAIT, a0[0], a0[1], a0[2], a0[3], 0) ) {
        
        strcpy(msgBuf, "C preprocessor failed on input!");
        return false;
    }
    
    if( _spawnl(_P_WAIT, a1[0], a1[1], a1[2], a1[3], a1[4], a1[5], 0) ) {
        
        strcpy(msgBuf, "C compiler failed on input! (see err.txt)");
        return false;
    }
    
    if( ma->Assemble(a1[5], errFileName) == false ) {
        
        strcpy(msgBuf, "Assembler failed on input!");
        return false;
    }
    return true;
}

//----------------------------------------------------------------------------

void Loader::Report() {
    
    char buf[32];
    unsigned long mr, mw;
    
    mr  = (mvm->iCache.numReadMiss * (mvm->iCache.line >> 2));
    mr += (mvm->dCache.numReadMiss * (mvm->dCache.line >> 2));
    mw  = (mvm->dCache.numWriteBack * (mvm->dCache.line >> 2));
    
    repStr << setw(14) << mvm->iCache.size;
    repStr << setw(14) << mvm->iCache.line;
    repStr << setw(14) << mvm->iCache.assoc;
    repStr << setw(14) << mvm->dCache.size;
    repStr << setw(14) << mvm->dCache.line;
    repStr << setw(14) << mvm->dCache.assoc;
    sprintf(buf, "%d(%d)/%d(%d)", 
        mvm->cpuICacheBus.addrState.width,
        mvm->cpuICacheBus.addrState.encoding,
        mvm->cpuICacheBus.dataState.width,
        mvm->cpuICacheBus.dataState.encoding);  
    repStr << setw(14) << buf;
    sprintf(buf, "%d(%d)/%d(%d)", 
        mvm->cpuDCacheBus.addrState.width,
        mvm->cpuDCacheBus.addrState.encoding,
        mvm->cpuDCacheBus.dataState.width,
        mvm->cpuDCacheBus.dataState.encoding);
    repStr << setw(14) << buf;
    sprintf(buf, "%d(%d)/%d(%d)", 
        mvm->cacheMemBus.addrState.width,
        mvm->cacheMemBus.addrState.encoding,
        mvm->cacheMemBus.dataState.width,
        mvm->cacheMemBus.dataState.encoding);
    repStr << setw(14) << buf;
    repStr << setw(14) << mvm->voltage;
    
    repStr << setprecision(4);
    
    repStr << setw(14) << mvm->cpuPower;
    repStr << setw(14) << mvm->iCachePower;
    repStr << setw(14) << mvm->dCachePower;
    repStr << setw(14) << mvm->cpuICacheBusPower;
    repStr << setw(14) << mvm->cpuDCacheBusPower;
    repStr << setw(14) << mvm->cacheMemBusPower;
    repStr << setw(14) << mvm->memPower;
    
    repStr << setw(14) << mvm->iCache.numRead;
    repStr << setw(14) << mvm->iCache.numReadMiss;
    repStr << setw(14) << mvm->dCache.numRead;
    repStr << setw(14) << mvm->dCache.numReadMiss;
    repStr << setw(14) << mvm->dCache.numWrite;
    repStr << setw(14) << mvm->dCache.numWriteMiss;
    repStr << setw(14) << mr;
    repStr << setw(14) << mw;
    
    repStr << setw(14) << ComputeClockSpeed(mvm->voltage) / 1e6;
    repStr << setw(14) << floor(mvm->exeTime * ComputeClockSpeed(mvm->voltage));
    repStr << setw(14) << mvm->power;
    repStr << setw(14) << mvm->exeTime;
    repStr << setw(14) << mvm->power * mvm->exeTime;
    repStr << setw(14) << mvm->power * mvm->exeTime * mvm->exeTime;
    repStr << endl;
}

//----------------------------------------------------------------------------

void Loader::Pareto(int *param, double power, double time) {
    
    char buf[32];
    
    parStr << setw(14) << param[0];
    parStr << setw(14) << param[1];
    parStr << setw(14) << param[2];
    parStr << setw(14) << param[3];
    parStr << setw(14) << param[4];
    parStr << setw(14) << param[5];
    sprintf(buf, "%d(%d)/%d(%d)", param[6], param[7], param[8], param[9]);
    parStr << setw(14) << buf;
    sprintf(buf, "%d(%d)/%d(%d)", param[10], param[11], param[12], param[13]);
    parStr << setw(14) << buf;
    sprintf(buf, "%d(%d)/%d(%d)", param[14], param[15], param[16], param[17]);
    parStr << setw(14) << buf;
    parStr << setw(14) << param[18] / 10.0;
    
    parStr << setprecision(4);
    
    parStr << setw(14) << power;
    parStr << setw(14) << time;
    parStr << setw(14) << power * time;
    parStr << setw(14) << power * time * time;
    parStr << endl;
}

//----------------------------------------------------------------------------

void Loader::Output(char c) {
    
    outStr << c;
    outStr.flush();
}

//----------------------------------------------------------------------------

void Loader::Reset() {
    
    if( outStr.is_open() ) {
        
        outStr.close();
    }
    
    if( repStr.is_open() ) {
        
        repStr.close();
    }
    
    if( parStr.is_open() ) {
        
        parStr.close();
    }
    
    outStr.open(outFileName);
    repStr.open(repFileName);
    parStr.open(parFileName);
    
    repStr << setw(14) << "I$-Size";
    repStr << setw(14) << "I$-Line";
    repStr << setw(14) << "I$-Assoc";
    repStr << setw(14) << "D$-Size";
    repStr << setw(14) << "D$-Line";
    repStr << setw(14) << "D$-Assoc";
    repStr << setw(14) << "C-I$-Bus";
    repStr << setw(14) << "C-D$-Bus";
    repStr << setw(14) << "$-M-Bus";
    repStr << setw(14) << "Voltage";
    
    repStr << setw(14) << "C(W)";
    repStr << setw(14) << "I$(W)";
    repStr << setw(14) << "D$(W)";
    repStr << setw(14) << "C-I$-Bus(W)";
    repStr << setw(14) << "C-D$-Bus(W)";
    repStr << setw(14) << "$-M-Bus(W)";
    repStr << setw(14) << "M(W)";
    
    repStr << setw(14) << "I$-R";
    repStr << setw(14) << "I$-R-Miss";
    repStr << setw(14) << "D$-R";
    repStr << setw(14) << "D$-R-Miss";
    repStr << setw(14) << "D$-W";
    repStr << setw(14) << "D$-W-Miss";
    repStr << setw(14) << "M-Read";
    repStr << setw(14) << "M-Write";
    
    repStr << setw(14) << "Speed(MHz)";
    repStr << setw(14) << "Cycles";
    repStr << setw(14) << "Power(W)";
    repStr << setw(14) << "Time(sec)";
    repStr << setw(14) << "Energy(J)";
    repStr << setw(14) << "EnDel(Js)";
    
    repStr << endl;
    repStr.flush();
    
    parStr << setw(14) << "I$-Size";
    parStr << setw(14) << "I$-Line";
    parStr << setw(14) << "I$-Assoc";
    parStr << setw(14) << "D$-Size";
    parStr << setw(14) << "D$-Line";
    parStr << setw(14) << "D$-Assoc";
    parStr << setw(14) << "C-I$-Bus";
    parStr << setw(14) << "C-D$-Bus";
    parStr << setw(14) << "$-M-Bus";
    parStr << setw(14) << "Voltage";
    
    parStr << setw(14) << "Power(W)";
    parStr << setw(14) << "Time(sec)";
    parStr << setw(14) << "Energy(J)";
    parStr << setw(14) << "EnDel(Js)";
    
    parStr << endl;
    parStr.flush();
}