/*
 * File post.c for PVSC
 *
 * original by P.S.Elliott / 29/10/93
 *
 * Modified by M. Gumm / Nov. 94
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "mvpdef.h"


/**********************************************************************
 * PostScript definitions
 **********************************************************************/
#define PR_PS_EOL(F) 		fprintf(F, ")s nl \n");

#define PR_PS_CHANGE_STYLE(F,S) switch( S ) \
				{ case italic : fprintf(F, ")s i("); break; \
				  case bold   : fprintf(F, ")s b("); break; \
				  default     : fprintf(F, ")s n("); break; }

#define PR_PS_STYLE(F,S) 	switch( S ) \
				{ case italic : fprintf(F, "i("); break; \
				  case bold   : fprintf(F, "b("); break; \
				  default     : fprintf(F, "n("); break; }

char *ps_header_1[]= 
{
"%!PS-Adobe-2.0 ",
"%%PageLength 60 / LineLength 80",
"%%EndComments",
"",
"/inch { 72 mul } bind def",
"",
"/fullwidth 8.5 inch def	%set 8.5 inch for US Letter / A4",
"/fullheight 11.4 inch def	%set 11 inch for US Letter / 11.4 inch for A4",
"",
NULL
};

/* FSize must be defined before using ps_header_2 */

char *ps_header_2[]=
{
"/LnHt FSize 1.2 mul 0.5 add cvi def",
"/s {show} def",
"",
"/setmargin {",
"  /leftedge 0 def",
"  /bottomedge 0 def",
"  fullheight fullwidth LandscapeMode { exch } if",
"    /rightedge exch def",
"    /topedge exch def",
"  9 7 LandscapeMode { exch } if pop",
"    %/FontSize exch def",
"",
"  /fontHd /Helvetica-BoldOblique findfont 14 scalefont def",
"  /fontH2 /Helvetica-BoldOblique findfont 10 scalefont def",
"  /fontNd /Times-Bold            findfont 12 scalefont def",
"  /fontmn /Courier		 findfont FSize scalefont def",
"  /fontmb /Courier-Bold		 findfont FSize scalefont def",
"  /fontmi /Courier-Oblique	 findfont FSize scalefont def",
"",
"  /fontH         { fontHd setfont } bind def",
"  /fontD         { fontH2 setfont } bind def",
"  /fontN         { fontNd setfont } bind def",
"  /n		 { fontmn setfont } bind def",
"  /b		 { fontmb setfont } bind def",
"  /i		 { fontmi setfont } bind def",
"} def",
"",
"/graybox	% grey x1 y1 x2 y2 graybox --",
"{",
"    newpath",
"      4 -1 roll 2 copy			% y2 x1",
"     10 sub exch moveto",
"      4 -1 roll 2 copy			% x1 y1",
"      10 180 270 arc",
"      4 -1 roll 1 copy 3 -1 roll	% x2 y1",
"      10 270 0 arc",
"      3 -1 roll 1 copy 3 1 roll	% x1 y2 x2 y2",
"      10 0 90 arc",
"      10 90 180 arc",
"    closepath",
"    gsave",
"      setgray fill",
"    grestore",
"    0 setgray stroke",
"} def",
"",
"/endpage		% page_number endpage --",
"{",
"  gsave",
"    % listing type, user, date across the bottom",
"    0.88                                % gray level",
"    leftedge  50 add topedge 50 sub     % bottom lt corner (org te 56)",
"    rightedge 95 sub topedge 42 sub     % top rt corner (org te 36)",
"    graybox",
"    leftedge 50 add topedge 50 sub moveto",
"    fontD",
"    MailFor show",
"    fontH",
"    User show",
"    fontD",
"    rightedge 95 sub TimeNow stringwidth pop sub",
"    topedge 52 sub moveto TimeNow show",
"    % page number to the upper right corner",
"    .98                                 % gray level (org 1)",
"    rightedge 70 sub topedge 50 sub     % bottom lt (org te 56)",
"    rightedge 50 sub topedge 40 sub     % top rt (org te 36)",
"    graybox",
"    fontD",
"    (Page) dup stringwidth",
"       topedge 40 sub exch sub                  % y-centered",
"       exch 2 div rightedge 60 sub exch sub     % x-centered",
"       exch moveto show",
"    fontH",
"    dup stringwidth",
"       topedge 58 sub exch sub                  % y-centered",
"       exch 2 div rightedge 60 sub exch sub     % x-centered",
"       exch moveto show",
"    0.88                                   % gray level",
"    leftedge  50 add bottomedge 56 add     % bottom lt corner (org be 50)",
"    rightedge 50 sub bottomedge 64 add     % top rt corner (org be 70)",
"    graybox",
"    fontH",
"    % compute x pos for centring",
"      % ((l+50) + (r-50)) / 2 is the page centre",
"      leftedge rightedge add 2 div",
"      % but we want to start at the beginning of the string, not the middle",
"      Subject stringwidth pop 2 div sub",
"    % y pos is simple",
"      bottomedge 54 add",
"      moveto",
"    Subject show",
"  grestore",
"  PageSave restore      % restore *always* before showpage",
"  showpage		% display it",
"} def",
"",
"/endcol			% page_number col_number endcol --",
"{",
"  pop			% get rid of col_number",
"  pop                   % get rid of page_number",
"  /ypos topedge 92 sub def",
"  /xpos rightedge leftedge sub 2 div def",
"  xpos ypos moveto",
"} def",
"",
"/newpage	% page_number newpage --",
"{",
"  pop		% pageno",
"  /PageSave save def",
"  LandscapeMode {setlandscape} if",
"  /lct 0 def",
"  /ypos topedge 92 sub def",
"  /xpos leftedge 50 add def",
"  xpos ypos moveto",
"  n",
"} def",
"",
"/setlandscape",
"{",
"  612 0 translate",
"  90 rotate",
"} def",
"",
"/nl",
"{",
"  /ypos ypos LnHt sub def",
"  xpos ypos moveto",
"} def",
"",
"106 45 {dup mul exch dup mul add 1.0 exch sub} setscreen",
"",
"/LandscapeMode false def",
"",
"setmargin",
"%%EndProlog",
"",
NULL
};

char *ps_page_header[]=
{
"/MailFor () def",
"/User (VHDL Source Code Lister) def",
NULL
};

char *ps_duplex[]=
{
"%Duplexmode for hp lj4",
"statusdict begin mark",
"true setduplexmode false settumble",
"cleartomark end",
"",
NULL
};

	
/********************************************************
 * Returns the style of the next token given
 * the required emphasis & current token.
 ********************************************************/
style_t lookup_style( emphasis, tok )
	emphasis_t	emphasis;
	int		tok;
{
	switch( tok ) {

		case KEYWORD :
			if ( emphasis == key )
				return( bold );
			else
				return( normal );
			break;

		case ATTRIBUTE :
			if ( emphasis == key )
				return( bold );
			else
				return( normal );
			break;

		case IDENTIFIER :
			if ( emphasis == ident )
				return( bold );
			else
				return( normal );
			break;

		case COMMENT :
			return( italic );
			break;

		default :
			return( normal );
			break;
	}
}

/****************************************************************
 * Updates style given emphasis & current token & returns TRUE
 * if different from last, otherwise FALSE.
 ****************************************************************/
int get_style( emphasis, tok, style )
	emphasis_t	emphasis;
	int		tok;
	style_t		*style;

{
	style_t	tmp_style;

	/* don't change style at end of line */
	if ( tok == ENDLINE ) return( FALSE );

	tmp_style = lookup_style( emphasis, tok );
	if ( *style == start ) {
		*style = tmp_style;
		return( FALSE );
	}
	else {
		if ( *style != tmp_style ) {
			*style = tmp_style;
			return( TRUE );
		}
		else
			return( FALSE );
	}
}


/****************************************************************
 * ps_checked_out()
 * 	checks if column position exeeds paperwith
 *	inserts newline eventually 
 ****************************************************************/

int ps_checked_out( out, str, col, style, mode)
	FILE		*out;
	char		*str;
	int		*col;
	outmode_t	mode;
{
	int	inc = 0;	

	if (*col > maxcharperline )
	{
		PR_PS_EOL( out);
		*col =0;
		inc = 1;
		PR_PS_STYLE(out, style);
	}

	if ( mode == pr)
		PR(out, str);
	else if (mode == pl) 
		PL(out, str);
	else
		PC(out, *str);
	return(inc);
}

/******************************************************************
 * Generate output of token in PostScript
 * Automatically insert newline if line too long
 * Return 1 if newline was generated else 0
 ******************************************************************/

int ps_out( out, emphasis, tok, str )
	FILE		*out;
	emphasis_t	emphasis;
	int		tok;
	char		*str;
{
	static	style_t	style = start;
	static	int	col = 0;
	int		i,k;
	int		incrline = 0;
	style_t		old_style;
	char		help[4];

	old_style = style;

	if ( style == start && tok != ENDLINE ) {
		(void)get_style( emphasis, tok, &style );
		PR_PS_STYLE( out, style );

	}
	else
		if ( get_style( emphasis, tok, &style ) )
			PR_PS_CHANGE_STYLE( out, style );

	/* process the current text for PS output */
	switch( tok ) 
	{
		case DELIMITER :
			/* escape brackets */
			col++;
			if ( str[0] == '(' || str[0] == ')' )
			{
				help[0] = '\\';
				help[1] = str[0];	
				help[2] = '\0';
				incrline = ps_checked_out(out, help, &col, style, pr);
			}
			else
				incrline = ps_checked_out(out, str, &col, style, pr);

			break;

		case QUOTED_CHAR :
			/* escape brackets & backslash*/
			col +=3;
			k = 0;
			help[k++] = '\'';
			if ( str[1] == '(' || str[1] == ')' || str[1] == '\\' )
				help[k++] = '\\';
			help[k++] = str[1];
			help[k++] = '\'';
			help[k] = '\0';
			incrline = ps_checked_out(out, help, &col, style, pr);	
			break;

		case TAB :
			/* remove the tab */
			for ( i = 0; i < 8 - col % 8; i++ )
			{
				col++;
				incrline = ps_checked_out(out, " ", &col, style, pr);
			}	
			break;

		case COMMENT : 
		case STRING :
			/* process comments  and strings*/
			while ( *str != NULL ) {
				/* escape brackets & backslash */
				if ( *str == '(' || *str == ')' || *str == '\\' )
				{	
					col++;
					help[0] = '\\';
					help[1] = *str;	
					help[2] = '\0';
					incrline += ps_checked_out(out, help, &col, style, pr);
				}
				/* remove tabs */
				else if ( *str == '\t' )
					for ( i = 0; i < 8 - col % 8; i++ )
					{
						col++;	
						incrline += ps_checked_out(out, " ", 
									  &col, style, pr);
					}
				else 
				{
					col++;
					incrline += ps_checked_out(out, str, &col, style, pc);
				}
				str++;
			}
			break;


		case ENDLINE :
			/* reset line */
			if ( old_style != start )
				PR_PS_EOL( out )
			else
				PL( out, "nl" );
			col = 0;
			style = start;
			break;

		default :
			col += yyleng;
			incrline = ps_checked_out(out, str, &col, style, pr);
			break;

	}	/* switch*/
	
	return(incrline);
}

/******************************************************************
 * Output of line number in PostScript
 ******************************************************************/

int ps_lineno_out( out, lineno )
	FILE		*out;
	int		lineno;
{
	fprintf(out, "i(%4d  )s ", lineno);
}
