--
-- Copyright 1990 
--
-- WAVES Standard Packages Version 3.3.2
-- 25 October 1990
--
-- This code is distributed for the purposes of evaluating the 
-- Waveform And Vector Exchange Specification (WAVES) proposal 
-- presented to the IEEE by the WAVES Analysis and Standardization 
-- Group.  This code may not be used for commercial purposes and 
-- may not be redistributed or published without permission of the 
-- Chairman of the WAVES Analysis and Standardization Group, 
-- Mr Robert Hillman.
--
-- Address comments or questions to:
--    Robert Hillman           
--    RADC/RBRP                
--    Griffis AFB, NY                
--    (315) 330-2241                 
--                                   
--    hillman@tisss.radc.af.mil      
--
use STD.TEXTIO.all;
library WAVES_STANDARD;
use WAVES_STANDARD.WAVES_SYSTEM;
use WAVES_STANDARD.WAVES_EVENTS.all ;
-- A context clause providing visibility to an analyzed copy of 
-- WAVES_INTERFACE is required at this point.
use WAVES_STANDARD.WAVES_INTERFACE.all ;

library WORK ;
use WORK.WAVES_DEVICE.all ;
use WORK.WAVES_FRAMES.all ;
-- previous 5 lines added by GLN on 02-09-91

package WAVES_OBJECTS is
  

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- The ETIME function returns a EVENT_TIME object.  These functions
  -- overload ETIME functions previously defined in the WAVES INTERFACE
  -- but are defined here for visibility to the TEST_PINS object which
  -- is used by the BASE_PIN parameter.
  --
  function ETIME (
      NOMINAL            : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN		 : TEST_PINS)
    return EVENT_TIME;
  
  function ETIME (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME;
      BASE_EVENT          : POSITIVE;
      BASE_PIN		  : TEST_PINS)
    return EVENT_TIME;
  
  function ETIME (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN		 : TEST_PINS)
    return EVENT_TIME;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- Calling DELAY with a TEST PIN and an EVENT NUMBER references the 
  -- event number on the pin.  Calling DELAY with a TEST PIN and a 
  -- LOGIC VALUE causes DELAY to wait for a transition to the specified
  -- logic value on that pin.
  --
  function DELAY (
      NOMINAL            : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL            : TIME;
      BASE_LOGIC         : LOGIC_VALUE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME;
      BASE_LOGIC         : LOGIC_VALUE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME;
      BASE_LOGIC         : LOGIC_VALUE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The following definitions build up a WAVE TIMING object.
  --
  -- TIME DATA is a pointer to a FRAME SET ARRAY.  A pointer is used
  -- to conserve space, since only a few FRAME SET ARRAY's are typically
  -- defined, but are referenced in many places. 
  --
  -- A WAVE_TIMING combines all the timing information needed for the
  -- APPLY procedure.  It contains a DELAY (one Delay Element) and a
  -- TIME DATA. 
  --
  type TIME_DATA is access FRAME_SET_ARRAY;
  
  type WAVE_TIMING is record
    DELAY                 : DELAY_TIME;
    TIMING                : TIME_DATA;
  end record;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The following definitions allow unconstrained arrays of WAVES
  -- objects.
  --
  type TIME_DATA_LIST is array (POSITIVE range <>) of TIME_DATA;
  
  type WAVE_TIMING_LIST is array (POSITIVE range <>) of WAVE_TIMING;
  
  type TEST_PINS_LIST is array (POSITIVE range <>) of TEST_PINS;
  
  subtype PIN_CODE_STRING is STRING (
      TEST_PINS'POS(TEST_PINS'LEFT) + 1 to
      TEST_PINS'POS(TEST_PINS'RIGHT) + 1);
  
  type PIN_CODE_LIST is array (POSITIVE range <>) of PIN_CODE_STRING;
  
  type INDEX_SLICE is record
      CODES        : PIN_CODE_STRING;
      INDEX        : INTEGER;
  end record;

  type INDEX_SLICE_LIST is array (POSITIVE range <>) of 
      INDEX_SLICE; 

  type TIMED_SLICE is record
      CODES        : PIN_CODE_STRING;
      STIME        : TIME;
  end record;

  type TIMED_SLICE_LIST is array (POSITIVE range <>) of 
      TIMED_SLICE; 

  type WTIME_SLICE is record
      CODES        : PIN_CODE_STRING;
      WTIME        : WAVE_TIMING;
  end record;

  type WTIME_SLICE_LIST is array (POSITIVE range <>) of 
      WTIME_SLICE; 
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The following definitions provide support for pinsets.  A PINSET
  -- is an array of booleans, one for each test pin.  Two deferred
  -- constants are defined, ALL PINS and NO PINS. 
  --
  type PINSET is array (TEST_PINS) of BOOLEAN;
  
  constant ALL_PINS       : PINSET;
  
  constant NO_PINS        : PINSET;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- Three functions are provided to create a pinset.  The first
  -- returns a completely populated pinset.  The second creates a
  -- pinset whose single member is the named test pin.  The third
  -- creates a pinset whose members are those pins named in the list
  -- parameter. 
  --
  function NEW_PINSET
     return PINSET;
  
  function NEW_PINSET (
      PIN                 : in TEST_PINS )
    return PINSET;
  
  function NEW_PINSET (
      PINS                : in TEST_PINS_LIST )
    return PINSET;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following are the basic interface types into the WAVES
  -- dataset.  Waves_Port_List record is the port connects into
  -- the WAVES dataset.  Waves_Match_List record provides Match 
  -- and Delay Logic values. 
  -- 
  subtype WAVES_PORT_LIST is WAVES_SYSTEM.SYSTEM_WAVES_PORT_LIST;

  subtype WAVES_MATCH_LIST is WAVES_SYSTEM.SYSTEM_WAVES_MATCH_LIST;

-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The APPLY procedures to generate waveform slices.
  --
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME;
      constant ACTIVE_PINS : in    TEST_PINS );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME_SET;
      constant ACTIVE_PINS : in    TEST_PINS_LIST );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME;
      constant ACTIVE_PINS : in    TEST_PINS );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME_SET;
      constant ACTIVE_PINS : in    TEST_PINS_LIST );
-- 
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      constant CODES       : in    PIN_CODE_STRING;
      constant DELAY       : in    DELAY_TIME;
      variable FRAMES      : in    TIME_DATA;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant CODES       : in    PIN_CODE_STRING;
      constant DELAY       : in    DELAY_TIME;
      variable FRAMES      : in    TIME_DATA;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      constant CODES       : in    PIN_CODE_STRING;
      variable SLICE       : in    WAVE_TIMING;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant CODES       : in    PIN_CODE_STRING;
      variable SLICE       : in    WAVE_TIMING;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS );
  
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The TAG procedures apply the string parameter to the output
  -- signals, in effect labeling (tagging) the next call to APPLY. 
  --
  procedure TAG (
      signal CONNECT     : out WAVES_PORT_LIST ;
      constant TAG_LABEL : in STRING );
  
  procedure TAG (
      signal CONNECT     : out WAVES_PORT_LIST ;
      constant TAG_LABEL : in STRING;
      constant PINS      : in PINSET );
  
  procedure TAG (
      signal CONNECT     : out WAVES_PORT_LIST ;
      constant TAG_LABEL : in STRING;
      constant PINS      : in TEST_PINS );
  
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The MATCH procedure enables/disables MATCH mode comparisons on
  -- the specified pins.  The MATCH function samples the results of a
  -- MATCH mode comparison. 
  --

  type MATCH_CONTROL_TYPE is (HOLD, SAMPLE);
  
  procedure MATCH (
      signal   CONNECT      : out WAVES_PORT_LIST;
      constant CONTROL      : in  MATCH_CONTROL_TYPE ;
      constant ACTIVE_PIN   : in  TEST_PINS );
  
  procedure MATCH (
      signal   CONNECT      : out WAVES_PORT_LIST;
      constant CONTROL      : in  MATCH_CONTROL_TYPE;
      constant ACTIVE_PINS  : in  PINSET := ALL_PINS );
  
  function MATCH (
      constant CONNECT      : in  WAVES_MATCH_LIST;
      constant ACTIVE_PIN   : in  TEST_PINS )
    return BOOLEAN;
  
  function MATCH (
      constant CONNECT      : in  WAVES_MATCH_LIST;
      constant ACTIVE_PINS  : in  PINSET := ALL_PINS )
    return BOOLEAN;
  
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- A FRAME SET ARRAY is created in which a FRAME SET is assigned to
  -- some combination of pins and logic values.  If a FRAME SET ARRAY
  -- is two dimensional array of FRAME SETs, then it is described as
  -- FSA(pin,logic).  Then each function creates a FRAME SET ARRAY
  -- such that: 
  --     FSA(P,L) = FRAMES     if P is in PINS and L is in LOGIC
  --     FSA(P,L) is empty     otherwise.
  --
  -- The MERGE FRAME SET ARRAY operation "+" produces a new FRAME SET
  -- ARRAY such that: 
  --     FSA(P,L) = A(P,L)     if A(P,L) is not empty
  --     FSA(P,L) = B(P,L)     otherwise.
  -- Note that B(P,L) may be empty.
  --
  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME;
      PINS               : TEST_PINS;
      CODES              : STRING := PIN_CODES;
      LOGIC              : LOGIC_SET := (others => TRUE) )
    return FRAME_SET_ARRAY;     

  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME;
      PINS               : PINSET;
      CODES              : STRING := PIN_CODES;
      LOGIC              : LOGIC_SET := (others => TRUE) )
    return FRAME_SET_ARRAY;     

  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME_SET;
      PINS               : TEST_PINS )
    return FRAME_SET_ARRAY;
  
  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME_SET;
      PINS               : PINSET )
    return FRAME_SET_ARRAY;
  
  function NEW_FRAME_SET_ARRAY (
      FRAMES             : LOGIC_MAP;
      PINS               : TEST_PINS )
    return FRAME_SET_ARRAY;
  
  function NEW_FRAME_SET_ARRAY (
      FRAMES             : LOGIC_MAP;
      PINS               : PINSET )
    return FRAME_SET_ARRAY;
  
  -- Merge Frame Set Array operator
  --
  function "+" (
      A			 : FRAME_SET_ARRAY;
      B              	 : FRAME_SET_ARRAY )
    return FRAME_SET_ARRAY;
  
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- This function returns a pointer (TIME DATA) to a FRAME SET ARRAY.
  --
  function NEW_TIME_DATA (
      FRAMES             : FRAME_SET_ARRAY )
    return TIME_DATA;
  
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following functions merge the characters of the second string 
  -- into the first string according to the locations specified in the 
  -- control list.  The control list must be exactly as long as the 
  -- second string or must be a single value.  If the control list has 
  -- a single value then the second string is copied into the first 
  -- beginning at the control list point.  The control list may not 
  -- specify character locations outside the range of the first string.
  -- 
  function MERGE_STRING( A : STRING;  B : CHARACTER; CONTROL : INTEGER) 
    return STRING;

  function MERGE_STRING( A : STRING;  B : CHARACTER; CONTROL : TEST_PINS) 
    return STRING;

  function MERGE_STRING( A, B : STRING; CONTROL : INTEGER) 
    return STRING;

  function MERGE_STRING( A, B : STRING; CONTROL : TEST_PINS) 
    return STRING;

  function MERGE_STRING( A, B : STRING; CONTROL : INTEGER_LIST) 
    return STRING;

  function MERGE_STRING( A, B : STRING; CONTROL : TEST_PINS_LIST) 
    return STRING;

-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The following definitions support the fixed file format of Level
  -- 1 WAVES.  The FILE SLICE record stores information most recently
  -- read by the READ FILE SLICE procedure.  The flag END_OF_FILE is
  -- set if an EOF is encountered while scanning for the current file
  -- slice. END_OF_FILE is always FALSE if a wave slice is
  -- successfully (no errors) read. 
  --
  type FILE_SLICE is record
    CODES                 : STD.TEXTIO.LINE;
    FS_TIME               : TIME;
    FS_INTEGER            : INTEGER;
    END_OF_FILE           : BOOLEAN;
  end record;
  
  type FILE_SLICE_LIST is array (POSITIVE range <>) of FILE_SLICE;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- These functions initialize a FILE SLICE to contain a LINE of
  -- the proper length.  The first returns a LINE of length TEST PINS,
  -- and the second returns a LINE as long as the input parameter.
  -- In addition, FS_TIME is initialized to 0 hr, FS_INTEGER to 0, and
  -- END_OF_FILE to FALSE.
  --
  function NEW_FILE_SLICE
    return FILE_SLICE;
  
  function NEW_FILE_SLICE (
      constant LENGTH  : in POSITIVE )
    return FILE_SLICE;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- These functions read a file slice from the input file and store
  -- it in the FILE SLICE record.  If an ACTIVE PINS parameter is
  -- specified then indexing into the file slice is re-ordered by the
  -- ACTIVE PINS parameter.  
  --
  procedure READ_FILE_SLICE (
      variable EX_FILE    : in    TEXT;
      variable FILE_DATA  : inout FILE_SLICE );
  
  procedure READ_FILE_SLICE (
      variable EX_FILE     : in    TEXT;
      variable FILE_DATA   : inout FILE_SLICE;
      constant ACTIVE_PIN : in    TEST_PINS );
  
  procedure READ_FILE_SLICE (
      variable EX_FILE     : in    TEXT;
      variable FILE_DATA   : inout FILE_SLICE;
      constant ACTIVE_PINS : in    TEST_PINS_LIST );
  
end WAVES_OBJECTS;
