--
-- Copyright 1990 
--
-- WAVES Standard Packages Version 3.3.2
-- 25 October 1990
--
-- This code is distributed for the purposes of evaluating the 
-- Waveform And Vector Exchange Specification (WAVES) proposal 
-- presented to the IEEE by the WAVES Analysis and Standardization 
-- Group.  This code may not be used for commercial purposes and 
-- may not be redistributed or published without permission of the 
-- Chairman of the WAVES Analysis and Standardization Group, 
-- Mr Robert Hillman.
--
-- Address comments or questions to:
--    Robert Hillman           
--    RADC/RBRP                
--    Griffis AFB, NY                
--    (315) 330-2241                 
--                                   
--    hillman@tisss.radc.af.mil      
--
use STD.TEXTIO.all;
library WAVES_STANDARD;
use WAVES_STANDARD.WAVES_SYSTEM;
use WAVES_STANDARD.WAVES_EVENTS.all ;
-- A context clause providing visibility to an analyzed copy of 
-- WAVES_INTERFACE is required at this point.
use WAVES_STANDARD.WAVES_INTERFACE.all ;

library WORK ;
use WORK.WAVES_DEVICE.all ;
use WORK.WAVES_FRAMES.all ;
-- previous 5 lines added by GLN on 02-09-91

package WAVES_OBJECTS is
  

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- The ETIME function returns a EVENT_TIME object.  These functions
  -- overload ETIME functions previously defined in the WAVES INTERFACE
  -- but are defined here for visibility to the TEST_PINS object which
  -- is used by the BASE_PIN parameter.
  --
  function ETIME (
      NOMINAL            : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN		 : TEST_PINS)
    return EVENT_TIME;
  
  function ETIME (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME;
      BASE_EVENT          : POSITIVE;
      BASE_PIN		  : TEST_PINS)
    return EVENT_TIME;
  
  function ETIME (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN		 : TEST_PINS)
    return EVENT_TIME;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- Calling DELAY with a TEST PIN and an EVENT NUMBER references the 
  -- event number on the pin.  Calling DELAY with a TEST PIN and a 
  -- LOGIC VALUE causes DELAY to wait for a transition to the specified
  -- logic value on that pin.
  --
  function DELAY (
      NOMINAL            : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL            : TIME;
      BASE_LOGIC         : LOGIC_VALUE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME;
      BASE_LOGIC         : LOGIC_VALUE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME;
      BASE_LOGIC         : LOGIC_VALUE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME;
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The following definitions build up a WAVE TIMING object.
  --
  -- TIME DATA is a pointer to a FRAME SET ARRAY.  A pointer is used
  -- to conserve space, since only a few FRAME SET ARRAY's are typically
  -- defined, but are referenced in many places. 
  --
  -- A WAVE_TIMING combines all the timing information needed for the
  -- APPLY procedure.  It contains a DELAY (one Delay Element) and a
  -- TIME DATA. 
  --
  type TIME_DATA is access FRAME_SET_ARRAY;
  
  type WAVE_TIMING is record
    DELAY                 : DELAY_TIME;
    TIMING                : TIME_DATA;
  end record;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The following definitions allow unconstrained arrays of WAVES
  -- objects.
  --
  type TIME_DATA_LIST is array (POSITIVE range <>) of TIME_DATA;
  
  type WAVE_TIMING_LIST is array (POSITIVE range <>) of WAVE_TIMING;
  
  type TEST_PINS_LIST is array (POSITIVE range <>) of TEST_PINS;
  
  subtype PIN_CODE_STRING is STRING (
      TEST_PINS'POS(TEST_PINS'LEFT) + 1 to
      TEST_PINS'POS(TEST_PINS'RIGHT) + 1);
  
  type PIN_CODE_LIST is array (POSITIVE range <>) of PIN_CODE_STRING;
  
  type INDEX_SLICE is record
      CODES        : PIN_CODE_STRING;
      INDEX        : INTEGER;
  end record;

  type INDEX_SLICE_LIST is array (POSITIVE range <>) of 
      INDEX_SLICE; 

  type TIMED_SLICE is record
      CODES        : PIN_CODE_STRING;
      STIME        : TIME;
  end record;

  type TIMED_SLICE_LIST is array (POSITIVE range <>) of 
      TIMED_SLICE; 

  type WTIME_SLICE is record
      CODES        : PIN_CODE_STRING;
      WTIME        : WAVE_TIMING;
  end record;

  type WTIME_SLICE_LIST is array (POSITIVE range <>) of 
      WTIME_SLICE; 
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The following definitions provide support for pinsets.  A PINSET
  -- is an array of booleans, one for each test pin.  Two deferred
  -- constants are defined, ALL PINS and NO PINS. 
  --
  type PINSET is array (TEST_PINS) of BOOLEAN;
  
  constant ALL_PINS       : PINSET;
  
  constant NO_PINS        : PINSET;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- Three functions are provided to create a pinset.  The first
  -- returns a completely populated pinset.  The second creates a
  -- pinset whose single member is the named test pin.  The third
  -- creates a pinset whose members are those pins named in the list
  -- parameter. 
  --
  function NEW_PINSET
     return PINSET;
  
  function NEW_PINSET (
      PIN                 : in TEST_PINS )
    return PINSET;
  
  function NEW_PINSET (
      PINS                : in TEST_PINS_LIST )
    return PINSET;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following are the basic interface types into the WAVES
  -- dataset.  Waves_Port_List record is the port connects into
  -- the WAVES dataset.  Waves_Match_List record provides Match 
  -- and Delay Logic values. 
  -- 
  subtype WAVES_PORT_LIST is WAVES_SYSTEM.SYSTEM_WAVES_PORT_LIST;

  subtype WAVES_MATCH_LIST is WAVES_SYSTEM.SYSTEM_WAVES_MATCH_LIST;

-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The APPLY procedures to generate waveform slices.
  --
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME;
      constant ACTIVE_PINS : in    TEST_PINS );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME_SET;
      constant ACTIVE_PINS : in    TEST_PINS_LIST );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME;
      constant ACTIVE_PINS : in    TEST_PINS );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME_SET;
      constant ACTIVE_PINS : in    TEST_PINS_LIST );
-- 
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      constant CODES       : in    PIN_CODE_STRING;
      constant DELAY       : in    DELAY_TIME;
      variable FRAMES      : in    TIME_DATA;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant CODES       : in    PIN_CODE_STRING;
      constant DELAY       : in    DELAY_TIME;
      variable FRAMES      : in    TIME_DATA;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      constant CODES       : in    PIN_CODE_STRING;
      variable SLICE       : in    WAVE_TIMING;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS );
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant CODES       : in    PIN_CODE_STRING;
      variable SLICE       : in    WAVE_TIMING;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS );
  
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The TAG procedures apply the string parameter to the output
  -- signals, in effect labeling (tagging) the next call to APPLY. 
  --
  procedure TAG (
      signal CONNECT     : out WAVES_PORT_LIST ;
      constant TAG_LABEL : in STRING );
  
  procedure TAG (
      signal CONNECT     : out WAVES_PORT_LIST ;
      constant TAG_LABEL : in STRING;
      constant PINS      : in PINSET );
  
  procedure TAG (
      signal CONNECT     : out WAVES_PORT_LIST ;
      constant TAG_LABEL : in STRING;
      constant PINS      : in TEST_PINS );
  
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The MATCH procedure enables/disables MATCH mode comparisons on
  -- the specified pins.  The MATCH function samples the results of a
  -- MATCH mode comparison. 
  --

  type MATCH_CONTROL_TYPE is (HOLD, SAMPLE);
  
  procedure MATCH (
      signal   CONNECT      : out WAVES_PORT_LIST;
      constant CONTROL      : in  MATCH_CONTROL_TYPE ;
      constant ACTIVE_PIN   : in  TEST_PINS );
  
  procedure MATCH (
      signal   CONNECT      : out WAVES_PORT_LIST;
      constant CONTROL      : in  MATCH_CONTROL_TYPE;
      constant ACTIVE_PINS  : in  PINSET := ALL_PINS );
  
  function MATCH (
      constant CONNECT      : in  WAVES_MATCH_LIST;
      constant ACTIVE_PIN   : in  TEST_PINS )
    return BOOLEAN;
  
  function MATCH (
      constant CONNECT      : in  WAVES_MATCH_LIST;
      constant ACTIVE_PINS  : in  PINSET := ALL_PINS )
    return BOOLEAN;
  
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- A FRAME SET ARRAY is created in which a FRAME SET is assigned to
  -- some combination of pins and logic values.  If a FRAME SET ARRAY
  -- is two dimensional array of FRAME SETs, then it is described as
  -- FSA(pin,logic).  Then each function creates a FRAME SET ARRAY
  -- such that: 
  --     FSA(P,L) = FRAMES     if P is in PINS and L is in LOGIC
  --     FSA(P,L) is empty     otherwise.
  --
  -- The MERGE FRAME SET ARRAY operation "+" produces a new FRAME SET
  -- ARRAY such that: 
  --     FSA(P,L) = A(P,L)     if A(P,L) is not empty
  --     FSA(P,L) = B(P,L)     otherwise.
  -- Note that B(P,L) may be empty.
  --
  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME;
      PINS               : TEST_PINS;
      CODES              : STRING := PIN_CODES;
      LOGIC              : LOGIC_SET := (others => TRUE) )
    return FRAME_SET_ARRAY;     

  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME;
      PINS               : PINSET;
      CODES              : STRING := PIN_CODES;
      LOGIC              : LOGIC_SET := (others => TRUE) )
    return FRAME_SET_ARRAY;     

  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME_SET;
      PINS               : TEST_PINS )
    return FRAME_SET_ARRAY;
  
  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME_SET;
      PINS               : PINSET )
    return FRAME_SET_ARRAY;
  
  function NEW_FRAME_SET_ARRAY (
      FRAMES             : LOGIC_MAP;
      PINS               : TEST_PINS )
    return FRAME_SET_ARRAY;
  
  function NEW_FRAME_SET_ARRAY (
      FRAMES             : LOGIC_MAP;
      PINS               : PINSET )
    return FRAME_SET_ARRAY;
  
  -- Merge Frame Set Array operator
  --
  function "+" (
      A			 : FRAME_SET_ARRAY;
      B              	 : FRAME_SET_ARRAY )
    return FRAME_SET_ARRAY;
  
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- This function returns a pointer (TIME DATA) to a FRAME SET ARRAY.
  --
  function NEW_TIME_DATA (
      FRAMES             : FRAME_SET_ARRAY )
    return TIME_DATA;
  
-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following functions merge the characters of the second string 
  -- into the first string according to the locations specified in the 
  -- control list.  The control list must be exactly as long as the 
  -- second string or must be a single value.  If the control list has 
  -- a single value then the second string is copied into the first 
  -- beginning at the control list point.  The control list may not 
  -- specify character locations outside the range of the first string.
  -- 
  function MERGE_STRING( A : STRING;  B : CHARACTER; CONTROL : INTEGER) 
    return STRING;

  function MERGE_STRING( A : STRING;  B : CHARACTER; CONTROL : TEST_PINS) 
    return STRING;

  function MERGE_STRING( A, B : STRING; CONTROL : INTEGER) 
    return STRING;

  function MERGE_STRING( A, B : STRING; CONTROL : TEST_PINS) 
    return STRING;

  function MERGE_STRING( A, B : STRING; CONTROL : INTEGER_LIST) 
    return STRING;

  function MERGE_STRING( A, B : STRING; CONTROL : TEST_PINS_LIST) 
    return STRING;

-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The following definitions support the fixed file format of Level
  -- 1 WAVES.  The FILE SLICE record stores information most recently
  -- read by the READ FILE SLICE procedure.  The flag END_OF_FILE is
  -- set if an EOF is encountered while scanning for the current file
  -- slice. END_OF_FILE is always FALSE if a wave slice is
  -- successfully (no errors) read. 
  --
  type FILE_SLICE is record
    CODES                 : STD.TEXTIO.LINE;
    FS_TIME               : TIME;
    FS_INTEGER            : INTEGER;
    END_OF_FILE           : BOOLEAN;
  end record;
  
  type FILE_SLICE_LIST is array (POSITIVE range <>) of FILE_SLICE;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- These functions initialize a FILE SLICE to contain a LINE of
  -- the proper length.  The first returns a LINE of length TEST PINS,
  -- and the second returns a LINE as long as the input parameter.
  -- In addition, FS_TIME is initialized to 0 hr, FS_INTEGER to 0, and
  -- END_OF_FILE to FALSE.
  --
  function NEW_FILE_SLICE
    return FILE_SLICE;
  
  function NEW_FILE_SLICE (
      constant LENGTH  : in POSITIVE )
    return FILE_SLICE;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- These functions read a file slice from the input file and store
  -- it in the FILE SLICE record.  If an ACTIVE PINS parameter is
  -- specified then indexing into the file slice is re-ordered by the
  -- ACTIVE PINS parameter.  
  --
  procedure READ_FILE_SLICE (
      variable EX_FILE    : in    TEXT;
      variable FILE_DATA  : inout FILE_SLICE );
  
  procedure READ_FILE_SLICE (
      variable EX_FILE     : in    TEXT;
      variable FILE_DATA   : inout FILE_SLICE;
      constant ACTIVE_PIN : in    TEST_PINS );
  
  procedure READ_FILE_SLICE (
      variable EX_FILE     : in    TEXT;
      variable FILE_DATA   : inout FILE_SLICE;
      constant ACTIVE_PINS : in    TEST_PINS_LIST );
  
end WAVES_OBJECTS;
--
-- Copyright 1990 
--
-- WAVES Standard Packages Version 3.3.2
-- 25 October 1990
--
-- This code is distributed for the purposes of evaluating the 
-- Waveform And Vector Exchange Specification (WAVES) proposal 
-- presented to the IEEE by the WAVES Analysis and Standardization 
-- Group.  This code may not be used for commercial purposes and 
-- may not be redistributed or published without permission of the 
-- Chairman of the WAVES Analysis and Standardization Group, 
-- Mr Robert Hillman.
--
-- Address comments or questions to:
--    Robert Hillman           
--    RADC/RBRP                
--    Griffis AFB, NY          
--    (315) 330-2241           
--                             
--    hillman@tisss.radc.af.mil
--
library WAVES_STANDARD;
use WAVES_STANDARD.WAVES_SYSTEM.all;
package body WAVES_OBJECTS is

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=- INTERNAL =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- Internal functions used to translate Pin Codes into index values.
  -- A negetive is returned if Code is not in PIN CODES.
  -- 
  function INDEX (
      VALUE              : CHARACTER )
    return INTEGER
  is
  begin
    for C in PIN_CODES'RANGE loop
      if PIN_CODES(C) = VALUE then
        return C;
      end if;
    end loop;
    return -1;
  end INDEX;

  -- Internal function to translate Test Pins into index value.
  --
  function INDEX (
      VALUE              : TEST_PINS )
    return INTEGER
  is
  begin
    return TEST_PINS'POS(VALUE) + 1;
  end INDEX;

  -- Internal function to translate Logic Value into index value.
  --
  function INDEX (
      VALUE              : LOGIC_VALUE )
    return INTEGER
  is
  begin
    return LOGIC_VALUE'POS(VALUE) + 1;
  end INDEX;

  subtype TEST_PIN_RANGE is POSITIVE range 
    INDEX(TEST_PINS'LEFT) to INDEX(TEST_PINS'RIGHT);

--
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- Overloaded ETIME functions which include the BASE_PIN parameter
  -- used for relative timing.
  --
  function ETIME (
      NOMINAL            : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN		 : TEST_PINS)
    return EVENT_TIME
  is
  begin
    return (NOMINAL, (FALSE, 0 ns, 0 ns, BASE_EVENT, INDEX(BASE_PIN)));
  end ETIME;
  
  function ETIME (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME;
      BASE_EVENT          : POSITIVE;
      BASE_PIN		  : TEST_PINS)
    return EVENT_TIME
  is
  begin
    return (NOMINAL, (TRUE, EARLIEST_AND_LATEST, EARLIEST_AND_LATEST,
	    BASE_EVENT, INDEX(BASE_PIN)));
  end ETIME;
  
  function ETIME (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN		 : TEST_PINS)
    return EVENT_TIME
  is
  begin
    return (NOMINAL, (TRUE, EARLIEST, LATEST, BASE_EVENT, INDEX(BASE_PIN)));
  end ETIME;
  

  -- -=-=-=-=-=-=-=-=-=-=-=- DELAY FUNCTIONS -=-=-=-=-=-=-=-=-=-=-=-=-=-=- 
  --
  -- Calling DELAY with a TEST PIN and an EVENT NUMBER references the 
  -- event number on the pin.  Calling DELAY with a TEST PIN and a 
  -- LOGIC VALUE causes DELAY to wait for a transition to the specified
  -- logic value on that pin.
  --
  function DELAY (
      NOMINAL            : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME
  is begin
    return ( NOMINAL, (
        FALSE, 0 hr, 0 hr, 
        WAVES_SYSTEM.TIMED_DELAY, BASE_EVENT, 0, INDEX(BASE_PIN) ));
  end;
  
  function DELAY (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME
  is begin
    return ( NOMINAL, (
        TRUE, EARLIEST_AND_LATEST, EARLIEST_AND_LATEST, 
        WAVES_SYSTEM.TIMED_DELAY, BASE_EVENT, 0, INDEX(BASE_PIN) ));
  end;
  
  function DELAY (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME;
      BASE_EVENT         : POSITIVE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME
  is begin
    return ( NOMINAL, (
        TRUE, EARLIEST, LATEST, 
        WAVES_SYSTEM.TIMED_DELAY, BASE_EVENT, 0, INDEX(BASE_PIN) ));
  end;
  
  function DELAY (
      NOMINAL            : TIME;
      BASE_LOGIC         : LOGIC_VALUE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME
  is begin
    return ( NOMINAL, (
        FALSE, 0 hr, 0 hr, 
        WAVES_SYSTEM.HANDSHAKE_DELAY, 0, INDEX(BASE_LOGIC), INDEX(BASE_PIN) ));
  end;
  
  function DELAY (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME;
      BASE_LOGIC          : LOGIC_VALUE;
      BASE_PIN            : TEST_PINS )
    return DELAY_TIME
  is begin
    return ( NOMINAL, (
        TRUE, EARLIEST_AND_LATEST, EARLIEST_AND_LATEST, 
        WAVES_SYSTEM.HANDSHAKE_DELAY, 0, INDEX(BASE_LOGIC), INDEX(BASE_PIN) ));
  end;
  
  function DELAY (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME;
      BASE_LOGIC         : LOGIC_VALUE;
      BASE_PIN           : TEST_PINS )
    return DELAY_TIME
  is begin
    return ( NOMINAL, (
        TRUE, EARLIEST, LATEST, 
        WAVES_SYSTEM.HANDSHAKE_DELAY, 0, INDEX(BASE_LOGIC), INDEX(BASE_PIN) ));
  end;
-- 
  -- -=-=-=-=-=-=-=-=-=-= PINSET FUNCTIONS =--=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- Two deferred constants.
  -- 
  constant ALL_PINS       : PINSET := (others => TRUE);

  constant NO_PINS        : PINSET := (others => FALSE);

  -- 
  -- Three functions are provided to create a pinset.  The first returns 
  -- a pinset containing all test pins.  The second returns a pinset
  -- containing TEST_PINS.  The third returns a pinset containing
  -- TEST_PINS_LIST
  -- 
  -- Pinsets may be added and subtracted using the predefined VHDL logical 
  -- operators, which operate on boolean arrays.  These are:
  --   function  or ( A, B : PINSET ) return PINSET;
  --   function and ( A, B : PINSET ) return PINSET;
  --   function not ( A, B : PINSET ) return PINSET;
  -- 
  function NEW_PINSET
    return PINSET
  is
  begin
    return (others => TRUE);
  end NEW_PINSET;
  
  function NEW_PINSET ( 
      PIN                 : in TEST_PINS ) 
    return PINSET
  is
    variable PIN_SET      : PINSET := (others => FALSE);
  begin
    PIN_SET(PIN) := TRUE;
    return PIN_SET;
  end NEW_PINSET;

  function NEW_PINSET ( 
      PINS                : in TEST_PINS_LIST ) 
    return PINSET
  is
    variable PIN_SET      : PINSET := (others => FALSE);
  begin
    for PIN in PINS'RANGE loop
      PIN_SET(PINS(PIN)) := TRUE;
    end loop;
    return PIN_SET;
  end NEW_PINSET;

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-= INTERNAL -=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The following is an internal function that returns an event time
  -- tolerance.  Currently the actual time is calculated by adding the 
  -- nominal value to the difference between latest and earliest times.
  --
  function TIME_TOL (E_TIME : WAVES_SYSTEM.SYSTEM_EVENT_TIME) return TIME
  is
  begin
    return E_TIME.NOMINAL + E_TIME.LATEST - E_TIME.EARLIEST;
  end TIME_TOL;

  type MAP_RANGE is array (TEST_PIN_RANGE) of INTEGER;

  -- The following procedure is used to calculate relative timing in
  -- a Frame Set Array.  Since the FSA argument is a pointer (TIME_DATA)
  -- a procedure is used instead of a function.
  -- 
  procedure BASE_TIMING (
    variable FSA     : in TIME_DATA;
    constant FSA_MAP : in MAP_RANGE;
    constant PIN     : in NATURAL;
    constant E_TIME  : in WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    variable R_TIME  : inout TIME)
  is
  begin
    if E_TIME.TOLERANCES_SPECIFIED then
      R_TIME := R_TIME + TIME_TOL(E_TIME);
    else
      R_TIME := R_TIME + E_TIME.NOMINAL;
    end if;
    if E_TIME.BASE_EVENT > 0 and FSA_MAP(PIN) > 0 then
      if E_TIME.BASE_PIN > 0 then
        BASE_TIMING (FSA, FSA_MAP, E_TIME.BASE_PIN, 
	  FSA.all(FSA_MAP(E_TIME.BASE_PIN) + 
	    E_TIME.BASE_EVENT - 1).TAGGED.TIME, R_TIME);
      else	-- Event Time is relative to itself.
        BASE_TIMING (FSA, FSA_MAP, PIN, 
	  FSA.all(FSA_MAP(PIN) + E_TIME.BASE_EVENT - 1).TAGGED.TIME, R_TIME);
      end if;
    end if;
  end BASE_TIMING;

  -- The following function calculates relative timing in a Frame Set.
  -- Timing should only be relative to events in Frames which are
  -- members of the Frame Set.
  -- 
  function BASE_TIMING (
    constant F_SET   : in FRAME_SET;
    constant FSA_MAP : in MAP_RANGE;
    constant PIN     : in NATURAL;
    constant E_TIME  : in WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    constant R_TIME  : in TIME := 0 hr)
  return TIME
  is
    variable TOTAL_TIME : TIME;
  begin
    if E_TIME.TOLERANCES_SPECIFIED then
      TOTAL_TIME := R_TIME + TIME_TOL(E_TIME);
    else
      TOTAL_TIME := R_TIME + E_TIME.NOMINAL;
    end if;
    if E_TIME.BASE_EVENT > 0 and FSA_MAP(PIN) > 0 then
      if E_TIME.BASE_PIN > 0 then
        TOTAL_TIME := BASE_TIMING (F_SET, FSA_MAP, E_TIME.BASE_PIN, 
	  F_SET(FSA_MAP(E_TIME.BASE_PIN) + 
	    E_TIME.BASE_EVENT - 1).TAGGED.TIME, TOTAL_TIME);
      else	-- Event Time is relative to itself.
        TOTAL_TIME := BASE_TIMING (F_SET, FSA_MAP, PIN, 
	  F_SET(FSA_MAP(PIN) + E_TIME.BASE_EVENT - 1).TAGGED.TIME, TOTAL_TIME);
      end if;
    end if;
    return TOTAL_TIME;
  end BASE_TIMING;

  -- The following function calculates relative timing in a Frame.
  -- Timing should only be relative to events in the Frame.
  -- 
  function BASE_TIMING (
    constant A_FRAME : in FRAME;
    constant FSA_MAP : in MAP_RANGE;
    constant PIN     : in NATURAL;
    constant E_TIME  : in WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    constant R_TIME  : in TIME := 0 hr)
  return TIME
  is
    variable TOTAL_TIME : TIME;
  begin
    if E_TIME.TOLERANCES_SPECIFIED then
      TOTAL_TIME := R_TIME + TIME_TOL(E_TIME);
    else
      TOTAL_TIME := R_TIME + E_TIME.NOMINAL;
    end if;
    if E_TIME.BASE_EVENT > 0 and FSA_MAP(PIN) > 0 then
      TOTAL_TIME := BASE_TIMING (A_FRAME, FSA_MAP, PIN, 
	A_FRAME(FSA_MAP(PIN) + E_TIME.BASE_EVENT - 1).TAGGED.TIME, TOTAL_TIME);
    end if;
    return TOTAL_TIME;
  end BASE_TIMING;

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- An internal procedure used to apply HANDSHAKE_DELAY values for a
  -- wave slice.  The delay procedure waits for BASE_PIN to transition
  -- to DELAY_LOGIC which is signaled TRUE by the WAVES MATCH LIST. 
  --
  procedure APPLY_HANDSHAKE (
    signal CONNECT : inout WAVES_PORT_LIST;
    signal   MATCH : in WAVES_MATCH_LIST;
    constant DELAY : in DELAY_TIME)
  is
  begin
    CONNECT(DELAY.DT_BASIS.BASE_PIN).DELAY_LOGIC <= 
	transport DELAY.DT_BASIS.BASE_LOGIC;
    wait until MATCH(DELAY.DT_BASIS.BASE_PIN).D_VALUE; 
    CONNECT(DELAY.DT_BASIS.BASE_PIN).DELAY_LOGIC <= transport -1; 
  end APPLY_HANDSHAKE;

  --
  -- An internal function used to return an EVENT_TIME given an
  -- DELAY_TIME.
  --
  function DELAY_EVENT_TIME (DELAY_VALUE : DELAY_TIME) 
    return WAVES_SYSTEM.SYSTEM_EVENT_TIME
  is
    variable NEW_ETIME : EVENT_TIME;
  begin
    if DELAY_VALUE.DT_BASIS.TOLERANCES_SPECIFIED then
      NEW_ETIME := ETIME(DELAY_VALUE.NOMINAL, DELAY_VALUE.DT_BASIS.EARLIEST,
	DELAY_VALUE.DT_BASIS.LATEST);
    else
      NEW_ETIME := ETIME(DELAY_VALUE.NOMINAL);
    end if;
    return (NEW_ETIME.NOMINAL, 
	NEW_ETIME.ET_BASIS.TOLERANCES_SPECIFIED,
	NEW_ETIME.ET_BASIS.EARLIEST,
	NEW_ETIME.ET_BASIS.LATEST,
	NEW_ETIME.ET_BASIS.BASE_EVENT,
	NEW_ETIME.ET_BASIS.BASE_PIN);
  end DELAY_EVENT_TIME; 


  -- The following procedure produces a map of index values into the 
  -- Frame Set Array (FRAMES) for each Test Pin.  Frame selection 
  -- is based on the Test Pins current Logic Value and a given set
  -- of Pin CODES.  A -1 index indicates an empty frame for that pin. 
  --

  procedure FRAME_MAP (
    signal   CONNECT     : in WAVES_PORT_LIST;
    constant CODES       : in PIN_CODE_STRING;
    variable FRAMES      : in TIME_DATA;
    variable FSA_MAP     : out MAP_RANGE ) 
  is
    variable L_INDEX : INTEGER;
    variable C_INDEX : INTEGER;
  begin
    FSA_MAP := (others => -1);
    for PIN in TEST_PIN_RANGE loop
      assert INDEX(CODES(PIN)) /= -1 
        report "Illegal character code in wave slice."
        severity ERROR;
      if FRAMES.all(PIN).NEXT_INDEX > 0 then
	L_INDEX := FRAMES.all(PIN).NEXT_INDEX + CONNECT(PIN).L_VALUE;
        if FRAMES.all(L_INDEX).NEXT_INDEX > 0 then
	  C_INDEX := FRAMES.all(L_INDEX).NEXT_INDEX + INDEX(CODES(PIN)) - 1;
	  if FRAMES.all(C_INDEX).NEXT_INDEX > 0 then
	    FSA_MAP(PIN) := FRAMES.all(C_INDEX).NEXT_INDEX;
	    assert FRAMES.all(FRAMES.all(C_INDEX).NEXT_INDEX).KIND = 
		WAVES_SYSTEM.T_FRAME
	      report "Warning - Frame Map not mapped to Tagged Frame."
	      severity WARNING;
	  end if;
	end if;
      end if;
    end loop;
  end FRAME_MAP;

  -- The following procedure produces a map of index values into the 
  -- Frame Set for corresponding Test Pins.  The number of ACTIVE_PINS
  -- should correspond to the number of Frames in the FRAME_SET.
  -- A -1 index indicates an empty frame for that pin. 
  --
  procedure FRAME_MAP (
    constant FRAMES  : in FRAME_SET;
    variable FS_MAP  : out MAP_RANGE;
    constant ACTIVE_PINS : in TEST_PINS_LIST)
  is
    variable PIN        : INTEGER;
    variable F_INDEX    : INTEGER := 1;
    variable AN_EVENT   : INTEGER;
    constant LAST_EVENT : INTEGER := 0;
  begin
    assert FRAMES'LENGTH > 0
      report "Error - Empty frame set."
      severity ERROR;

    FS_MAP := (others => -1);
    -- Create a Map for a Frame Set. 
    for I in ACTIVE_PINS'RANGE loop
      PIN := INDEX(ACTIVE_PINS(I));
      FS_MAP(PIN) := F_INDEX;
      loop
	assert F_INDEX <= FRAMES'LENGTH
	  report "Error - Missing frame in Frame Set."
	  severity ERROR;
        AN_EVENT := FRAMES(F_INDEX).NEXT_INDEX;
	exit when AN_EVENT <= LAST_EVENT;
	F_INDEX := AN_EVENT;
      end loop;
      F_INDEX := F_INDEX + 1;
    end loop;
  end FRAME_MAP;

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- The following are three overloaded internal procedures used to apply 
  -- Frame Set Arrays, Frame Sets, and a Frame to a wave slice.

  -- The first procedure takes a pointer from a Frame Set Array (TIME_DATA) 
  -- and applies frames, corresponding to the PIN_CODE_STRING, for
  -- each active test pin. 
  
  procedure APPLY_FRAMES (
    signal   CONNECT     : inout WAVES_PORT_LIST;
    constant CODES       : in PIN_CODE_STRING;
    variable FRAMES      : in TIME_DATA;
    constant FSA_MAP     : in MAP_RANGE;
    constant ACTIVE_PINS : in PINSET)
  is
    variable REL_TIME      : TIME;
    variable AN_EVENT      : INTEGER;
    constant EMPTY_EVENT   : INTEGER := -1;
    constant LAST_EVENT    : NATURAL :=  0;
  begin
    for PIN in TEST_PIN_RANGE loop
      if ACTIVE_PINS(TEST_PINS'VAL(PIN - 1)) then
	if FSA_MAP(PIN) /= EMPTY_EVENT then
          -- Get first event in the frame 
          AN_EVENT := FSA_MAP(PIN);
          loop
	    REL_TIME := 0 hr;	-- Init relative time to zero. 
	    -- BASE_TIMING is a procedure because it's passed a pointer.
	    BASE_TIMING (FRAMES, FSA_MAP, PIN,
	       FRAMES.all(AN_EVENT).TAGGED.TIME, REL_TIME);
            if FRAMES.all(AN_EVENT).NEXT_INDEX /= EMPTY_EVENT then
              CONNECT(PIN).L_VALUE <= transport 
                FRAMES(AN_EVENT).TAGGED.VALUE after REL_TIME; 
            end if;
            AN_EVENT := FRAMES.all(AN_EVENT).NEXT_INDEX;
            exit when AN_EVENT <= LAST_EVENT; 
          end loop;
	end if;
      end if;
    end loop;
  end APPLY_FRAMES;


  -- The next procedure applies Frame Sets to a list of corresponding
  -- test pins.

  procedure APPLY_FRAMES (
    signal   CONNECT     : inout WAVES_PORT_LIST;
    constant FRAMES      : in FRAME_SET;
    constant FS_MAP      : in MAP_RANGE;
    constant ACTIVE_PINS : in TEST_PINS_LIST)
  is
    variable PIN           : INTEGER;
    variable AN_EVENT      : INTEGER;
    variable F_INDEX       : INTEGER :=  1;
    constant EMPTY_EVENT   : INTEGER := -1;
    constant LAST_EVENT    : NATURAL :=  0;
  begin
    for I in ACTIVE_PINS'RANGE loop
      PIN := INDEX(ACTIVE_PINS(I));
      loop
        AN_EVENT := FRAMES(F_INDEX).NEXT_INDEX;
        if AN_EVENT /= EMPTY_EVENT then
          CONNECT(PIN).L_VALUE <= transport 
              FRAMES(F_INDEX).TAGGED.VALUE 
	  after BASE_TIMING(FRAMES, FS_MAP, PIN, FRAMES(F_INDEX).TAGGED.TIME);
        end if;
	exit when AN_EVENT <= LAST_EVENT;
	F_INDEX := AN_EVENT;
      end loop;
      F_INDEX := F_INDEX + 1;
    end loop;
  end APPLY_FRAMES;


  -- The following procedure applies a frame to a test pin.

  procedure APPLY_FRAMES (
    signal   CONNECT     : inout WAVES_PORT_LIST;
    constant FRAMES      : in FRAME;
    constant F_MAP       : MAP_RANGE;
    constant ACTIVE_PIN  : in TEST_PINS)
  is
    variable PIN           : INTEGER;
    variable AN_EVENT      : INTEGER :=  1;
    constant EMPTY_EVENT   : INTEGER := -1;
    constant LAST_EVENT    : NATURAL :=  0;
  begin
    PIN := INDEX(ACTIVE_PIN);
    loop
      if FRAMES(AN_EVENT).NEXT_INDEX /= EMPTY_EVENT then
        CONNECT(PIN).L_VALUE <= transport 
            FRAMES(AN_EVENT).TAGGED.VALUE
	after BASE_TIMING(FRAMES, F_MAP, PIN, FRAMES(AN_EVENT).TAGGED.TIME);
      end if;
      AN_EVENT := FRAMES(AN_EVENT).NEXT_INDEX;
      exit when AN_EVENT <= LAST_EVENT; 
    end loop;
  end APPLY_FRAMES;

  -- -=-=-=-=-=-=-=-=-=-= APPLY PROCEDURES -=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- The APPLY procedures used to generate waveform slices.
  --
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME;
      constant ACTIVE_PINS : in    TEST_PINS)
  is
    variable F_MAP : MAP_RANGE := (others => -1);
  begin
    assert FRAMES'LENGTH > 0
      report "Error - Empty frame."
      severity ERROR;
    F_MAP(INDEX(ACTIVE_PINS)) := 1;
    APPLY_FRAMES (CONNECT, FRAMES, F_MAP, ACTIVE_PINS);
    assert DELAY.DT_BASIS.DELAY_TYPE = WAVES_SYSTEM.TIMED_DELAY 
      report "Error - TIMED DELAY value expected."
      severity ERROR;
    wait for BASE_TIMING(FRAMES, F_MAP, DELAY.DT_BASIS.BASE_PIN, 
	DELAY_EVENT_TIME(DELAY));
  end APPLY;

  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST ;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME;
      constant ACTIVE_PINS : in    TEST_PINS)
  is
    variable F_MAP : MAP_RANGE := (others => -1);
  begin
    assert FRAMES'LENGTH > 0
      report "Error - Empty frame."
      severity ERROR;
    F_MAP(INDEX(ACTIVE_PINS)) := 1;
    APPLY_FRAMES (CONNECT, FRAMES, F_MAP, ACTIVE_PINS);
    assert DELAY.DT_BASIS.DELAY_TYPE = WAVES_SYSTEM.HANDSHAKE_DELAY
      report "Error - HANDSHAKE DELAY value expected."
      severity ERROR;
    APPLY_HANDSHAKE (CONNECT, MATCH, DELAY);
    wait for BASE_TIMING(FRAMES, F_MAP, DELAY.DT_BASIS.BASE_PIN, 
	DELAY_EVENT_TIME(DELAY));
  end APPLY;

  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST ;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME_SET;
      constant ACTIVE_PINS : in    TEST_PINS_LIST)
  is
    variable FS_MAP : MAP_RANGE;
  begin
    assert FRAMES'LENGTH > 0
      report "Error - Empty frame set."
      severity ERROR;
    FRAME_MAP(FRAMES, FS_MAP, ACTIVE_PINS);
    APPLY_FRAMES (CONNECT, FRAMES, FS_MAP, ACTIVE_PINS);
    assert DELAY.DT_BASIS.DELAY_TYPE = WAVES_SYSTEM.TIMED_DELAY 
      report "Error - TIMED DELAY value expected."
      severity ERROR;
    wait for BASE_TIMING(FRAMES, FS_MAP, DELAY.DT_BASIS.BASE_PIN, 
	DELAY_EVENT_TIME(DELAY)); 
  end APPLY;
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST ;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant DELAY       : in    DELAY_TIME;
      constant FRAMES      : in    FRAME_SET;
      constant ACTIVE_PINS : in    TEST_PINS_LIST)
  is
    variable FS_MAP : MAP_RANGE;
  begin
    assert FRAMES'LENGTH > 0
      report "Error - Empty frame set."
      severity ERROR;
    FRAME_MAP(FRAMES, FS_MAP, ACTIVE_PINS);
    APPLY_FRAMES (CONNECT, FRAMES, FS_MAP, ACTIVE_PINS);
    assert DELAY.DT_BASIS.DELAY_TYPE = WAVES_SYSTEM.HANDSHAKE_DELAY
      report "Error - HANDSHAKE DELAY value expected."
      severity ERROR;
    APPLY_HANDSHAKE (CONNECT, MATCH, DELAY);
    wait for BASE_TIMING(FRAMES, FS_MAP, DELAY.DT_BASIS.BASE_PIN, 
	DELAY_EVENT_TIME(DELAY)); 
  end APPLY;
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST ;
      constant CODES       : in    PIN_CODE_STRING;
      constant DELAY       : in    DELAY_TIME;
      variable FRAMES      : in    TIME_DATA;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS )
  is
    variable FSA_MAP  : MAP_RANGE;
    variable REL_TIME : TIME := 0 hr;
  begin
    if FRAMES = null then 
      assert FALSE
        report "Malformed wave slice - missing time data."
        severity ERROR;
    else 
      -- Create a Fsa map of current Pin Code String for relative
      -- timing.
      FRAME_MAP(CONNECT, CODES, FRAMES, FSA_MAP);
 
      APPLY_FRAMES (CONNECT, CODES, FRAMES, FSA_MAP, ACTIVE_PINS);
      assert DELAY.DT_BASIS.DELAY_TYPE = WAVES_SYSTEM.TIMED_DELAY 
	report "Error - TIMED DELAY value expected."
	severity ERROR;
      BASE_TIMING(FRAMES, FSA_MAP, DELAY.DT_BASIS.BASE_PIN, 
	DELAY_EVENT_TIME(DELAY), REL_TIME); 
      wait for REL_TIME;
    end if;
  end APPLY;

  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST ;
      constant CODES       : in    PIN_CODE_STRING;
      variable SLICE       : in    WAVE_TIMING;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS )
  is
    variable FSA_MAP  : MAP_RANGE;
    variable REL_TIME : TIME := 0 hr;
  begin
    if SLICE.TIMING = null then 
      assert FALSE
        report "Malformed wave slice - missing time data."
        severity ERROR;
    else 
      -- Create a Fsa map of current Pin Code String for relative
      -- timing.
      FRAME_MAP(CONNECT, CODES, SLICE.TIMING, FSA_MAP);
 
      APPLY_FRAMES (CONNECT, CODES, SLICE.TIMING, FSA_MAP, ACTIVE_PINS);
      assert SLICE.DELAY.DT_BASIS.DELAY_TYPE = WAVES_SYSTEM.TIMED_DELAY 
	report "Error - TIMED DELAY value expected."
	severity ERROR;
      BASE_TIMING(SLICE.TIMING, FSA_MAP, SLICE.DELAY.DT_BASIS.BASE_PIN, 
	DELAY_EVENT_TIME(SLICE.DELAY), REL_TIME); 
      wait for REL_TIME;
    end if;
  end APPLY;
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST ;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant CODES       : in    PIN_CODE_STRING;
      constant DELAY       : in    DELAY_TIME;
      variable FRAMES      : in    TIME_DATA;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS )
  is
    variable FSA_MAP  : MAP_RANGE;
    variable REL_TIME : TIME := 0 hr;
  begin
    if FRAMES = null then 
      assert FALSE
        report "Malformed wave slice - missing time data."
        severity ERROR;
    else 
      -- Create a Fsa map of current Pin Code String for relative
      -- timing.
      FRAME_MAP(CONNECT, CODES, FRAMES, FSA_MAP);
 
      APPLY_FRAMES (CONNECT, CODES, FRAMES, FSA_MAP, ACTIVE_PINS);
      assert DELAY.DT_BASIS.DELAY_TYPE = WAVES_SYSTEM.HANDSHAKE_DELAY
	report "Error - HANDSHAKE DELAY value expected."
	severity ERROR;
      APPLY_HANDSHAKE (CONNECT, MATCH, DELAY);
      BASE_TIMING(FRAMES, FSA_MAP, DELAY.DT_BASIS.BASE_PIN, 
	DELAY_EVENT_TIME(DELAY), REL_TIME); 
      wait for REL_TIME;
    end if;
  end APPLY;
  
  procedure APPLY (
      signal   CONNECT     : inout WAVES_PORT_LIST ;
      signal   MATCH       : in    WAVES_MATCH_LIST;
      constant CODES       : in    PIN_CODE_STRING;
      variable SLICE       : in    WAVE_TIMING;
      constant ACTIVE_PINS : in    PINSET := ALL_PINS )
  is
    variable FSA_MAP  : MAP_RANGE;
    variable REL_TIME : TIME := 0 hr;
  begin
    if SLICE.TIMING = null then 
      assert FALSE
        report "Malformed wave slice - missing time data."
        severity ERROR;
    else 
      -- Create a Fsa map of current Pin Code String for relative
      -- timing.
      FRAME_MAP(CONNECT, CODES, SLICE.TIMING, FSA_MAP);
 
      APPLY_FRAMES (CONNECT, CODES, SLICE.TIMING, FSA_MAP, ACTIVE_PINS);
      assert SLICE.DELAY.DT_BASIS.DELAY_TYPE = WAVES_SYSTEM.HANDSHAKE_DELAY
	report "Error - HANDSHAKE DELAY value expected."
	severity ERROR;
      APPLY_HANDSHAKE (CONNECT, MATCH, SLICE.DELAY);
      BASE_TIMING(SLICE.TIMING, FSA_MAP, SLICE.DELAY.DT_BASIS.BASE_PIN, 
	DELAY_EVENT_TIME(SLICE.DELAY), REL_TIME); 
      wait for REL_TIME;
    end if;
  end APPLY;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-= TAG PROCEDURES -=-=-=-=-=-=-=-=-=-=-=-=-=-=- 
  -- 
  -- This procedure applies a WAVES label to the output ports of the 
  -- waveform generator.
  -- 

  procedure TAG (
      signal CONNECT     : out WAVES_PORT_LIST;
      constant TAG_LABEL : in STRING )
  is
  begin
    for I in TAG_LABEL'RANGE loop
      CONNECT(CONNECT'LEFT).TAG_VALUE <= transport TAG_LABEL(I);
      wait for 0 hr;
    end loop;
    CONNECT(CONNECT'LEFT).TAG_VALUE <= transport CR;
    wait for 0 hr;
  end TAG;

  procedure TAG (
      signal CONNECT     : out WAVES_PORT_LIST;
      constant TAG_LABEL : in STRING;
      constant PINS      : in PINSET ) 
  is
  begin
    for P in TEST_PIN_RANGE loop
      if PINS(TEST_PINS'VAL(P - 1)) then
        for I in TAG_LABEL'RANGE loop
           CONNECT(P).TAG_VALUE <= transport TAG_LABEL(I);
           wait for 0 hr;
        end loop;
        CONNECT(P).TAG_VALUE <= transport CR;
        wait for 0 hr;
      end if;
    end loop;
  end TAG;

  procedure TAG (
      signal CONNECT     : out WAVES_PORT_LIST;
      constant TAG_LABEL : in STRING;
      constant PINS      : in TEST_PINS )
  is
  begin
    for I in TAG_LABEL'RANGE loop
      CONNECT(INDEX(PINS)).TAG_VALUE <= transport TAG_LABEL(I);
      wait for 0 hr;
    end loop;
    CONNECT(INDEX(PINS)).TAG_VALUE <= transport CR;
    wait for 0 hr;
  end TAG;

  -- -=-=-=-=-=-=-=-=-=-=- MATCH FUNCTIONS -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- 
  -- 
  -- This function returns the AND of the MATCH signals of the specified 
  -- pins.
  -- 
  function MATCH (
      constant CONNECT             : in WAVES_MATCH_LIST;
      constant ACTIVE_PINS         : in PINSET := ALL_PINS)
    return BOOLEAN
  is
    variable FLAG        : BOOLEAN := TRUE;
  begin
    for PIN in TEST_PIN_RANGE loop
      if ACTIVE_PINS(TEST_PINS'VAL(PIN - 1)) then 
        FLAG := FLAG and CONNECT(PIN).M_VALUE;
      end if;
    end loop;
    return FLAG;
  end MATCH;


  function MATCH (
      constant CONNECT             : in WAVES_MATCH_LIST;
      constant ACTIVE_PIN          : in TEST_PINS )
    return BOOLEAN
  is
  begin
    return CONNECT(INDEX(ACTIVE_PIN)).M_VALUE;
  end MATCH;

  -- The following MATCH procedures are used to set the match control
  -- (HOLD or SAMPLE) on the Waves Port List.  The first procedure 
  -- controls a single Test Pin and the second procedure controls 
  -- multiple Test Pins (PINSET). 
  --
  procedure MATCH (
      signal CONNECT       : out WAVES_PORT_LIST;
      constant CONTROL     : in MATCH_CONTROL_TYPE;
      constant ACTIVE_PIN  : in TEST_PINS)
  is
  begin
    if CONTROL = HOLD then 
      CONNECT(INDEX(ACTIVE_PIN)).M_CONTROL <= transport WAVES_SYSTEM.HOLD;
    else 
      CONNECT(INDEX(ACTIVE_PIN)).M_CONTROL <= transport WAVES_SYSTEM.SAMPLE;
    end if;
  end MATCH;

  procedure MATCH (
      signal CONNECT       : out WAVES_PORT_LIST;
      constant CONTROL     : in MATCH_CONTROL_TYPE;
      constant ACTIVE_PINS : in PINSET := ALL_PINS)
  is
    variable TEMP : WAVES_SYSTEM.SYSTEM_MATCH_CONTROL;
  begin
    if CONTROL = HOLD then 
      TEMP := WAVES_SYSTEM.HOLD;
    else 
      TEMP := WAVES_SYSTEM.SAMPLE;
    end if;
    for PIN in TEST_PIN_RANGE loop
      if ACTIVE_PINS(TEST_PINS'VAL(PIN - 1)) then 
        CONNECT(PIN).M_CONTROL <= transport TEMP;
      end if;
    end loop; 
  end MATCH;

--
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following function copies the characters of the second string 
  -- into the first string according to the locations specified in the 
  -- control list.  The control list must be exactly as long as the 
  -- second string or must be a single value.  If the control list has 
  -- a single value then the second string is copied into the first 
  -- beginning at the control list point.  The control list may not 
  -- specify character locations outside the range of the first string.
  -- 
  function MERGE_STRING( A : STRING;  B : CHARACTER;  CONTROL : INTEGER )
    return STRING
  is
    variable X : STRING(A'RANGE) := A;
  begin
    X(CONTROL) := B;
    return X;
  end;

  function MERGE_STRING( A : STRING;  B : CHARACTER;  CONTROL : TEST_PINS )
    return STRING
  is
    variable X : STRING(A'RANGE) := A;
  begin
    X(INDEX(CONTROL)) := B;
    return X;
  end;

  function MERGE_STRING( A, B : STRING; CONTROL : INTEGER) 
    return STRING
  is
    variable X : STRING(A'RANGE) := A;
  begin
    for I in B'RANGE loop 
      X(I + CONTROL - 1) := B(I);
    end loop;
    return X;
  end;

  function MERGE_STRING( A, B : STRING; CONTROL : TEST_PINS) 
    return STRING
  is
    variable X : STRING(A'RANGE) := A;
    variable DELTA : INTEGER := INDEX(CONTROL);
  begin
    for I in B'RANGE loop 
      X(DELTA) := B(I);
    end loop;
    return X;
  end;

  function MERGE_STRING( A, B : STRING; CONTROL : INTEGER_LIST) 
    return STRING
  is
    variable X : STRING(A'RANGE) := A;
  begin
    for I in B'RANGE loop
      X(CONTROL(I)) := B(I);
    end loop;
    return X;
  end;

  function MERGE_STRING( A, B : STRING; CONTROL : TEST_PINS_LIST) 
    return STRING
  is
    variable X : STRING(A'RANGE) := A;
  begin
    for I in B'RANGE loop
      X(INDEX(CONTROL(I))) := B(I);
    end loop;
    return X;
  end;

-- 
  -- -=-=-=-=-=-=-= NEW_FRAME_SET_ARRAY FUNCTIONS =-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following functions are used to create Frame Set Arrays given
  -- a Frame or Frame Set.
  -- 

  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME;
      PINS               : TEST_PINS;
      CODES              : STRING := PIN_CODES;
      LOGIC              : LOGIC_SET := (others => TRUE) )
    return FRAME_SET_ARRAY
  is
    constant LOGIC_INDEX : INTEGER := TEST_PIN_RANGE'RIGHT + 1;
    variable CODE_INDEX  : INTEGER := LOGIC_INDEX + 
			   INDEX(LOGIC_VALUE'RIGHT);
    variable INIT_TIME   : WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    variable FRAME_INDEX : INTEGER := CODE_INDEX + PIN_CODES'LENGTH;
    variable FSA         : FRAME_SET_ARRAY
	(1 to (FRAME_INDEX - 1 + FRAMES'LENGTH))
	:= (others => (WAVES_SYSTEM.UNUSED, -1, (0, INIT_TIME)));
  begin
    -- Set Test Pin Logic Value index.
    FSA(INDEX(PINS)).KIND := WAVES_SYSTEM.T_PIN;
    FSA(INDEX(PINS)).NEXT_INDEX := LOGIC_INDEX; 

    -- Set Logic Value Pin Code index
    for I in LOGIC_INDEX to CODE_INDEX - 1 loop
      if LOGIC(LOGIC_VALUE'VAL(I - TEST_PIN_RANGE'RIGHT - 1)) then
        FSA(I).KIND := WAVES_SYSTEM.T_LOGIC;
        FSA(I).NEXT_INDEX := CODE_INDEX;
    end if;
    end loop;

    -- Set Pin Code Frame index 
    for I in CODES'RANGE loop
      if INDEX(CODES(I)) > 0 then
        FSA(CODE_INDEX + INDEX(CODES(I)) - 1).KIND := WAVES_SYSTEM.T_CODE;
        FSA(CODE_INDEX + INDEX(CODES(I)) - 1).NEXT_INDEX := FRAME_INDEX;
      else
	assert FALSE
	  report "Error - Invalid Pin Code"
	  severity ERROR; 
      end if;
    end loop;

    -- Insert FRAMES into Frame Set Array
    assert FRAMES'LENGTH > 0
      report "Empty frame in NEW FRAME SET ARRAY."
      severity ERROR;
    for I in FRAMES'RANGE loop
      FSA(FRAME_INDEX + I - 1) := FRAMES(I);
      if FRAMES(I).NEXT_INDEX > 0 then
        FSA(FRAME_INDEX + I - 1).NEXT_INDEX := FRAME_INDEX + I;
      end if;
    end loop;
    return FSA;
  end NEW_FRAME_SET_ARRAY;

  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME;
      PINS               : PINSET;
      CODES              : STRING := PIN_CODES;
      LOGIC              : LOGIC_SET := (others => TRUE) )
    return FRAME_SET_ARRAY     
  is
    constant LOGIC_INDEX : INTEGER := TEST_PIN_RANGE'RIGHT + 1;
    variable CODE_INDEX  : INTEGER := LOGIC_INDEX + 
			   INDEX(LOGIC_VALUE'RIGHT);
    variable INIT_TIME   : WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    variable FRAME_INDEX : INTEGER := CODE_INDEX + PIN_CODES'LENGTH;
    variable FSA         : FRAME_SET_ARRAY
	(1 to (FRAME_INDEX - 1 + FRAMES'LENGTH))
	:= (others => (WAVES_SYSTEM.UNUSED, -1, (0, INIT_TIME)));
  begin
    -- Set Test Pin(s) Logic Value index
    for PIN in TEST_PIN_RANGE loop
      if PINS(TEST_PINS'VAL(PIN - 1)) then
	FSA(PIN).KIND := WAVES_SYSTEM.T_PIN;
	FSA(PIN).NEXT_INDEX := LOGIC_INDEX;
      end if;
    end loop;

    -- Set Logic Value Pin Code index
    for I in LOGIC_INDEX to CODE_INDEX - 1 loop
      if LOGIC(LOGIC_VALUE'VAL(I - TEST_PIN_RANGE'RIGHT - 1)) then
        FSA(I).KIND := WAVES_SYSTEM.T_LOGIC;
        FSA(I).NEXT_INDEX := CODE_INDEX;
    end if;
    end loop;

    -- Set Pin Code Frame index 
    for I in CODES'RANGE loop
      if INDEX(CODES(I)) > 0 then 
        FSA(CODE_INDEX + INDEX(CODES(I)) - 1).KIND := WAVES_SYSTEM.T_CODE;
        FSA(CODE_INDEX + INDEX(CODES(I)) - 1).NEXT_INDEX := FRAME_INDEX;
      else
	assert FALSE
	  report "Error - Invalid Pin Code"
	  severity ERROR; 
      end if;
    end loop;

    -- Insert FRAMES into Frame Set Array
    assert FRAMES'LENGTH > 0
      report "Empty frame in NEW FRAME SET ARRAY."
      severity ERROR;
    for I in FRAMES'RANGE loop
      FSA(FRAME_INDEX + I - 1) := FRAMES(I);
      if FRAMES(I).NEXT_INDEX > 0 then
        FSA(FRAME_INDEX + I - 1).NEXT_INDEX := FRAME_INDEX + I;
      end if;
    end loop;
    return FSA;
  end NEW_FRAME_SET_ARRAY;

  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME_SET;
      PINS               : TEST_PINS)
    return FRAME_SET_ARRAY
  is
    constant LOGIC_INDEX : INTEGER := TEST_PIN_RANGE'RIGHT + 1;
    variable CODE_INDEX  : INTEGER := LOGIC_INDEX + 
			   INDEX(LOGIC_VALUE'RIGHT);
    variable FRAME_INDEX : INTEGER := CODE_INDEX + PIN_CODES'LENGTH;
    variable INIT_TIME   : WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    variable FSA         : FRAME_SET_ARRAY
	(1 to (FRAME_INDEX - 1 + FRAMES'LENGTH))
	:= (others => (WAVES_SYSTEM.UNUSED, -1, (0, INIT_TIME)));
  begin
    -- Set Test Pin Logic Value index.
    FSA(INDEX(PINS)).KIND := WAVES_SYSTEM.T_PIN; 
    FSA(INDEX(PINS)).NEXT_INDEX := LOGIC_INDEX; 

    -- Set Logic Value Pin Code index.
    for I in LOGIC_INDEX to CODE_INDEX - 1 loop
      FSA(I).KIND := WAVES_SYSTEM.T_LOGIC;
      FSA(I).NEXT_INDEX := CODE_INDEX;
    end loop;

    -- Set Pin Code Frame index 
    FSA(CODE_INDEX).KIND := WAVES_SYSTEM.T_CODE;
    FSA(CODE_INDEX).NEXT_INDEX := FRAME_INDEX;
    for I in FRAMES'RANGE loop
      FSA(FRAME_INDEX + I - 1) := FRAMES(I);
      if FRAMES(I).NEXT_INDEX <= 0 then
	CODE_INDEX := CODE_INDEX + 1;
	if CODE_INDEX < FRAME_INDEX then
	  FSA(CODE_INDEX).KIND := WAVES_SYSTEM.T_CODE; 
	  FSA(CODE_INDEX).NEXT_INDEX := FRAME_INDEX + I; 
	end if;
      else
	FSA(FRAME_INDEX + I - 1).NEXT_INDEX := FRAME_INDEX + I;
      end if;
    end loop;

    assert CODE_INDEX = FRAME_INDEX 
        report "Frame Set does not match number of Pin Codes."
        severity ERROR;
  return FSA;
  end NEW_FRAME_SET_ARRAY;

  function NEW_FRAME_SET_ARRAY (
      FRAMES             : FRAME_SET;
      PINS               : PINSET)
    return FRAME_SET_ARRAY
  is
    constant LOGIC_INDEX : INTEGER := TEST_PIN_RANGE'RIGHT + 1;
    variable CODE_INDEX  : INTEGER := LOGIC_INDEX + 
			   INDEX(LOGIC_VALUE'RIGHT);
    variable FRAME_INDEX : INTEGER := CODE_INDEX + PIN_CODES'LENGTH;
    variable INIT_TIME   : WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    variable FSA         : FRAME_SET_ARRAY
	(1 to (FRAME_INDEX - 1 + FRAMES'LENGTH))
	:= (others => (WAVES_SYSTEM.UNUSED, -1, (0, INIT_TIME)));
    variable SET_FRAME   : BOOLEAN := FALSE;
  begin
    -- Set Test Pin(s) Logic Value index
    for PIN in TEST_PIN_RANGE loop
      if PINS(TEST_PINS'VAL(PIN - 1)) then
	SET_FRAME := TRUE;
	FSA(PIN).KIND := WAVES_SYSTEM.T_PIN;
	FSA(PIN).NEXT_INDEX := LOGIC_INDEX;
      end if;
    end loop;

    if SET_FRAME then
      -- Set Logic Value Pin Code index
      for I in LOGIC_INDEX to CODE_INDEX - 1 loop
        FSA(I).KIND := WAVES_SYSTEM.T_LOGIC;
        FSA(I).NEXT_INDEX := CODE_INDEX;
      end loop;

      -- Set Pin Code Frame index 
      FSA(CODE_INDEX).KIND := WAVES_SYSTEM.T_CODE;
      FSA(CODE_INDEX).NEXT_INDEX := FRAME_INDEX;
      for I in FRAMES'RANGE loop
        FSA(FRAME_INDEX + I - 1) := FRAMES(I);
        if FRAMES(I).NEXT_INDEX <= 0 then
	  CODE_INDEX := CODE_INDEX + 1;
	  if CODE_INDEX < FRAME_INDEX then
	    FSA(CODE_INDEX).KIND := WAVES_SYSTEM.T_CODE; 
	    FSA(CODE_INDEX).NEXT_INDEX := FRAME_INDEX + I;
	  end if;
        else
	  FSA(FRAME_INDEX + I - 1).NEXT_INDEX := FRAME_INDEX + I;
        end if;
      end loop;

      assert CODE_INDEX = FRAME_INDEX 
        report "Frame Set does not match number of Pin Codes."
	severity ERROR;
    end if;
    return FSA;
  end NEW_FRAME_SET_ARRAY;

  function NEW_FRAME_SET_ARRAY (
      FRAMES             : LOGIC_MAP;
      PINS               : TEST_PINS )
    return FRAME_SET_ARRAY
  is
    constant LOGIC_INDEX  : INTEGER := TEST_PIN_RANGE'RIGHT + 1;
    variable OFF_SET      : INTEGER := TEST_PIN_RANGE'RIGHT;
    variable CODE_INDEX   : INTEGER := LOGIC_INDEX + 
			    INDEX(LOGIC_VALUE'RIGHT);
    variable CODES_LENGTH : INTEGER := 0;
    variable FRAME_INDEX  : INTEGER := CODE_INDEX + PIN_CODES'LENGTH;
    variable INIT_TIME    : WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    variable FSA          : FRAME_SET_ARRAY
	(1 to (FRAME_INDEX - 1 + FRAMES'LENGTH))
	:= (others => (WAVES_SYSTEM.UNUSED, -1, (0, INIT_TIME)));
  begin
    --  Get Pin Codes length of Logic Map
    for I in INDEX(LOGIC_VALUE'RIGHT) + 1 to FRAMES'LENGTH loop
      exit when FRAMES(I).KIND /= WAVES_SYSTEM.T_CODE;
      CODES_LENGTH := CODES_LENGTH + 1;
    end loop;

    assert CODES_LENGTH = PIN_CODES'LENGTH
      report "Error - Logic Map doesn't have correct number of Pin Codes."
      severity ERROR;
 
    -- Tag Test Pin 
    FSA(INDEX(PINS)).KIND := WAVES_SYSTEM.T_PIN;

    -- Set Test Pin index to Logic Values.
    FSA(INDEX(PINS)).NEXT_INDEX := LOGIC_INDEX;

    -- Add Logic Map Frames to Frame Set Array
    for I in FRAMES'RANGE loop
      FSA(I + OFF_SET) := FRAMES(I);
      if FRAMES(I).NEXT_INDEX > 0 then
        FSA(I + OFF_SET).NEXT_INDEX := FRAMES(I).NEXT_INDEX + OFF_SET;
      end if;
    end loop; 
    return FSA;
  end NEW_FRAME_SET_ARRAY;

  
  function NEW_FRAME_SET_ARRAY (
      FRAMES             : LOGIC_MAP;
      PINS               : PINSET )
    return FRAME_SET_ARRAY
  is
    constant LOGIC_INDEX  : INTEGER := TEST_PIN_RANGE'RIGHT + 1;
    variable OFF_SET      : INTEGER := TEST_PIN_RANGE'RIGHT;
    variable CODE_INDEX   : INTEGER := LOGIC_INDEX + 
			    INDEX(LOGIC_VALUE'RIGHT);
    variable CODES_LENGTH : INTEGER := 0;
    variable FRAME_INDEX  : INTEGER := CODE_INDEX + PIN_CODES'LENGTH;
    variable INIT_TIME    : WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    variable FSA          : FRAME_SET_ARRAY
	(1 to (FRAME_INDEX - 1 + FRAMES'LENGTH))
	:= (others => (WAVES_SYSTEM.UNUSED, -1, (0, INIT_TIME)));
  begin
    --  Get Pin Codes length of Logic Map
    for I in INDEX(LOGIC_VALUE'RIGHT) + 1 to FRAMES'LENGTH loop
      exit when FRAMES(I).KIND /= WAVES_SYSTEM.T_CODE;
      CODES_LENGTH := CODES_LENGTH + 1;
    end loop;

    assert CODES_LENGTH = PIN_CODES'LENGTH
      report "Error - Logic Map doesn't have correct number of Pin Codes."
      severity ERROR;
 
    -- Tag Test Pins and set Logic Value index. 
    for I in TEST_PIN_RANGE loop
      if PINS(TEST_PINS'VAL(I - 1)) then
	FSA(I).KIND := WAVES_SYSTEM.T_PIN;
	FSA(I).NEXT_INDEX := LOGIC_INDEX;
      end if;
    end loop;

    -- Add Logic Map Frames to Frame Set Array
    for I in FRAMES'RANGE loop
      FSA(I + OFF_SET) := FRAMES(I);
      if FRAMES(I).NEXT_INDEX > 0 then
        FSA(I + OFF_SET).NEXT_INDEX := FRAMES(I).NEXT_INDEX + OFF_SET;
      end if;
    end loop; 
    return FSA;
  end NEW_FRAME_SET_ARRAY;


  -- -=-=-=-=-=-=-= MERGE_FRAME_SET_ARRAY "+" FUNCTION =-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- This functions merges two Frame Set Arrays and returns a new 
  -- Frame Set Array.  New_fsa(i) = A_fsa(i) if /= empty else, B_fsa(i). 
  -- 
  function "+" (
      A                  : FRAME_SET_ARRAY;
      B                  : FRAME_SET_ARRAY )
    return FRAME_SET_ARRAY
  is
    constant OFF_SET     : INTEGER := A'LENGTH - TEST_PIN_RANGE'RIGHT;
    variable X           : FRAME_SET_ARRAY 
	(1 to (A'LENGTH + (B'LENGTH - TEST_PIN_RANGE'RIGHT)));
    variable TEMP        : FRAME_SET_ARRAY(1 to B'LENGTH);
  begin
    -- Add off set to fsa B
    for I in 1 to B'LENGTH loop
      TEMP(I) := B(I);
      if B(I).NEXT_INDEX > 0 then
	TEMP(I).NEXT_INDEX := B(I).NEXT_INDEX + OFF_SET;
      end if;
    end loop;

    -- Put fsa A into new array
    for I in 1 to A'LENGTH loop
      X(I) := A(I);
    end loop;

    -- Put fsa B (TEMP) frame sets into new array
    for I in A'LENGTH + 1 to X'LENGTH loop
      X(I) := TEMP(I - A'LENGTH + TEST_PIN_RANGE'RIGHT);
    end loop;

    -- Merge fsa Test Pin values
    for PIN in TEST_PIN_RANGE loop
      if X(PIN).NEXT_INDEX = -1 then
	X(PIN) := TEMP(PIN);
      else  -- Merge fsa B if fsa A Logic Value is empty. 
	for L in 0 to LOGIC_VALUE'POS(LOGIC_VALUE'RIGHT) loop
	  if X(X(PIN).NEXT_INDEX + L).NEXT_INDEX = -1 and
			    B(PIN).NEXT_INDEX /= -1 then
	    X(X(PIN).NEXT_INDEX + L) := TEMP(B(PIN).NEXT_INDEX + L);
	  -- Merge fsa B Pin Codes if fsa A Pin Codes is empty. 
	  elsif B(PIN).NEXT_INDEX /= -1 then  
	    for J in 0 to PIN_CODES'LENGTH - 1 loop
	      if X(X(X(PIN).NEXT_INDEX + L).NEXT_INDEX + J).NEXT_INDEX = -1 and
		   B(B(PIN).NEXT_INDEX + L).NEXT_INDEX /= -1 then	
		X(X(X(PIN).NEXT_INDEX + L).NEXT_INDEX + J) := 
		  TEMP(B(B(PIN).NEXT_INDEX + L).NEXT_INDEX + J); 
	      end if; 
	    end loop;
	  end if;
	end loop;
      end if;
    end loop;
    return X;
  end "+";

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- Internal procedure used to check consitancy of the Frame
  -- Set Array.  Asserts WARNINGS for undefined Logic Values, 
  -- Pin Codes, Frames, and ERROR for relative timing errors.
  -- Set MESSAGE to TRUE to disable assertion Warnings.
  procedure TIME_DATA_CHECK(
    variable DATA : TIME_DATA;
    constant PIN  : NATURAL;
    variable SET  : inout PINSET)
  is
    constant MESSAGE     : BOOLEAN := FALSE;
    variable L_INDEX     : INTEGER;
    variable C_INDEX     : INTEGER := -1;
    variable F_INDEX     : INTEGER;
    variable NEXT_PIN    : NATURAL;
    constant EMPTY_EVENT : INTEGER := -1;
  begin

  SET(TEST_PINS'VAL(PIN - 1)) := TRUE;
  -- Get Test Pin Logic Value index.
  if DATA.all(PIN).NEXT_INDEX /= EMPTY_EVENT then
    assert DATA.all(PIN).KIND = WAVES_SYSTEM.T_PIN
      report "WARNING - Expected a Tagged Pin."
      severity WARNING;
 
    L_INDEX := DATA.all(PIN).NEXT_INDEX; 
    for I in 0 to LOGIC_VALUE'POS(LOGIC_VALUE'RIGHT) loop
      -- Get Logic Value Pin Code index.
      if DATA.all(L_INDEX + I).NEXT_INDEX /= EMPTY_EVENT then
      assert DATA.all(L_INDEX + I).KIND = WAVES_SYSTEM.T_LOGIC
        report "WARNING - Expected a Tagged Logic."
        severity WARNING;
 
	-- If next logic index is same code index then skip check.
	if C_INDEX = DATA.all(L_INDEX + I).NEXT_INDEX then
	  next;
	end if;
	C_INDEX := DATA.all(L_INDEX + I).NEXT_INDEX;
	for J in 0 to PIN_CODES'RIGHT - 1 loop
	  -- Get Pin Code Frame index.
	  F_INDEX := DATA.all(C_INDEX + J).NEXT_INDEX;
          assert DATA.all(C_INDEX + J).KIND = WAVES_SYSTEM.T_CODE
            report "WARNING - Expected a Tagged Code."
            severity WARNING;
 
	  if F_INDEX /= EMPTY_EVENT then
	    NEXT_PIN := DATA.all(F_INDEX).TAGGED.TIME.BASE_PIN;
            assert DATA.all(F_INDEX).KIND = WAVES_SYSTEM.T_FRAME
              report "WARNING - Expected a Tagged Frame."
              severity WARNING;
 
	    -- If event is not empty and is relative then set base pin flag.
	    if DATA.all(F_INDEX).NEXT_INDEX /= EMPTY_EVENT and 
	       NEXT_PIN /= 0 then
	      -- See if base pin has already been examined.
	      if SET(TEST_PINS'VAL(NEXT_PIN - 1)) then
	        assert FALSE 
		  report "Relative loop back timing error." 
		  severity ERROR;
	      else	-- Recursive Time Data Check
	        TIME_DATA_CHECK(DATA, NEXT_PIN, SET); 
		SET(TEST_PINS'VAL(NEXT_PIN - 1)) := FALSE;
	      end if;
	    end if;
	  else
	    assert MESSAGE
	      report "Undefined Pin Code index in Frame Set Array."
	      severity WARNING; 
	  end if;
	end loop;
      else
	assert MESSAGE
	  report "Undefined Logic Value index in Frame Set Array."
	  severity WARNING; 
      end if;
    end loop;
  else
    assert MESSAGE
      report "Undefined Test Pin index in Frame Set Array."
      severity WARNING; 
  end if;

  end TIME_DATA_CHECK;

  -- -=-=-=-=-=-=-=-= NEW_TIME_DATA FUNCTION -=-=-=-=-=-=-=-=-
  --
  -- This function returns a pointer (TIME_DATA) to a Frame Set Array.
  --
  function NEW_TIME_DATA (
      FRAMES             : FRAME_SET_ARRAY )
    return TIME_DATA
  is
    variable DATA : TIME_DATA := new FRAME_SET_ARRAY'(FRAMES);
    variable CHECK_LIST : PINSET;
  begin
    -- Check Frame Set Array (TIME DATA) definition. 
    for PIN in TEST_PIN_RANGE loop
      CHECK_LIST := (others=>FALSE);
      TIME_DATA_CHECK(DATA, PIN, CHECK_LIST);
    end loop;
    return DATA; 
  end NEW_TIME_DATA;

--
  -- -=-=-=-=-=-=-=-=-=-= NEW_FILE_SLICE FUNCTION =-=-=-=-=-=-=-=-=-=-=-=- 
  --
  -- The following functions initialize a FILE_SLICE with a STRING of
  -- proper length.  The first returns length of TEST_PINS and
  -- the second return length of parameter LENGTH. 
  --
  function NEW_FILE_SLICE return FILE_SLICE
  is
    variable X : LINE := new STRING(TEST_PIN_RANGE);
  begin
    return(X, 0 hr, 0, FALSE);
  end NEW_FILE_SLICE;

  function NEW_FILE_SLICE (
    constant LENGTH : in POSITIVE)
  return FILE_SLICE
  is
    variable X : LINE := new STRING(1 to LENGTH);
  begin
    return(X, 0 hr, 0, FALSE);
  end NEW_FILE_SLICE;
  
--
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- 
    -- 
    -- Internal procedure for reading a slice from the external file.
    -- Procedure READ_SLICE scans the WAVES external file for key characters
    -- and does the necessary operations to create a proper file slice.
    -- The meaning of the key characters are: 
    -- 		'=' - indicates an index value is next.
    --		':' - indicates timing data is next.
    --		';' - indicates the end of a slice.
    --		'%' - indicates a comment to end of line.
    --		'"' - begins a quoted string until space is found. 
    -- Note: all key characters should begin either in the first field 
    --       or be proceeded by a space.  Otherwise, these characters
    --       maybe interpretated as a quoted character code.
    --
    procedure READ_SLICE (
	variable EX_FILE     : in TEXT;
	variable FILE_DATA   : inout FILE_SLICE;
	constant ACTIVE_PINS : TEST_PINS_LIST) 
    is
    variable CHAR      : CHARACTER;
    variable GOOD      : BOOLEAN;
    variable ILINE     : LINE; 
    variable INDEX     : NATURAL;
    variable PIN_INDEX : NATURAL := 1;


    -- Return TRUE if the input character is white space.
    --
    function IS_WHITE ( 
        C                : CHARACTER ) 
      return BOOLEAN 
    is
    begin
      return (C = ' ') or (C = HT);
    end IS_WHITE;


    -- Ensure that a non-white character remains on the current line, 
    -- reading new lines as needed.  Sets ENDFILE status. 
    -- 
    procedure SKIP_WHITE (
	IS_EOF : inout BOOLEAN) 
    is
      variable TRASH     : CHARACTER;
    begin
    IS_EOF := FALSE;
    loop
      while ILINE'LENGTH > 0 loop
	if IS_WHITE(ILINE(ILINE.all'LEFT)) then
	  READ(ILINE, TRASH);
	else
	  return; 
	end if;
      end loop;
      IS_EOF := ENDFILE(EX_FILE);
      exit when IS_EOF; 
      READLINE(EX_FILE, ILINE); 
    end loop;
    end SKIP_WHITE;


    -- Read the next non-white character from the input line, reading new 
    -- lines as necessary.
    -- 
    procedure READ_CHAR 
    is
    begin
      SKIP_WHITE(FILE_DATA.END_OF_FILE);
      if not FILE_DATA.END_OF_FILE then
        READ(ILINE, CHAR);
      end if;
    end READ_CHAR;      


  -- Set the ACTIVE_PINS index which corresponds to TEST_PINS.
  procedure SET_INDEX
  is
  begin
    if PIN_INDEX <= ACTIVE_PINS'LENGTH then
      INDEX := TEST_PINS'POS(ACTIVE_PINS(PIN_INDEX)) + 1;
    else
      INDEX := PIN_INDEX;
    end if;
    assert INDEX >= FILE_DATA.CODES.all'LEFT and 
           INDEX <= FILE_DATA.CODES.all'RIGHT
      report "Error - index is out of Code Pin Range."
      severity ERROR;
    PIN_INDEX := PIN_INDEX + 1;
  end SET_INDEX;


    -- Read an index from the file into PIN_INDEX variable.
    --
    procedure UPDATE_INDEX 
    is
    begin
      SKIP_WHITE(FILE_DATA.END_OF_FILE);
      READ(ILINE, PIN_INDEX, GOOD);
      assert GOOD 
          report "Error reading index from external file."
          severity ERROR;
    end UPDATE_INDEX;


    -- Read either a time or an integer for timing data.  
    --
    procedure GET_TIMING
    is
      variable PLOCAL : TIME;
    begin
      SKIP_WHITE(FILE_DATA.END_OF_FILE);
      READ(ILINE, PLOCAL, GOOD);
      if GOOD then
	FILE_DATA.FS_TIME := PLOCAL;
	FILE_DATA.FS_INTEGER := 0;
      else
        READ(ILINE, FILE_DATA.FS_INTEGER, GOOD);
	FILE_DATA.FS_TIME := 0 hr;
        assert GOOD
            report "Error in timing format in external file."
            severity ERROR;
      end if;
    end GET_TIMING;


    -- Read a quoted string from the external file.  Continue reading
    -- until white space is seen.  If end of line occurs before
    -- white space, next line is read and string continues.  If current
    -- CHARacter is a '"' then QUOTE_CHAR should be set true. 
    --
    procedure GET_QUOTED_STRING(
      constant A_QUOTE : in BOOLEAN)
    is
      variable QUOTE_CHAR : BOOLEAN := A_QUOTE;
    begin
      loop
        if ILINE'LENGTH = 0 then
	  READLINE(EX_FILE, ILINE);
          if FILE_DATA.END_OF_FILE then 
	    assert FALSE
	      report "Error - Unexpected EOF in quoted string."
	      severity ERROR;
	  end if;
	end if;
	if QUOTE_CHAR then
          READ(ILINE, CHAR);	-- Quoted character
	else
	  QUOTE_CHAR := TRUE;
	end if;
        if IS_WHITE(CHAR) then return; end if;
	SET_INDEX;
        FILE_DATA.CODES(INDEX) := CHAR;
      end loop;  
    end GET_QUOTED_STRING;


    begin 
      FILE_DATA.END_OF_FILE := ENDFILE(EX_FILE);
      if FILE_DATA.END_OF_FILE then return; end if;
      READLINE(EX_FILE, ILINE); 
    loop		-- Begin reading until end of slice. 
      READ_CHAR;
      case CHAR is
        when '=' => 
          UPDATE_INDEX;
        when ':' =>
          GET_TIMING;
        when ';' =>
          return;
	when '%' =>			-- Ignore rest of line
          FILE_DATA.END_OF_FILE := ENDFILE(EX_FILE);
          if FILE_DATA.END_OF_FILE then
	    return;
	  else
	    READLINE(EX_FILE, ILINE);	-- Get next line
	  end if;
	when '"' =>
	  GET_QUOTED_STRING(TRUE);
        when others =>
          if FILE_DATA.END_OF_FILE then return;
	  else
	    GET_QUOTED_STRING(FALSE);
	  end if;
        end case;
      end loop;
    end READ_SLICE;

  -- -=-=-=-=-=-=-=-=-= READ_FILE_SLICE PROCEDURES =-=-=-=-=-=-=-=-=-=-=-=- 
  -- 
  -- The following definitions support the fixed file format of Level 1
  -- WAVES.  The FILE DATA record stores information most recently 
  -- read by the procedure.  The optional ACTIVE PINS indicates the
  -- external file supplies data for only those pins, the default is 
  -- all TEST PINS.  The order of the ACTIVE PINS must correspond to
  -- the order of signals in the external file.  Note the order of the
  -- ACTIVE PINS does not have to match the order of the TEST PINS. 
  -- 

  procedure READ_FILE_SLICE (
      variable EX_FILE      : in TEXT;
      variable FILE_DATA    : inout FILE_SLICE )
  is
    variable ACTIVE_PINS : TEST_PINS_LIST (TEST_PIN_RANGE);
  begin
    -- Create a TEST_PINS_LIST of all TEST_PINS
    for I in TEST_PIN_RANGE loop
      ACTIVE_PINS(I) := TEST_PINS'VAL(I - 1);
    end loop;
    READ_SLICE(EX_FILE, FILE_DATA, ACTIVE_PINS);
  end READ_FILE_SLICE;


  -- This procedure accepts a TEST_PINS_LIST which corresponds to
  -- the signals in the external file. 
  --
  procedure READ_FILE_SLICE (
      variable EX_FILE      : in TEXT;
      variable FILE_DATA    : inout FILE_SLICE; 
      constant ACTIVE_PINS  : in TEST_PINS_LIST) 
  is
  begin
    READ_SLICE(EX_FILE, FILE_DATA, ACTIVE_PINS);
  end READ_FILE_SLICE;

  -- This procedure takes a single TEST_PINS vector.
  --
  procedure READ_FILE_SLICE (
      variable EX_FILE      : in TEXT;
      variable FILE_DATA    : inout FILE_SLICE; 
      constant ACTIVE_PIN   : in TEST_PINS) 
  is
    variable ACTIVE_PINS : TEST_PINS_LIST(1 to 1) := (1 => ACTIVE_PIN); 
  begin
    READ_SLICE(EX_FILE, FILE_DATA, ACTIVE_PINS);
  end READ_FILE_SLICE;

--
-- END OF WAVES_OBJECTS PACKAGE BODY
--
end WAVES_OBJECTS;
