--
-- Copyright 1990 
--
-- WAVES Standard Packages Version 3.3.2
-- 25 October 1990
--
-- This code is distributed for the purposes of evaluating the 
-- Waveform And Vector Exchange Specification (WAVES) proposal 
-- presented to the IEEE by the WAVES Analysis and Standardization 
-- Group.  This code may not be used for commercial purposes and 
-- may not be redistributed or published without permission of the 
-- Chairman of the WAVES Analysis and Standardization Group, 
-- Mr Robert Hillman.
--
-- Address comments or questions to:
--    Robert Hillman            
--    RADC/RBRP                 
--    Griffis AFB, NY           
--    (315) 330-2241            
--                              
--    hillman@tisss.radc.af.mil 
--
library WAVES_STANDARD;
use WAVES_STANDARD.WAVES_SYSTEM;
-- A context clause providing visibility to LOGIC_VALUE is
-- required at this point.
use WAVES_STANDARD.waves_events.all ;
-- previous line added by GLN on 02-09-91

package WAVES_INTERFACE is 
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following are type definitions for creating Logic Sets and
  -- Logic Lists.  A LOGIC SET is a array of booleans whose length is
  -- LOGIC_VALUE'LENGTH and in which a TRUE value indicates the
  -- corresponding Logic Value is a member. 
  --
  type LOGIC_SET is array (LOGIC_VALUE) of BOOLEAN;
  
  type LOGIC_LIST is array (POSITIVE range <>) of LOGIC_VALUE;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- Functions to create a LOGIC SET given a LOGIC VALUE, LOGIC LIST
  -- or no parameter in which case all Logic Values are returned as
  -- TRUE. 
  --
  function NEW_LOGIC_SET return LOGIC_SET;
  
  function NEW_LOGIC_SET (
      constant MEMBER      : in LOGIC_VALUE )
    return LOGIC_SET;
  
  function NEW_LOGIC_SET (
      constant MEMBER      : in LOGIC_LIST )
    return LOGIC_SET;
  
-- 
   -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- An EVENT_TIME_BASIS is a private type.
  -- 
  subtype EVENT_TIME_BASIS is WAVES_SYSTEM.SYSTEM_EVENT_TIME_BASIS;

  type EVENT_TIME is record
    NOMINAL        : TIME;
    ET_BASIS       : EVENT_TIME_BASIS;
  end record;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following functions are used to declare an EVENT_TIME. For
  -- each of these functions the EVENT TIME BASIS defaults to a 
  -- reference at the beginning of the slice.
  -- 
  -- The first function returns an event time with no tolerances.  
  -- 
  -- The second function returns an event time with symmetric bounds. 
  -- 
  -- The third function allows the bounds to be specified
  -- independently. 
  -- 
  -- The MERGE ETIME function returns the lefthand argument with 
  -- tolerances taken from the lefthand argument if specified, 
  -- otherwise taken from the righthand argument.
  -- 
  function ETIME (
      NOMINAL            : TIME;
      BASE_EVENT         : NATURAL := 0 )
    return EVENT_TIME;
  
  function ETIME (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME;
      BASE_EVENT          : NATURAL := 0 )
    return EVENT_TIME;
  
  function ETIME (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME;
      BASE_EVENT         : NATURAL := 0 )
    return EVENT_TIME;
  
  function MERGE_ETIME (
      A                  : EVENT_TIME;
      B                  : EVENT_TIME )
    return EVENT_TIME;


  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- Arithmetic functions to manipulate event times.  The functions
  -- that operate on two event times require that the ET_BASIS 
  -- fields be the same.
  -- 
  function "+"(A : TIME;        B : EVENT_TIME ) return EVENT_TIME;
  function "+"(A : EVENT_TIME;  B : TIME       ) return EVENT_TIME;

  function "-"(A : EVENT_TIME;  B : TIME       ) return EVENT_TIME;

  function "*"(A : REAL;        B : EVENT_TIME ) return EVENT_TIME;
  function "*"(A : EVENT_TIME;  B : REAL       ) return EVENT_TIME;

  function "/"(A : EVENT_TIME;  B : TIME       ) return REAL;

  function "+"(A : EVENT_TIME;  B : EVENT_TIME ) return EVENT_TIME;

  function "-"(A : EVENT_TIME;  B : EVENT_TIME ) return EVENT_TIME;


-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- An EVENT is a record containing a LOGIC VALUE and an EVENT TIME. 
  -- The LOGIC VALUE may be converted into an EVENT VALUE through 
  -- the required VALUE DICTIONARY function. 
  -- 
  type EVENT is record
     VALUE               : LOGIC_VALUE;
     TIME                : EVENT_TIME;
  end record;
  
  type EVENT_LIST is array (POSITIVE range <>) of EVENT;


  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- A FRAME is a list of events.
  -- 
  -- A FRAME_SET is a list of FRAMEs.  When inserted into a FRAME SET
  -- ARRAY or into a LOGIC MAP, a FRAME SET must have one FRAME for 
  -- every PIN CODE. 
  -- 
  -- A LOGIC MAP is a list of FRAME SETS, one for every LOGIC VALUE.
  -- 
  -- A FRAME SET ARRAY is a list of FRAME_SETs.
  -- 
  subtype FRAME is WAVES_SYSTEM.SYSTEM_FRAME;
  
  subtype FRAME_SET is WAVES_SYSTEM.SYSTEM_FRAME_SET;
  
  subtype LOGIC_MAP is WAVES_SYSTEM.SYSTEM_LOGIC_MAP;

  subtype FRAME_SET_ARRAY is WAVES_SYSTEM.SYSTEM_FRAME_SET_ARRAY;

-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following functions convert a single event or a single event
  -- list into a frame (a list of tagged events).  The empty case
  -- returns a frame of length one with the tagged event marked
  -- unspecified. 
  -- 
  function FRAME_EVENT 
    return FRAME;
  
  function FRAME_EVENT (
      EVENTS             : EVENT )
    return FRAME;
  
  function FRAME_ELIST 
    return FRAME;
  
  function FRAME_ELIST (
      EVENTS             : EVENT_LIST )
    return FRAME;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The set of operations are array concatenation, treating a frame
  -- set as an array of frames and a frame set array as an array of
  -- frame sets. 
  -- 
  function "+"(A : FRAME;      B : FRAME) return FRAME_SET;
  function "+"(A : FRAME_SET;  B : FRAME) return FRAME_SET;
  function "+"(A : FRAME;      B : FRAME_SET) return FRAME_SET;
  function "+"(A : FRAME_SET;  B : FRAME_SET) return FRAME_SET;
  

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- Create a LOGIC MAP from a FRAME SET.
  -- 
  function NEW_LOGIC_MAP (
      FRAMES       : FRAME_SET;
      LOGIC        : LOGIC_VALUE ) return LOGIC_MAP;

  function NEW_LOGIC_MAP (
      FRAMES       : FRAME_SET;
      LOGIC        : LOGIC_SET ) return LOGIC_MAP;


  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- Merge two LOGIC MAP's together, building a result that uses 
  -- FRAME SETS from the first operand, if present, otherwise from 
  -- the second operand.
  -- 
  function "+"(A, B : LOGIC_MAP ) return LOGIC_MAP;

  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- Delay specification for APPLY.
  -- 
  subtype DELAY_TIME_BASIS is WAVES_SYSTEM.SYSTEM_DELAY_TIME_BASIS;

  type DELAY_TIME is record
    NOMINAL     : TIME;
    DT_BASIS    : DELAY_TIME_BASIS;
  end record;

  function DELAY (
      NOMINAL            : TIME )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME )
    return DELAY_TIME;
  
  function DELAY (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME )
    return DELAY_TIME;
  

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- Arithmetic functions to manipulate delay times.  The functions
  -- that operate on two delay times require that the DT_BASIS 
  -- fields be the same.
  -- 
  function "+"(A : TIME;        B : DELAY_TIME ) return DELAY_TIME;
  function "+"(A : DELAY_TIME;  B : TIME       ) return DELAY_TIME;

  function "-"(A : DELAY_TIME;  B : TIME       ) return DELAY_TIME;

  function "*"(A : REAL;        B : DELAY_TIME ) return DELAY_TIME;
  function "*"(A : DELAY_TIME;  B : REAL       ) return DELAY_TIME;

  function "/"(A : DELAY_TIME;  B : TIME       ) return REAL;

  function "+"(A : DELAY_TIME;  B : DELAY_TIME ) return DELAY_TIME;

  function "-"(A : DELAY_TIME;  B : DELAY_TIME ) return DELAY_TIME;


  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- Useful types that should not be defered until WAVES OBJECTS
  -- 
  type INTEGER_LIST is array (POSITIVE range <>) of INTEGER;

  type TIME_LIST is array (POSITIVE range <>) of TIME;

  type EVENT_TIME_LIST is array (POSITIVE range <>) of EVENT_TIME;

  type DELAY_TIME_LIST is array (POSITIVE range <>) of DELAY_TIME;

end WAVES_INTERFACE;

--
-- Copyright 1990 
--
-- WAVES Standard Packages Version 3.3.2
-- 25 October 1990
--
-- This code is distributed for the purposes of evaluating the 
-- Waveform And Vector Exchange Specification (WAVES) proposal 
-- presented to the IEEE by the WAVES Analysis and Standardization 
-- Group.  This code may not be used for commercial purposes and 
-- may not be redistributed or published without permission of the 
-- Chairman of the WAVES Analysis and Standardization Group, 
-- Mr Robert Hillman.
--
-- Address comments or questions to:
--    Robert Hillman           
--    RADC/RBRP                
--    Griffis AFB, NY          
--    (315) 330-2241           
--                             
--    hillman@tisss.radc.af.mil
--
library WAVES_STANDARD;
use WAVES_STANDARD.WAVES_SYSTEM.all;
package body WAVES_INTERFACE is 

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- Return sets of LOGIC VALUE's.
  -- 
  function NEW_LOGIC_SET 
    return LOGIC_SET
  is
    variable LS    : LOGIC_SET := (others => TRUE);
  begin
    return LS;
  end NEW_LOGIC_SET;

  function NEW_LOGIC_SET (
      constant MEMBER      : in LOGIC_VALUE )
    return LOGIC_SET
  is
    variable LS    : LOGIC_SET := (others => FALSE);
  begin
    LS(MEMBER) := TRUE;
    return LS;
  end NEW_LOGIC_SET;

  function NEW_LOGIC_SET  (
      constant MEMBER      : in LOGIC_LIST )
    return LOGIC_SET
  is
    variable LS    : LOGIC_SET := (others => FALSE);
  begin
    for I in MEMBER'RANGE loop
      LS(MEMBER(I)) := TRUE;
    end loop;
    return LS;
  end NEW_LOGIC_SET;

-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following three functions declare an EVENT_TIME.
  -- 
  -- The first function returns an event time with no tolerances.  The 
  -- nominal time defaults to 0 hr.
  -- 
  -- The second function returns an event time with symmetric bounds.
  -- 
  -- The third function allows the bounds to be specified independently.
  -- 
  function ETIME (
      NOMINAL            : TIME; 
      BASE_EVENT	 : NATURAL := 0 )
    return EVENT_TIME
  is
  begin
    return ( NOMINAL, (FALSE, 0 hr, 0 hr, BASE_EVENT, 0 ));
  end ETIME;


  function ETIME (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME;
      BASE_EVENT	  : NATURAL := 0 )
    return EVENT_TIME
  is
  begin
    return ( NOMINAL, (TRUE, EARLIEST_AND_LATEST, 
		            EARLIEST_AND_LATEST, BASE_EVENT, 0));
  end ETIME;


  function ETIME (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME;
      BASE_EVENT	 : NATURAL := 0 )
    return EVENT_TIME
  is
  begin
    return ( NOMINAL, (TRUE, EARLIEST, LATEST, BASE_EVENT, 0));
  end ETIME;


  function MERGE_ETIME (
      A                  : EVENT_TIME;
      B                  : EVENT_TIME )
    return EVENT_TIME
  is
    variable X           : EVENT_TIME := A;
  begin
    if not X.ET_BASIS.TOLERANCES_SPECIFIED then
      X.ET_BASIS.TOLERANCES_SPECIFIED := B.ET_BASIS.TOLERANCES_SPECIFIED;
      X.ET_BASIS.EARLIEST := B.ET_BASIS.EARLIEST;
      X.ET_BASIS.LATEST   := B.ET_BASIS.LATEST;
    end if;
    if X.ET_BASIS.BASE_EVENT = 0 then
      X.ET_BASIS.BASE_EVENT := B.ET_BASIS.BASE_EVENT;
    end if;
    if X.ET_BASIS.BASE_PIN = 0 then
      X.ET_BASIS.BASE_PIN := B.ET_BASIS.BASE_PIN;
    end if;
    return X;
  end MERGE_ETIME;
--
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- Arithmetic functions to manipulate event times.  The functions
  -- that operate on two event times require that the ET_BASIS 
  -- fields be the same.
  -- 
  function "+"(A : TIME;        B : EVENT_TIME ) return EVENT_TIME
  is
    variable X : EVENT_TIME := B;
  begin
    X.NOMINAL := X.NOMINAL + A;
    return X;
  end "+";

  function "+"(A : EVENT_TIME;  B : TIME       ) return EVENT_TIME
  is
    variable X : EVENT_TIME := A;
  begin
    X.NOMINAL := X.NOMINAL + B;
    return X;
  end "+";

  function "-"(A : EVENT_TIME;  B : TIME       ) return EVENT_TIME
  is
    variable X : EVENT_TIME := A;
  begin
    X.NOMINAL := X.NOMINAL - B;
    return X;
  end "-";

  function "*"(A : REAL;        B : EVENT_TIME ) return EVENT_TIME
  is
    variable X : EVENT_TIME := B;
  begin
    X.NOMINAL := X.NOMINAL * A;
    return X;
  end "*";

  function "*"(A : EVENT_TIME;  B : REAL       ) return EVENT_TIME
  is
    variable X : EVENT_TIME := A;
  begin
    X.NOMINAL := X.NOMINAL * B;
    return X;
  end "*";

  function "/"(A : EVENT_TIME;  B : TIME       ) return REAL
  is
    variable X : EVENT_TIME := A;
  begin
    return (REAL(X.NOMINAL / B) * 1.0);
  end "/";

  function "+"(A : EVENT_TIME;  B : EVENT_TIME ) return EVENT_TIME
  is
    variable X : EVENT_TIME := A;
  begin
    X.NOMINAL := X.NOMINAL + B.NOMINAL;
    return X;
  end "+";

  function "-"(A : EVENT_TIME;  B : EVENT_TIME ) return EVENT_TIME
  is
    variable X : EVENT_TIME := A;
  begin
    X.NOMINAL := X.NOMINAL - B.NOMINAL;
    return X;
  end "-";


  -- -=-=-=-=-=-=-=-=-=-=-=-=-=- PACKAGE PRIVATE -=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- This constant is used to determine whether a tagged event contains a
  -- real event or is just an empty placeholder.
  -- 

  constant END_OF_FRAME    : INTEGER := 0;
  constant EMPTY_FRAME     : INTEGER := -1;

  -- An internal function for checking equality between
  -- Tagged Events.
  --
  function "="(A, B : WAVES_SYSTEM.SYSTEM_TAGGED_EVENT) return BOOLEAN
  is
  begin
    if A.KIND  = B.KIND and
      A.TAGGED.VALUE = B.TAGGED.VALUE and
      A.TAGGED.TIME.NOMINAL = B.TAGGED.TIME.NOMINAL and 
      A.TAGGED.TIME.TOLERANCES_SPECIFIED = 
      B.TAGGED.TIME.TOLERANCES_SPECIFIED then
      if A.TAGGED.TIME.TOLERANCES_SPECIFIED then
	return 
        A.TAGGED.TIME.EARLIEST   = B.TAGGED.TIME.EARLIEST and 
        A.TAGGED.TIME.LATEST     = B.TAGGED.TIME.LATEST and 
        A.TAGGED.TIME.BASE_EVENT = B.TAGGED.TIME.BASE_EVENT and 
        A.TAGGED.TIME.BASE_PIN   = B.TAGGED.TIME.BASE_PIN ;
      end if;
      return TRUE;
    else
      return FALSE;
    end if;
  end "=";

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=- FUNCTIONS -=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following set of operations are array concatenation, treating a 
  -- frame set as an array of frames and a frame set array as an array of 
  -- frame sets.
  -- 
  function "+"(A : FRAME; B : FRAME) return FRAME_SET
  is
    variable TEMP_1      : FRAME_SET (1 to A'LENGTH + B'LENGTH);
  begin
    for I in 1 to A'LENGTH loop
      TEMP_1(I) := A(I);
    end loop;
    for I in 1 to B'LENGTH loop
      TEMP_1(I + A'LENGTH) := B(I);
      if B(I).NEXT_INDEX > 0 then
	TEMP_1(I + A'LENGTH).NEXT_INDEX := B(I).NEXT_INDEX + A'LENGTH;
      end if;
    end loop;
    return TEMP_1;
  end "+";

  function "+"(A : FRAME_SET; B : FRAME) return FRAME_SET
  is
    variable TEMP_1      : FRAME_SET (1 to A'LENGTH + B'LENGTH);
  begin
    for I in 1 to A'LENGTH loop
      TEMP_1(I) := A(I);
    end loop;
    for I in 1 to B'LENGTH loop
      TEMP_1(I + A'LENGTH) := B(I);
      if B(I).NEXT_INDEX > 0 then
	TEMP_1(I + A'LENGTH).NEXT_INDEX := B(I).NEXT_INDEX + A'LENGTH;
      end if;
    end loop;
    return TEMP_1;
  end "+";

  function "+"(A : FRAME; B : FRAME_SET) return FRAME_SET
  is
    variable TEMP_1      : FRAME_SET (1 to A'LENGTH + B'LENGTH);
  begin
    for I in 1 to A'LENGTH loop
      TEMP_1(I) := A(I);
    end loop;
    for I in 1 to B'LENGTH loop
      TEMP_1(I + A'LENGTH) := B(I);
      if B(I).NEXT_INDEX > 0 then
	TEMP_1(I + A'LENGTH).NEXT_INDEX := B(I).NEXT_INDEX + A'LENGTH;
      end if;
    end loop;
    return TEMP_1;
  end "+";

  function "+"(A : FRAME_SET; B : FRAME_SET) return FRAME_SET
  is
    variable TEMP_1      : FRAME_SET (1 to A'LENGTH + B'LENGTH);
  begin
    for I in 1 to A'LENGTH loop
      TEMP_1(I) := A(I);
    end loop;
    for I in 1 to B'LENGTH loop
      TEMP_1(I + A'LENGTH) := B(I);
      if B(I).NEXT_INDEX > 0 then
	TEMP_1(I + A'LENGTH).NEXT_INDEX := B(I).NEXT_INDEX + A'LENGTH;
      end if;
    end loop;
    return TEMP_1;
  end "+";
--
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  function FRAME_SIZE ( F : FRAME_SET) return NATURAL
  is
    variable SIZE : NATURAL := 0;
  begin
    for I in F'RANGE loop
      if F(I).NEXT_INDEX <= 0 then
	SIZE := SIZE + 1;
      end if;
    end loop;
    return SIZE;
  end FRAME_SIZE;

  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- Create a LOGIC MAP from a FRAME SET.
  -- 
  function NEW_LOGIC_MAP (
      FRAMES       : FRAME_SET;
      LOGIC        : LOGIC_VALUE ) return LOGIC_MAP
  is
    constant CODES_LENGTH : NATURAL := FRAME_SIZE(FRAMES);
    constant LOGIC_LENGTH : NATURAL := LOGIC_VALUE'POS(LOGIC_VALUE'RIGHT) + 1;
    variable CODES_INDEX  : NATURAL := LOGIC_LENGTH + 1;
    variable FRAME_INDEX  : NATURAL := CODES_INDEX + CODES_LENGTH; 
    variable OFF_SET      : NATURAL := 0;
    variable INIT_TIME    : WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    variable L_MAP : LOGIC_MAP (1 to LOGIC_LENGTH + CODES_LENGTH +
	FRAMES'LENGTH) := (others => (WAVES_SYSTEM.UNUSED, -1, (0, INIT_TIME)));
  begin
    -- Set Pin Codes index of Logic Value.
    L_MAP(LOGIC_VALUE'POS(LOGIC) + 1).KIND := WAVES_SYSTEM.T_LOGIC;
    L_MAP(LOGIC_VALUE'POS(LOGIC) + 1).NEXT_INDEX := CODES_INDEX; 

    -- Insert Frame Set and set Frame index. 
    for I in 0 to FRAMES'LENGTH - 1 loop
      L_MAP(FRAME_INDEX + OFF_SET) := FRAMES(I + 1); 
      if FRAMES(I + 1).NEXT_INDEX <= 0 then
	L_MAP(CODES_INDEX).KIND := WAVES_SYSTEM.T_CODE;
	L_MAP(CODES_INDEX).NEXT_INDEX := FRAME_INDEX;
	FRAME_INDEX := FRAME_INDEX + OFF_SET + 1;
	OFF_SET := 0;
	CODES_INDEX := CODES_INDEX + 1;
      else
        L_MAP(FRAME_INDEX + OFF_SET).NEXT_INDEX := 
	  FRAMES(I + 1).NEXT_INDEX + LOGIC_LENGTH + CODES_LENGTH;
        OFF_SET := OFF_SET + 1;
      end if;
    end loop;
    return L_MAP;
  end NEW_LOGIC_MAP;

  function NEW_LOGIC_MAP (
      FRAMES       : FRAME_SET;
      LOGIC        : LOGIC_SET ) return LOGIC_MAP
  is
    constant CODES_LENGTH : NATURAL := FRAME_SIZE(FRAMES);
    constant LOGIC_LENGTH : NATURAL := LOGIC_VALUE'POS(LOGIC_VALUE'RIGHT) + 1;
    variable CODES_INDEX  : NATURAL := LOGIC_LENGTH + 1;
    variable FRAME_INDEX  : NATURAL := CODES_INDEX + CODES_LENGTH; 
    variable OFF_SET      : NATURAL := 0;
    variable INIT_TIME    : WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    variable L_MAP : LOGIC_MAP (1 to LOGIC_LENGTH + CODES_LENGTH +
	FRAMES'LENGTH) := (others => (WAVES_SYSTEM.UNUSED, -1, (0, INIT_TIME)));
  begin
    -- Set Pin Codes index of Logic Value.
    for I in LOGIC'RANGE loop
      if LOGIC(I) then
        L_MAP(LOGIC_VALUE'POS(I) + 1).KIND := WAVES_SYSTEM.T_LOGIC;
        L_MAP(LOGIC_VALUE'POS(I) + 1).NEXT_INDEX := CODES_INDEX; 
      end if;
    end loop;

    -- Insert Frame Set and set Frame index. 
    for I in 0 to FRAMES'LENGTH - 1 loop
      L_MAP(FRAME_INDEX + OFF_SET) := FRAMES(I + 1); 
      if FRAMES(I + 1).NEXT_INDEX <= 0 then
	L_MAP(CODES_INDEX).KIND := WAVES_SYSTEM.T_CODE;
	L_MAP(CODES_INDEX).NEXT_INDEX := FRAME_INDEX;
	FRAME_INDEX := FRAME_INDEX + OFF_SET + 1;
	OFF_SET := 0;
	CODES_INDEX := CODES_INDEX + 1;
      else
        L_MAP(FRAME_INDEX + OFF_SET).NEXT_INDEX := 
	  FRAMES(I + 1).NEXT_INDEX + LOGIC_LENGTH + CODES_LENGTH; 
        OFF_SET := OFF_SET + 1;
      end if;
    end loop;
    return L_MAP;
  end NEW_LOGIC_MAP;


  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  --
  -- Merge two LOGIC MAP's together, building a result that uses 
  -- FRAME SETS from the first operand, if present, otherwise from 
  -- the second operand.
  -- 
  function "+"(A, B : LOGIC_MAP ) return LOGIC_MAP
  is
    constant LOGIC_LENGTH : NATURAL := LOGIC_VALUE'POS(LOGIC_VALUE'RIGHT) + 1;
    variable INIT_TIME    : WAVES_SYSTEM.SYSTEM_EVENT_TIME;
    variable L_MAP : LOGIC_MAP (1 to A'LENGTH + B'LENGTH) := 
	(others => (WAVES_SYSTEM.UNUSED, -1, (0, INIT_TIME)));
  begin
    -- Insert Logic Map A
    for I in A'RANGE loop
      L_MAP(I) := A(I);
    end loop;

    -- Insert Logic Map B and update index values.
    for I in B'RANGE loop
      L_MAP(I + A'LENGTH) := B(I);
      if B(I).NEXT_INDEX > 0 then
        L_MAP(I + A'LENGTH).NEXT_INDEX := B(I).NEXT_INDEX + A'LENGTH; 
      end if;
    end loop;

    -- If Logic Value is empty for Logic Map A use Logic Map B.
    for I in 1 to LOGIC_LENGTH loop
      if A(I).NEXT_INDEX <= 0 and B(I).NEXT_INDEX > 0 then
	L_MAP(I) := B(I);
	L_MAP(I).NEXT_INDEX := B(I).NEXT_INDEX + A'LENGTH;
      end if;
    end loop;
    return L_MAP;
  end "+";

  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=- FUNCTIONS -=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- The following functions convert a single event or a single event list 
  -- into a frame (a list of tagged events).  The empty case returns a 
  -- frame of length one with the tagged event marked unspecified.
  -- 
  function FRAME_EVENT 
    return FRAME
  is
    variable F            : FRAME ( 1 to 1 );
  begin
    F(1).KIND := WAVES_SYSTEM.T_FRAME;
    F(1).NEXT_INDEX := EMPTY_FRAME;
    return F;
  end FRAME_EVENT;

  function FRAME_EVENT (
      EVENTS             : EVENT )
    return FRAME
  is
    variable F            : FRAME ( 1 to 1 );
  begin
    F(1) := (WAVES_SYSTEM.T_FRAME, END_OF_FRAME, (LOGIC_VALUE'POS(EVENTS.VALUE), 
	(EVENTS.TIME.NOMINAL,
 	 EVENTS.TIME.ET_BASIS.TOLERANCES_SPECIFIED, 
	 EVENTS.TIME.ET_BASIS.EARLIEST, EVENTS.TIME.ET_BASIS.LATEST,
	 EVENTS.TIME.ET_BASIS.BASE_EVENT, EVENTS.TIME.ET_BASIS.BASE_PIN)));
    return F;
  end FRAME_EVENT;

  function FRAME_ELIST 
    return FRAME
  is
    variable F            : FRAME ( 1 to 1 );
  begin
    F(1).KIND := WAVES_SYSTEM.T_FRAME;
    F(1).NEXT_INDEX := EMPTY_FRAME;
    return F;
  end FRAME_ELIST;

  function FRAME_ELIST (
      EVENTS             : EVENT_LIST )
    return FRAME
  is
    variable F            : FRAME (EVENTS'RANGE);
  begin
    for I in F'RANGE loop
      F(I) := (WAVES_SYSTEM.T_FRAME, I + 1, (LOGIC_VALUE'POS(EVENTS(I).VALUE), 
	(EVENTS(I).TIME.NOMINAL,
	 EVENTS(I).TIME.ET_BASIS.TOLERANCES_SPECIFIED, 
	 EVENTS(I).TIME.ET_BASIS.EARLIEST, EVENTS(I).TIME.ET_BASIS.LATEST,
	 EVENTS(I).TIME.ET_BASIS.BASE_EVENT, 
	 EVENTS(I).TIME.ET_BASIS.BASE_PIN)));
    end loop;
    F(F'LENGTH).NEXT_INDEX := END_OF_FRAME;
    return F;
  end FRAME_ELIST;

-- 
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- Delay specification for APPLY.
  -- 
  function DELAY (
      NOMINAL            : TIME )
    return DELAY_TIME
  is begin
    return (NOMINAL, (
      FALSE, 0 hr, 0 hr,
      WAVES_SYSTEM.TIMED_DELAY, 0, 0, 0 ) );
  end;
  
  function DELAY (
      NOMINAL             : TIME;
      EARLIEST_AND_LATEST : TIME )
    return DELAY_TIME
  is begin
    return (NOMINAL, (
      TRUE, EARLIEST_AND_LATEST, EARLIEST_AND_LATEST, 
      WAVES_SYSTEM.TIMED_DELAY, 0, 0, 0 ) );
  end;
  
  function DELAY (
      NOMINAL            : TIME;
      EARLIEST           : TIME;
      LATEST             : TIME )
    return DELAY_TIME
  is begin
    return (NOMINAL, (
      TRUE, EARLIEST, LATEST, 
      WAVES_SYSTEM.TIMED_DELAY, 0, 0, 0 ) );
  end;
  
  -- -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  -- 
  -- Arithmetic functions to manipulate delay times.  The functions
  -- that operate on two delay times require that the DT_BASIS 
  -- fields be the same.
  -- 
  function "+"(A : TIME;        B : DELAY_TIME ) return DELAY_TIME
  is
    variable X : DELAY_TIME := B;
  begin
    X.NOMINAL := X.NOMINAL + A;
    return X;
  end;

  function "+"(A : DELAY_TIME;  B : TIME       ) return DELAY_TIME
  is
    variable X : DELAY_TIME := A;
  begin
    X.NOMINAL := X.NOMINAL + B;
    return X;
  end;

  function "-"(A : DELAY_TIME;  B : TIME       ) return DELAY_TIME
  is
    variable X : DELAY_TIME := A;
  begin
    X.NOMINAL := X.NOMINAL - B;
    return X;
  end;

  function "*"(A : REAL;        B : DELAY_TIME ) return DELAY_TIME
  is
    variable X : DELAY_TIME := B;
  begin
    X.NOMINAL := X.NOMINAL * A;
    return X;
  end;

  function "*"(A : DELAY_TIME;  B : REAL       ) return DELAY_TIME
  is
    variable X : DELAY_TIME := A;
  begin
    X.NOMINAL := X.NOMINAL * B;
    return X;
  end;

  function "/"(A : DELAY_TIME;  B : TIME       ) return REAL
  is
  begin
    return (REAL(A.NOMINAL / B) * 1.0);
  end;

  function "+"(A : DELAY_TIME;  B : DELAY_TIME ) return DELAY_TIME
  is
    variable X : DELAY_TIME := A;
  begin
    X.NOMINAL := X.NOMINAL + B.NOMINAL;
    return X;
  end;

  function "-"(A : DELAY_TIME;  B : DELAY_TIME ) return DELAY_TIME
  is
    variable X : DELAY_TIME := A;
  begin
    X.NOMINAL := X.NOMINAL - B.NOMINAL;
    return X;
  end;

end WAVES_INTERFACE;
