-- FILE NAME:  waves_utilities.vhd
--
--
-- Version 2.1
--
--
--                                   NOTICE
--
--      LICENSEE acknowledges that it has read and executed the
--      Institute for Technology Development, VHDL Standard Component
--      Library Software Standard License Agreement.  Licensee may not
--      sublicense, assign or transfer the Software, the Documentation or its
--      rights under the Agreement.
--      
--      The Agreement authorizes Licensee to use at any one time only
--      one copy of the Software (either the original Software or the
--      Archival Copy but never both.)  Licensee may make one (1) copy
--      of the Software (the "Archival Copy") and one (1) copy of the
--      Documentation for archival purposes.  Licensee may not otherwise copy,
--      modify, transfer or disclose in any manner the Software or
--      Documentation.  Licensee shall store the Archival Copy in a secure
--      location to which access shall be restricted.  Licensee shall maintain
--      a written record of the data on which the Archival Copy is created and
--      its location.
--      
--      Licensee expressly recognizes that the Software is the proprietary
--      property of ITD.  Licensee warrants that it will not take any action
--      which would result in the impairment or loss of proprietary rights of
--      ITD in the Software.  Licensee warrants that the Software will
--      not be incorporated into any Licensee product.  Licensee warrants 
--      that it will not decompile, disassemble or reverse engineer the 
--      Software or attempt to do so for other than internal applications.
--      Licensee warrants that it will never divulge to any person without the
--      prior written consent of ITD the Software source code or Software
--      object code.
--      
--      The Software is provided "AS IS" WITHOUT WARRANTY of any kind, either
--      express or implied, including but not limited to implied warranties of
--      merchantability and fitness for a particular purpose.
--      
--      RIGHT TO ACCESS OF THIS SOFTWARE IS DENIED IF THIS NOTICE IS REMOVED.
--      
--
--        ------------------------------------------------------------------
--
--        This is a modified IEEE WAVES package.  This package was modified
--        to handle the std_ulogic state/strength value system used in the
--        models being tested.
--
--        ------------------------------------------------------------------

use STD.TEXTIO.all ;

library IEEE;
  use IEEE.STD_LOGIC_1164.all;

library STD_PACK ;
  use STD_PACK.MISC_FUNC.all ;

library WAVES_STANDARD ;
  use WAVES_STANDARD.WAVES_STANDARD.all ;
  use WAVES_STANDARD.WAVES_SYSTEM.all ;
  use WAVES_STANDARD.WAVES_EVENTS.all ;
  use WAVES_STANDARD.WAVES_INTERFACE.all ;

library WORK ;
  use WORK.WAVES_OBJECTS.all ;
  use WORK.WAVES_DEVICE.all ;

package WAVES_UTILITIES is

    type Comparison_Format is (
      Assign_To_Signal,
      Do_Assertion,
      Write_Comparison_To_File
    ) ;

    type Output_Format is (
      Standard,
      WAVES
    ) ;

    type Logic_Value_To_Std_Ulogic_Table is array 
      (Logic_Value'low to Logic_Value'high) of Std_Ulogic ;

    constant LOGIC_VALUE_TO_STD_ULOGIC : Logic_Value_To_Std_Ulogic_Table := (
      'U', -- Uninitialized
      'X', -- Unknown
      '0', -- Zero
      '1', -- One
      'Z', -- High_Z
      'W', -- Undriven_Unknown
      'L', -- Low
      'H', -- High
      '-'  -- Dont_Care
    ) ;

    function WAVES_PORT_TO_LOGIC_VALUE (INPUT:  System_Waves_Port) 
      return Logic_Value;

    procedure CHECK_RESPONSE (
        constant WHICH_ONE       : in positive ;
        signal   RESPONSE        : in Std_Ulogic;
        signal   PREDICTION      : in System_Waves_Port ;
        signal   COMPARE         : in boolean ;
        constant HOW_TO_REPORT   : in Comparison_Format ;
        constant ASSERTION_LEVEL : severity_level ;
        signal   MATCH           : out boolean
      ) ;

    procedure Signal_Monitor (
      constant OUTPUT_FILE_NAME : in string ;
      signal   SIGNALS : in Std_Ulogic_Vector ;
      constant FORMAT  : in Output_Format
    ) ;

end WAVES_UTILITIES ;

-- ======================================================================

package body WAVES_UTILITIES is

  ------------------------------------------------------------------------

  function WAVES_PORT_TO_LOGIC_VALUE(INPUT: System_Waves_Port) 
        return Logic_Value is
    begin
      if (INPUT.L_VALUE >= Logic_Value'pos(Logic_Value'left) and
          INPUT.L_VALUE <= Logic_Value'pos(Logic_Value'right)) then
        return Logic_Value'val(INPUT.L_VALUE);
      else
        return Unknown;
      end if;
  end WAVES_PORT_TO_LOGIC_VALUE ;

  ------------------------------------------------------------------------

  procedure CHECK_RESPONSE (
      constant WHICH_ONE : in positive ;
      signal RESPONSE : in Std_Ulogic;
      signal PREDICTION : in System_Waves_Port ;
      signal COMPARE : in boolean ;
      constant HOW_TO_REPORT : in Comparison_Format ;
      constant ASSERTION_LEVEL : Severity_Level ;
      signal MATCH : out boolean
  ) is
      variable COMPARISON_RESULT : boolean ;
      variable OUT_LINE : line ;
      constant RELEVANCE_REQUIRED : Relevance_Type := Relevance_Type'right ; -- Required
      variable RELEVANT : boolean := TRUE ;

      variable OLD_TIME : time := 0 ns ;
      variable OLD_PREDICTION : System_Waves_Port ;
      variable OLD_RESPONSE : Std_Ulogic ;

      variable CHANGED : boolean := TRUE ;

      -- The "old" variables are used for actual comparisons
      -- to be sure that comparisons are only made at the end
      -- of a simulation timestep (so that all events that are
      -- supposed to occur at a certain time in simulation have
      -- already happened, and there are no remaining delta cycles.

  begin

    wait on PREDICTION.L_VALUE, RESPONSE ;
    old_time := now ;

    while true loop

      if ((now > OLD_TIME) and COMPARE and CHANGED) then

          -- This means that the simulator is now in a new timestep, and
          -- it is ok to compare the prediction and response from the
          -- previous timestep

          -- Look at the event value returned by value dictionary for the 
          -- prediction to determine whether this predicted value is "masked"
          -- (Relevance /= Required => don't care)

          for I in 1 to VALUE_DICTIONARY(WAVES_PORT_TO_LOGIC_VALUE
                                         (OLD_PREDICTION))'length loop
              if (VALUE_DICTIONARY(WAVES_PORT_TO_LOGIC_VALUE
                                  (OLD_PREDICTION))(I).KIND = Relevance_Kind) then
                  RELEVANT := Relevance_Type'val(VALUE_DICTIONARY
                              (WAVES_PORT_TO_LOGIC_VALUE(OLD_PREDICTION))
                              (I).VALUE) = RELEVANCE_REQUIRED ;
              end if ;
          end loop;

          if RELEVANT then
          -- determine whether the predicted value is the same as the response
              COMPARISON_RESULT := LOGIC_VALUE_TO_STD_ULOGIC 
                                   (WAVES_PORT_TO_LOGIC_VALUE(OLD_PREDICTION)) 
                                   = OLD_RESPONSE ;

              case HOW_TO_REPORT is
                when Assign_To_Signal =>
                     MATCH <= COMPARISON_RESULT ;

                when Do_Assertion =>
                    if (COMPARISON_RESULT = false) then 
                        write(OUT_LINE, string'("Device "));
                        write(OUT_LINE, DEVICE_ID) ;
                        write(OUT_LINE, string'(" : Pin "));
                        write(OUT_LINE, PIN_NAME(Test_Pins'val(WHICH_ONE-1)));
                        write(OUT_LINE, string'(": Comparison failed at "));
                        write(OUT_LINE, OLD_TIME);
                        write(OUT_LINE, string'(" ; Actual Response is : "));
                        write(OUT_LINE, STD_ULOGIC_TO_CHAR(OLD_RESPONSE));
                        write(OUT_LINE, string'(" ; Prediction was : "));
                        write(OUT_LINE, STD_ULOGIC_TO_CHAR
                                        (LOGIC_VALUE_TO_STD_ULOGIC
                                        (WAVES_PORT_TO_LOGIC_VALUE(OLD_PREDICTION)))) ;

                        assert COMPARISON_RESULT
                          report OUT_LINE.all
                          severity Assertion_Level ;
                        deallocate(OUT_LINE) ;
                    end if ;

                when Write_Comparison_To_File =>
                    if COMPARISON_RESULT then
                        write(OUT_LINE, string'("                     "));
                    else
                        write(OUT_LINE, string'("Comparison failed at "));
                    end if;

                    write(OUT_LINE, OLD_TIME);
                    write(OUT_LINE, string'(" ; Actual Response is : "));
                    write(OUT_LINE, STD_ULOGIC_TO_CHAR(Response));
                    write(OUT_LINE, string'(" ; Prediction was : "));
                    write(OUT_LINE, STD_ULOGIC_TO_CHAR
                                    (LOGIC_VALUE_TO_STD_ULOGIC
                                    (WAVES_PORT_TO_LOGIC_VALUE(OLD_PREDICTION))));
                    writeline(output, OUT_LINE);
              end case ;
          end if ;
      else
          if HOW_TO_REPORT = Assign_To_Signal then
              MATCH <= true ;
          end if;
      end if;

      -- store the old variables

      CHANGED := (RESPONSE /= OLD_RESPONSE) or (PREDICTION /= OLD_PREDICTION) ;

      OLD_RESPONSE := RESPONSE ;
      OLD_PREDICTION := PREDICTION ;
      OLD_TIME := now ;

      -- Decide when to wake up next. If the reason the procedure woke up this time
      -- was an actual event on the prediction or response signal, then wait until
      -- there is another event or for 1 ns, whichever comes first (The reason to wait
      -- for 1 ns is in case there are no more events on these signals, in which
      -- case the procedure will need to wake up to compare the last set of values).

      -- If it woke up only because it has been 1 ns since the last event (and there
      -- were no events this time), then don't wake up again until there are some
      -- more events.

      if RESPONSE'event OR PREDICTION.L_VALUE'event then
          wait on PREDICTION.L_VALUE, RESPONSE for 1 ns ;
      else
          wait on PREDICTION.L_VALUE, RESPONSE ;
      end if ;

    end loop;

  end CHECK_RESPONSE ;

    ------------------------------------------------------------------------

  -- The following procedure produces an output trace file of the simulation.
  -- The output can be in one of two formats (depending on the value of a
  -- constant in the test bench).  One format is WAVES level 1, the other
  -- format is absolute time followed by signal values.

  procedure SIGNAL_MONITOR (
    constant OUTPUT_FILE_NAME : in string ;
    signal SIGNALS : in Std_Ulogic_Vector ;
    constant FORMAT : in Output_Format

  ) is
    file OUT_FILE : text is out Output_File_Name ;
    variable LAST_TIME : time := 0 ns ;
    variable OLD_TIME : time := 0 ns ;
    variable OUT_LINE : line ;
    variable CODES : string(SIGNALS'left + 1 to SIGNALS'right + 1) ;
    variable FIRST_TIME : boolean := true ;

    variable NAME_LENGTH : integer := 0 ;
    variable PIN : Test_Pins ;

  begin

    if (FORMAT = Standard) then

      -- print the name of the DUT and the header (pin names) 

        write (OUT_LINE, string'("--                *** "));
        write (OUT_LINE, DEVICE_ID) ;
        write (OUT_LINE, string'(" *** "));
        writeline (OUT_FILE, OUT_LINE) ;
        write (OUT_LINE, string'("--"));
        writeline (OUT_FILE, OUT_LINE) ;
    
      -- determine the length of the longest pin name
        for PIN in Test_Pins loop
            if PIN_NAME(PIN)'length + 1 > NAME_LENGTH then
                NAME_LENGTH := PIN_NAME(PIN)'length + 1 ;
            end if ;
        end loop ;
    
      -- print header
        for LINE_NUMBER in 1 to NAME_LENGTH loop
            write (OUT_LINE, string'("--                ")) ;
            for PIN in Test_Pins loop
                if PIN_NAME(PIN)'length >= LINE_NUMBER then
                    write (OUT_LINE, PIN_NAME(PIN)(LINE_NUMBER)) ;
                    write (OUT_LINE, string'(" ") ) ;
                else
                    write (OUT_LINE, string'(". ") ) ;
                end if ;
            end loop ;
            writeline (OUT_FILE, OUT_LINE) ;
        end loop ;
    

    -- print actual values
    
        for I in CODES'range loop
            CODES(I) := STD_ULOGIC_TO_CHAR(SIGNALS(I-1)) ;
        end loop ;
    
        while TRUE loop
    
            if (now > OLD_TIME) then
                write(OUT_LINE, OLD_TIME, right, 16, ps);
                write(OUT_LINE, string'(" "));
    
                for I in CODES'range loop
                    write (OUT_LINE, string'(" "));
                    write (OUT_LINE, CODES(I));
                end loop ;
    
                writeline(OUT_FILE, OUT_LINE);
            end if ;
    
            for I in CODES'range loop
                CODES(I) := STD_ULOGIC_TO_CHAR(SIGNALS(I-1)) ;
            end loop ;
    
            OLD_TIME := now ;

        -- if no signals change for 1000 ns, then stop looking    

            wait on SIGNALS for 1000 ns ;
            if not SIGNALS'event then
                wait;
            end if;

        end loop ;

    else     -- (format = WAVES level 1)

      -- print the name of the DUT and the header (pin names) 

        write (OUT_LINE, string'("% *** "));
        write (OUT_LINE, DEVICE_ID) ;
        write (OUT_LINE, string'(" *** "));
        writeline (OUT_FILE, OUT_LINE) ;
        write (OUT_LINE, string'("%"));
        writeline (OUT_FILE, OUT_LINE) ;
    
      -- determine the length of the longest pin name
        for PIN in Test_Pins loop
            if PIN_NAME(PIN)'length + 1 > NAME_LENGTH then
                NAME_LENGTH := PIN_NAME(PIN)'length + 1 ;
            end if ;
        end loop ;
    
      -- print header
        for LINE_NUMBER in 1 to NAME_LENGTH loop
            write (OUT_LINE, string'("% ")) ;
            for PIN in Test_Pins loop
                if PIN_NAME(PIN)'length >= LINE_NUMBER then
                    write (OUT_LINE, PIN_NAME(PIN)(LINE_NUMBER)) ;
                    write (OUT_LINE, string'(" ") ) ;
                else
                    write (OUT_LINE, string'(". ") ) ;
                end if ;
            end loop ;
            writeline (OUT_FILE, OUT_LINE) ;
        end loop ;
        write (OUT_LINE, string'(" ")) ;
    
    -- print actual values (in WAVES level 1 format)
    
        for I in CODES'range loop
            CODES(I) := STD_ULOGIC_TO_CHAR(SIGNALS(I-1)) ;
        end loop ;
      
        while true loop
    
            if (now > OLD_TIME) then
                for I in CODES'range loop
                    write (OUT_LINE, string'(" "));
                    write(OUT_LINE, CODES(I));
                end loop ;

                write (OUT_LINE, string'(" : ") ) ;
                write (OUT_LINE, now - OLD_TIME) ;
                write (OUT_LINE, string'("; % ")) ;
                write (OUT_LINE, OLD_TIME) ;
                writeline (OUT_FILE, OUT_LINE) ;
                write (OUT_LINE, string'(" ")) ;
  
            end if ;

            for I in CODES'range loop
                CODES(I) := STD_ULOGIC_TO_CHAR(SIGNALS(I-1)) ;
            end loop ;
      
            OLD_TIME := now ;
  
            wait on SIGNALS for 1000 ns ;
            if not SIGNALS'event then
                for I in CODES'range loop
                    write (OUT_LINE, string'(" "));
                    write(OUT_LINE, CODES(I));
                end loop ;

                write (OUT_LINE, string'(" : 100 NS ; % ") ) ;
                write (OUT_LINE, now) ;
                writeline (OUT_FILE, OUT_LINE) ;

                wait;
            end if;
        end loop ;
    end if ;

  end SIGNAL_MONITOR ;

end WAVES_UTILITIES ;
    
