--        Filename:         TIME_FUNC.vhd
--        Description:      Timing functions package
--
-- Version 2.1
--
--
--                                   NOTICE
--
--      LICENSEE acknowledges that it has read and executed the
--      Institute for Technology Development, VHDL Standard Component
--      Library Software Standard License Agreement.  Licensee may not
--      sublicense, assign or transfer the Software, the Documentation or its
--      rights under the Agreement.
--      
--      The Agreement authorizes Licensee to use at any one time only
--      one copy of the Software (either the original Software or the
--      Archival Copy but never both.)  Licensee may make one (1) copy
--      of the Software (the "Archival Copy") and one (1) copy of the
--      Documentation for archival purposes.  Licensee may not otherwise copy,
--      modify, transfer or disclose in any manner the Software or
--      Documentation.  Licensee shall store the Archival Copy in a secure
--      location to which access shall be restricted.  Licensee shall maintain
--      a written record of the data on which the Archival Copy is created and
--      its location.
--      
--      Licensee expressly recognizes that the Software is the proprietary
--      property of ITD.  Licensee warrants that it will not take any action
--      which would result in the impairment or loss of proprietary rights of
--      ITD in the Software.  Licensee warrants that the Software will
--      not be incorporated into any Licensee product.  Licensee warrants 
--      that it will not decompile, disassemble or reverse engineer the 
--      Software or attempt to do so for other than internal applications.
--      Licensee warrants that it will never divulge to any person without the
--      prior written consent of ITD the Software source code or Software
--      object code.
--      
--      The Software is provided "AS IS" WITHOUT WARRANTY of any kind, either
--      express or implied, including but not limited to implied warranties of
--      merchantability and fitness for a particular purpose.
--      
--      RIGHT TO ACCESS OF THIS SOFTWARE IS DENIED IF THIS NOTICE IS REMOVED.
--      
--
--        ------------------------------------------------------------------

library IEEE;
  use IEEE.STD_LOGIC_1164.all;

  use STD.TEXTIO.all;

package TIME_FUNC is

  -- resolution function for Boolean type
  type Boolean_Vector is array (natural range <>) of Boolean;
  function RESOLVE_BOOLEAN (FLAG : Boolean_Vector) return Boolean;
  subtype Resolved_Boolean is RESOLVE_BOOLEAN Boolean;
	
  function MAXTIME (A, B: Time) return Time;
  function MINTIME (A, B: Time) return Time;

  -- two-state logic (L and H)
  function TP_DELAY( NEWLV : Std_Ulogic; TPLH, TPHL : Time) return Time;

  -- three-state logic (L, H, and Z)
  function TP_DELAY( OLDVAL, NEWVAL : Std_Ulogic; 
                     TPLH, TPHL, TPLZ, TPHZ, TPZH, TPZL : Time) return Time;

end TIME_FUNC ;

-- ======================================================================

package body TIME_FUNC is

  function RESOLVE_BOOLEAN (FLAG : Boolean_Vector) return Boolean is
    variable RESULT : Boolean := FALSE;
    begin
      if  (FLAG'length = 1) then
         return FLAG(FLAG'low);
      else
        for I in FLAG'range loop
          if FLAG(I) = TRUE then
             RESULT := TRUE;
          end if;
        end loop;
        -- Return the resultant value 
        return RESULT;
      end if;
    end RESOLVE_BOOLEAN;
  
----------------------------------------------------------------------------
--	Returns the maximum of two times.
--
----------------------------------------------------------------------------
function MAXTIME (A, B: Time) return Time is
begin

  if (A >= B) then
    return A;
  else
    return B;
  end if;
        
end MAXTIME;

----------------------------------------------------------------------------
--	Returns the minimum of two times.
--
----------------------------------------------------------------------------
function MINTIME (A, B: Time) return Time is
begin

  if (A <= B) then
    return A;
  else
    return B;
  end if;

end MINTIME;

------------------------------------------------------
-- two-state logic (L and H)
------------------------------------------------------
function TP_DELAY( NEWLV : Std_Ulogic; TPLH, TPHL : Time)
         return Time is
begin 
 case NEWLV is
   when '0' | 'L'   => return TPHL;
   when '1' | 'H'   => return TPLH;
   when others => return MINTIME(TPLH,TPHL);
 end case;
end TP_DELAY;

------------------------------------------------------
-- three-state logic (L, H, and Z)
------------------------------------------------------
function TP_DELAY( OLDVAL, NEWVAL : Std_Ulogic; 
                   TPLH, TPHL, TPLZ, TPHZ, TPZH, TPZL : Time) 
         return Time is
begin
  case OLDVAL is
    when '0' | 'L' =>
      case NEWVAL is
        when '1' | 'H' => return TPLH;
        when 'Z'       => return TPLZ;
        when others    => return MINTIME(TPLH,TPLZ);
      end case;
    when 'H' | '1' =>
      case NEWVAL is 
        when '0' | 'L' => return TPHL;
        when 'Z'       => return TPHZ;
        when others    => return MINTIME(TPHL,TPHZ);
      end case;
    when 'Z' =>
      case NEWVAL is
        when '0' | 'L' => return TPZL;
        when '1' | 'H' => return TPZH;
        when others    => return MINTIME(TPZL,TPZH);
      end case;
    when others =>
      case NEWVAL is
        when '0' | 'L' => return MAXTIME(TPHL,TPZL);
        when '1' | 'H' => return MAXTIME(TPZH,TPLH);
        when 'Z'       => return MAXTIME(TPHZ,TPLZ);
        when others    => return 0 ns;
      end case;
    end case;
end TP_DELAY;

end TIME_FUNC ;

