-- Filename : MISC_FUNC.vhd
--
-- Version 2.1
--
--
--                                   NOTICE
--
--      LICENSEE acknowledges that it has read and executed the
--      Institute for Technology Development, VHDL Standard Component
--      Library Software Standard License Agreement.  Licensee may not
--      sublicense, assign or transfer the Software, the Documentation or its
--      rights under the Agreement.
--      
--      The Agreement authorizes Licensee to use at any one time only
--      one copy of the Software (either the original Software or the
--      Archival Copy but never both.)  Licensee may make one (1) copy
--      of the Software (the "Archival Copy") and one (1) copy of the
--      Documentation for archival purposes.  Licensee may not otherwise copy,
--      modify, transfer or disclose in any manner the Software or
--      Documentation.  Licensee shall store the Archival Copy in a secure
--      location to which access shall be restricted.  Licensee shall maintain
--      a written record of the data on which the Archival Copy is created and
--      its location.
--      
--      Licensee expressly recognizes that the Software is the proprietary
--      property of ITD.  Licensee warrants that it will not take any action
--      which would result in the impairment or loss of proprietary rights of
--      ITD in the Software.  Licensee warrants that the Software will
--      not be incorporated into any Licensee product.  Licensee warrants 
--      that it will not decompile, disassemble or reverse engineer the 
--      Software or attempt to do so for other than internal applications.
--      Licensee warrants that it will never divulge to any person without the
--      prior written consent of ITD the Software source code or Software
--      object code.
--      
--      The Software is provided "AS IS" WITHOUT WARRANTY of any kind, either
--      express or implied, including but not limited to implied warranties of
--      merchantability and fitness for a particular purpose.
--      
--      RIGHT TO ACCESS OF THIS SOFTWARE IS DENIED IF THIS NOTICE IS REMOVED.
--      
--
--        ------------------------------------------------------------------

library IEEE ;
  use IEEE.STD_LOGIC_1164.all;

package MISC_FUNC is

  type Std_Ulogic_To_Char_Table is array (Std_Ulogic'low to Std_Ulogic'high) 
    of Character ;

  type UX01_Vector is array (natural range <>) of UX01;

  function BIT_UNKNOWN (V1 : Std_Ulogic) return Boolean ; 

  function INT_TO_BITVECTOR  (INT, LENGTH:Natural)  return Std_Ulogic_Vector;
  function BITVECTOR_TO_INT  (BITS: Std_Ulogic_Vector)  return Natural;
  function BITVECTOR_TO_INT  (BITS: UX01_Vector)  return Natural;
  function UNKNOWN_ON_BUS    (BUS_VECTOR : Std_Ulogic_Vector) return Boolean;
  function UNKNOWN_ON_BUS    (BUS_VECTOR : UX01_Vector) return Boolean;


  constant STD_ULOGIC_TO_CHAR : Std_Ulogic_To_Char_Table := 
     ('U','X','0','1','Z','W','L','H','-') ;

end MISC_FUNC;

-- ======================================================================

package body MISC_FUNC is

  ----------------------------------------------------------------------------
  -- this function returns TRUE if an indeterminate logic state is passed in.
  
  function BIT_UNKNOWN    (V1 : Std_Ulogic)    
          return Boolean is
  begin
  
   case V1 is
    when 'U' | 'X' | 'Z' | 'W' | '-' => return TRUE;
    when others => return FALSE;
   end case ;
  
  end BIT_UNKNOWN ;
  
  ---------------------------------------------------------------------------
  -- This function takes as input a Std_Ulogic_Vector and returns
  -- TRUE if any of the bits are indeterminate, else it returns FALSE.
  
  function UNKNOWN_ON_BUS (BUS_VECTOR : Std_Ulogic_Vector) return Boolean is
  
  begin
  
  	for I in BUS_VECTOR'range loop
  		if  BIT_UNKNOWN(BUS_VECTOR(I)) then
  			return TRUE ;
  		end if ;
  	end loop ;
  
  	return FALSE;
  
  end UNKNOWN_ON_BUS ;

  -- overloaded for UX01_Vector

  function UNKNOWN_ON_BUS (BUS_VECTOR : UX01_Vector) return Boolean is
  
  begin
  
  	for I in BUS_VECTOR'range loop
  		if  BIT_UNKNOWN(BUS_VECTOR(I)) then
  			return TRUE ;
  		end if ;
  	end loop ;
  
  	return FALSE;
  
  end UNKNOWN_ON_BUS ;

  ---------------------------------------------------------------------------
  
  -- This function takes as input a Std_Ulogic_Vector and returns
  -- TRUE if any of the bits are 'U', else it returns FALSE.
  
  
  function UNINIT_ON_BUS    (BUS_VECTOR : Std_Ulogic_Vector) 
                                 return Boolean is
  begin

  	for I in BUS_VECTOR'range loop
  	    case BUS_VECTOR(I) is
  	       when 'U' => return TRUE ;
  	       when others => null ;
  	    end case ;   
  	end loop ;

  	return FALSE;

  end UNINIT_ON_BUS ;

  ---------------------------------------------------------------------------
  -- This function takes as input an integer and a length and returns
  -- a Std_Ulogic Vector of the input length with the value of the integer passed
  -- in.
  -- Example: INT_TO_BITVECTOR(6,4) would return a Std_Ulogic_Vector (of length 4)
  -- containing the value 0110.
  
  function INT_TO_BITVECTOR  (INT, LENGTH : Natural) return Std_Ulogic_Vector is
  
    variable DIGIT: Natural := 2 ** (LENGTH-1) ;
    variable LOCAL: Natural ;
    variable RESULT: Std_Ulogic_Vector (0 to LENGTH-1) ;
  begin
  
  	LOCAL := INT mod (2 **LENGTH) ;

  	for I in RESULT'range loop
      if LOCAL >= DIGIT then
        RESULT(I) := '1' ;
        LOCAL := LOCAL-DIGIT ;
      else
        RESULT(I) := '0' ;
      end if ;
      DIGIT := DIGIT/2 ;
  	end loop ;

  	return RESULT ;
  
  end INT_TO_BITVECTOR ;
  
  ---------------------------------------------------------------------------
  -- This function takes as input a Std_Ulogic_Vector and returns an integer.
  --
  -- Example: BITVECTOR_TO_INT(A), where A is a Std_Ulogic_Vector of length
  -- 4 containing the value 0110, would return the integer 6.
  -- The function returns the value 16 when the bit_vector is not a 
  -- valid "Bit" string.
  
  function BITVECTOR_TO_INT  (BITS: Std_Ulogic_Vector) return Natural is
  
  variable RESULT: Natural := 0 ;
  begin
  
    for I in BITS'range loop
      RESULT := RESULT * 2 ;
      if BITS(I) = '1' then
        RESULT := RESULT + 1 ;
      elsif BITS(I) /= '0' then
        return 16 ;		
      end if ;
    end loop ;
  
    return RESULT ;
  
  end BITVECTOR_TO_INT ;

  -- overloaded for UX01_Vector

  function BITVECTOR_TO_INT  (BITS: UX01_Vector) return Natural is
  
  variable RESULT: Natural := 0 ;
  begin
  
    for I in BITS'range loop
      RESULT := RESULT * 2 ;
      if BITS(I) = '1' then
        RESULT := RESULT + 1 ;
      elsif BITS(I) /= '0' then
        return 16 ;		
      end if ;
    end loop ;
  
    return RESULT ;
  
  end BITVECTOR_TO_INT ;

end MISC_FUNC;
