-- --------------------------------------------------------------------
--
--   File name :  std_logic_1164.pkg.vhdl
--   Title     :  Std_Logic_1164 multi-value logic system interface package
--   Library   :  IEEE
--   Author(s) :  W. Billowitch ( The VHDL Consulting Group )
--   Purpose   :  This packages defines a standard for digital designers
--             :  to use in describing the interconnection data types
--             :  used in modeling common ttl, cmos, GaAs, nmos, 
--             :  pmos, and ecl digital devices. 
--             : 
--   Limitation:  The logic system defined in this package may
--             :  be insufficient for modeling switched transistors,
--             :  since that requirement is out of the scope of this
--             :  effort.
--             :
--   Notation  :  No other declarations or definitions shall be included
--             :  in this package. Any additional declarations shall be 
--             :  placed in other orthogonal packages ( ie. timing, etc )
--             :
-- --------------------------------------------------------------------
--   Modification History :
-- --------------------------------------------------------------------
--   Version No:| Author:|  Mod. Date:| Changes Made:
--     v4.000   |  wdb   |   6/24/91  | Std_logic_1164 Ballot Source Code
-- --------------------------------------------------------------------
-- Library IEEE; -- proposed location of this package
-- --------------------------------------------------------------------

PACKAGE Std_Logic_1164 is

    -------------------------------------------------------------------    
    -- Logic State System  (unresolved)
    -------------------------------------------------------------------    
    TYPE std_ulogic is ( 'U',  -- Unitialized
                         'X',  -- Forcing  0 or 1
                         '0',  -- Forcing  0
                         '1',  -- Forcing  1
                         'Z',  -- High Impedance   
                         'W',  -- Weak     0 or 1
                         'L',  -- Weak     0       ( for ECL open emitter )
                         'H',  -- Weak     1       ( for open Drain or Collector )
                         '-'   -- don't care
                       );
    -------------------------------------------------------------------    
    -- Unconstrained array of std_ulogic for use with the resolution function
    -------------------------------------------------------------------    
    TYPE std_ulogic_vector IS ARRAY ( NATURAL RANGE <> ) of std_ulogic;
                                    
    -------------------------------------------------------------------    
    -- Resolution function
    -------------------------------------------------------------------    
    FUNCTION resolved ( s : std_ulogic_vector ) RETURN std_ulogic;
    
    -------------------------------------------------------------------    
    -- *** Industry Standard Logic Type ***
    -------------------------------------------------------------------    
    SUBTYPE std_logic IS resolved std_ulogic;

    -------------------------------------------------------------------    
    -- Unconstrained array of std_logic for use in declaring signal arrays
    -------------------------------------------------------------------    
    TYPE std_logic_vector IS ARRAY ( NATURAL RANGE <>) of std_logic;

    -------------------------------------------------------------------    
    -- Basic states + Test
    -------------------------------------------------------------------    
    SUBTYPE X01   is resolved std_ulogic range 'X' to '1'; -- ('X','0','1') 
    SUBTYPE X01Z  is resolved std_ulogic range 'X' to 'Z'; -- ('X','0','1','Z') 
    SUBTYPE UX01  is resolved std_ulogic range 'U' to '1'; -- ('U','X','0','1') 
    SUBTYPE UX01Z is resolved std_ulogic range 'U' to 'Z'; -- ('U','X','0','1','Z') 

    -------------------------------------------------------------------    
    -- Overloaded Logical Operators
    -------------------------------------------------------------------    

    FUNCTION "and"  ( l : std_ulogic; r : std_ulogic ) RETURN UX01;
    FUNCTION "nand" ( l : std_ulogic; r : std_ulogic ) RETURN UX01;
    FUNCTION "or"   ( l : std_ulogic; r : std_ulogic ) RETURN UX01;
    FUNCTION "nor"  ( l : std_ulogic; r : std_ulogic ) RETURN UX01;
    FUNCTION "xor"  ( l : std_ulogic; r : std_ulogic ) RETURN UX01;
    FUNCTION "not"  ( l : std_ulogic                 ) RETURN UX01;
    
    -------------------------------------------------------------------    
    -- Vectorized Overloaded Logical Operators
    -------------------------------------------------------------------    
    FUNCTION "and"  ( l, r : std_logic_vector ) RETURN std_logic_vector;
    FUNCTION "nand" ( l, r : std_logic_vector ) RETURN std_logic_vector;
    FUNCTION "or"   ( l, r : std_logic_vector ) RETURN std_logic_vector;
    FUNCTION "nor"  ( l, r : std_logic_vector ) RETURN std_logic_vector;
    FUNCTION "xor"  ( l, r : std_logic_vector ) RETURN std_logic_vector;
    FUNCTION "not"  ( l    : std_logic_vector ) RETURN std_logic_vector;

    FUNCTION "and"  ( l, r : std_ulogic_vector ) RETURN std_ulogic_vector;
    FUNCTION "nand" ( l, r : std_ulogic_vector ) RETURN std_ulogic_vector;
    FUNCTION "or"   ( l, r : std_ulogic_vector ) RETURN std_ulogic_vector;
    FUNCTION "nor"  ( l, r : std_ulogic_vector ) RETURN std_ulogic_vector;
    FUNCTION "xor"  ( l, r : std_ulogic_vector ) RETURN std_ulogic_vector;
    FUNCTION "not"  ( l    : std_ulogic_vector ) RETURN std_ulogic_vector;

    -------------------------------------------------------------------    
    -- Conversion Functions ( strips strength from buses + signals )
    -------------------------------------------------------------------    
    FUNCTION Convert_to_X01  ( s : std_logic_vector ) RETURN  std_logic_vector;
    FUNCTION Convert_to_X01  ( s : std_ulogic_vector) RETURN  std_ulogic_vector;
    FUNCTION Convert_to_X01  ( s : std_ulogic       ) RETURN  X01;
    FUNCTION Convert_to_X01  ( b : bit_vector       ) RETURN  std_logic_vector;
    FUNCTION Convert_to_X01  ( b : bit_vector       ) RETURN  std_ulogic_vector;
    FUNCTION Convert_to_X01  ( b : bit              ) RETURN  X01;       

    FUNCTION Convert_to_X01Z ( s : std_logic_vector ) RETURN  std_logic_vector;
    FUNCTION Convert_to_X01Z ( s : std_ulogic_vector) RETURN  std_ulogic_vector;
    FUNCTION Convert_to_X01Z ( s : std_ulogic       ) RETURN  X01Z;
    FUNCTION Convert_to_X01Z ( b : bit_vector       ) RETURN  std_logic_vector;
    FUNCTION Convert_to_X01Z ( b : bit_vector       ) RETURN  std_ulogic_vector;
    FUNCTION Convert_to_X01Z ( b : bit              ) RETURN  X01Z;      

    FUNCTION Convert_to_Bit  ( s : std_logic_vector ) RETURN  bit_vector;
    FUNCTION Convert_to_Bit  ( s : std_ulogic_vector) RETURN  bit_vector;
    FUNCTION Convert_to_Bit  ( s : std_ulogic       ) RETURN  bit;

    FUNCTION Convert_to_UX01 ( s : std_logic_vector ) RETURN  std_logic_vector;
    FUNCTION Convert_to_UX01 ( s : std_ulogic_vector) RETURN  std_ulogic_vector;
    FUNCTION Convert_to_UX01 ( s : std_ulogic       ) RETURN  UX01;

    -------------------------------------------------------------------    
    -- Edge Detection
    -------------------------------------------------------------------    
    FUNCTION rising_edge  (SIGNAL s : std_ulogic) RETURN boolean;
    FUNCTION falling_edge (SIGNAL s : std_ulogic) RETURN boolean;

END Std_Logic_1164;

-- --------------------------------------------------------------------
--
--   File name :  std_logic_1164.pkgbody.vhdl
--   Title     :  Std_Logic_1164 multi-value logic system interface package
--   Library   :  IEEE
--   Author(s) :  W. Billowitch ( The VHDL Consulting Group )
--   Purpose   :  This packages defines a standard for digital designers
--             :  to use in describing the interconnection data types
--             :  used in modeling common ttl, cmos, GaAs, nmos, 
--             :  pmos, and ecl digital devices. 
--             : 
--   Limitation:  The logic system defined in this package may
--             :  be insufficient for modeling switched transistors,
--             :  since that requirement is out of the scope of this
--             :  effort.
--             :
--   Notation  :  No other declarations or definitions shall be included
--             :  in this package. Any additional declarations shall be 
--             :  placed in other orthogonal packages ( ie. timing, etc )
--             :
-- --------------------------------------------------------------------
--   Modification History :
-- --------------------------------------------------------------------
--   Version No:| Author:|  Mod. Date:| Changes Made:
--     v4.000   |  wdb   |   6/24/91  | Std_logic_1164 Ballot Source Code
-- --------------------------------------------------------------------
-- Library IEEE; -- proposed location of this package
-- --------------------------------------------------------------------

PACKAGE BODY Std_Logic_1164 is
    -------------------------------------------------------------------    
    -- Local Types
    -------------------------------------------------------------------    
    TYPE stdlogic_1D is array (std_ulogic) of std_ulogic;
    TYPE stdlogic_table is array(std_ulogic, std_ulogic) of std_ulogic;

    -------------------------------------------------------------------    
    -- Resolution function
    -------------------------------------------------------------------    
    CONSTANT resolution_table : stdlogic_table := (
	--      ---------------------------------------------------------
	--      |  U    X    0    1    Z    W    L    H    - 	    |   |  
	--      ---------------------------------------------------------
            ( 'U', 'U', 'U', 'U', 'U', 'U', 'U', 'U', 'U' ), -- | U |
            ( 'U', 'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X' ), -- | X |
            ( 'U', 'X', '0', 'X', '0', '0', '0', '0', '0' ), -- | 0 |
            ( 'U', 'X', 'X', '1', '1', '1', '1', '1', '1' ), -- | 1 |
            ( 'U', 'X', '0', '1', 'Z', 'W', 'L', 'H', 'Z' ), -- | Z |
            ( 'U', 'X', '0', '1', 'W', 'W', 'W', 'W', 'W' ), -- | W |
            ( 'U', 'X', '0', '1', 'L', 'W', 'L', 'W', 'L' ), -- | L |
            ( 'U', 'X', '0', '1', 'H', 'W', 'W', 'H', 'H' ), -- | H |
            ( 'U', 'X', '0', '1', 'Z', 'W', 'L', 'H', '-' )  -- | - |
        );
        
    FUNCTION resolved ( s : std_ulogic_vector ) RETURN std_ulogic IS
        VARIABLE result : std_ulogic := '-';  -- weakest state default
    BEGIN
        IF    (s'LENGTH = 1) THEN    RETURN s(s'LOW);
        ELSE
            -- Iterate through all inputs
            FOR i IN s'RANGE LOOP
                result := resolution_table(result, s(i));
            END LOOP;
            -- Return the resultant value 
            RETURN result;
        END IF;
    END resolved;

    -------------------------------------------------------------------    
    -- Tables for Logical Operations
    -------------------------------------------------------------------    

	-- truth table for "and" function
    CONSTANT and_table : stdlogic_table := (
	--      ----------------------------------------------------
	--      |  U    X    0    1    Z    W    L    H    -         |   |  
	--      ----------------------------------------------------
            ( 'U', 'U', '0', 'U', 'U', 'U', '0', 'U', 'U' ),  -- | U |
            ( 'U', 'X', '0', 'X', 'X', 'X', '0', 'X', 'X' ),  -- | X |
            ( '0', '0', '0', '0', '0', '0', '0', '0', '0' ),  -- | 0 |
            ( 'U', 'X', '0', '1', 'X', 'X', '0', '1', 'X' ),  -- | 1 |
   	        ( 'U', 'X', '0', 'X', 'X', 'X', '0', 'X', 'X' ),  -- | Z |
   	        ( 'U', 'X', '0', 'X', 'X', 'X', '0', 'X', 'X' ),  -- | W |
   	        ( '0', '0', '0', '0', '0', '0', '0', '0', '0' ),  -- | L |
   	        ( 'U', 'X', '0', '1', 'X', 'X', '0', '1', 'X' ),  -- | H |
   	        ( 'U', 'X', '0', 'X', 'X', 'X', '0', 'X', 'X' )   -- | - |
    );

	-- truth table for "or" function
	CONSTANT or_table : stdlogic_table := (
	--      ----------------------------------------------------
	--      |  U    X    0    1    Z    W    L    H    -         |   |  
	--      ----------------------------------------------------
            ( 'U', 'U', 'U', '1', 'U', 'U', 'U', '1', 'U' ),  -- | U |
	        ( 'U', 'X', 'X', '1', 'X', 'X', 'X', '1', 'X' ),  -- | X |
	        ( 'U', 'X', '0', '1', 'X', 'X', '0', '1', 'X' ),  -- | 0 |
	        ( '1', '1', '1', '1', '1', '1', '1', '1', '1' ),  -- | 1 |
	        ( 'U', 'X', 'X', '1', 'X', 'X', 'X', '1', 'X' ),  -- | Z |
	        ( 'U', 'X', 'X', '1', 'X', 'X', 'X', '1', 'X' ),  -- | W |
	        ( 'U', 'X', '0', '1', 'X', 'X', '0', '1', 'X' ),  -- | L |
	        ( '1', '1', '1', '1', '1', '1', '1', '1', '1' ),  -- | H |
	        ( 'U', 'X', 'X', '1', 'X', 'X', 'X', '1', 'X' )   -- | - |
    );


	-- truth table for "xor" function
	CONSTANT xor_table : stdlogic_table := (
	--      ----------------------------------------------------
	--      |  U    X    0    1    Z    W    L    H    -         |   |  
	--      ----------------------------------------------------
            ( 'U', 'U', 'U', 'U', 'U', 'U', 'U', 'U', 'U' ),  -- | U |
	        ( 'U', 'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X' ),  -- | X |
	        ( 'U', 'X', '0', '1', 'X', 'X', '0', '1', 'X' ),  -- | 0 |
	        ( 'U', 'X', '1', '0', 'X', 'X', '1', '0', 'X' ),  -- | 1 |
	        ( 'U', 'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X' ),  -- | Z |
	        ( 'U', 'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X' ),  -- | W |
	        ( 'U', 'X', '0', '1', 'X', 'X', '0', '1', 'X' ),  -- | L |
	        ( 'U', 'X', '1', '0', 'X', 'X', '1', '0', 'X' ),  -- | H |
	        ( 'U', 'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X' )   -- | - |
    );

	-- truth table for "not" function
	CONSTANT not_table: stdlogic_1D := 
	--  -------------------------------------------------
	--  |   U    X    0    1    Z    W    L    H    -   |
	--  -------------------------------------------------
	     ( 'U', 'X', '1', '0', 'X', 'X', '1', '0', 'X' ); 

    -------------------------------------------------------------------    
    -- Overloaded Logical Operators ( with optimizing hints )
    -------------------------------------------------------------------    

    FUNCTION "and"  ( l : std_ulogic; r : std_ulogic ) RETURN UX01 IS
    BEGIN
        RETURN (and_table(L, R));
    END "and";

    FUNCTION "nand" ( l : std_ulogic; r : std_ulogic ) RETURN UX01 IS
    BEGIN
        RETURN  (not_table ( and_table(L, R)));
    END "nand";

    FUNCTION "or"   ( l : std_ulogic; r : std_ulogic ) RETURN UX01 IS
    BEGIN
        RETURN (or_table(L, R));
    END "or";

    FUNCTION "nor"  ( l : std_ulogic; r : std_ulogic ) RETURN UX01 IS
    BEGIN
        RETURN  (not_table ( or_table( L, R )));
    END "nor";

    FUNCTION "xor"  ( l : std_ulogic; r : std_ulogic ) RETURN UX01 IS
    BEGIN
        RETURN (xor_table(L, R));
    END "xor";

    FUNCTION "not"  ( l : std_ulogic ) RETURN UX01 IS
    BEGIN
        RETURN (not_table(L));
    END "not";
    
    -------------------------------------------------------------------    
    -- Vectorized Overloaded Logical Operators (resolved vectors)
    -------------------------------------------------------------------    
    FUNCTION "and"  ( L,R : std_logic_vector ) RETURN std_logic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_logic_vector ( 1 to L'length ) IS L;
        ALIAS RV : std_logic_vector ( 1 to R'length ) IS R;
        VARIABLE result : std_logic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        if ( L'length /= R'length ) then
            assert false
            report "Arguments of overloaded 'and' operator are not of the same length"
            severity FAILURE;
        else
            for i in result'range loop
                result(i) := and_table (LV(i), RV(i));
            end loop;
        end if;
        return result;
    end "and";

    ---------------------------------------------------------------------

    FUNCTION "nand" ( L,R : std_logic_vector ) RETURN std_logic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_logic_vector ( 1 to L'length ) IS L;
        ALIAS RV : std_logic_vector ( 1 to R'length ) IS R;
        VARIABLE result : std_logic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        if ( L'length /= R'length ) then
            assert false
            report "Arguments of overloaded 'nand' operator are not of the same length"
            severity FAILURE;
        else
            for i in result'range loop
                result(i) := not_table(and_table (LV(i), RV(i)));
            end loop;
        end if;
        return result;
    end "nand";

    ---------------------------------------------------------------------

    FUNCTION "or"  ( L,R : std_logic_vector ) RETURN std_logic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_logic_vector ( 1 to L'length ) IS L;
        ALIAS RV : std_logic_vector ( 1 to R'length ) IS R;
        VARIABLE result : std_logic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        if ( L'length /= R'length ) then
            assert false
            report "Arguments of overloaded 'or' operator are not of the same length"
            severity FAILURE;
        else
            for i in result'range loop
                result(i) := or_table (LV(i), RV(i));
            end loop;
        end if;
        return result;
    end "or";

    ---------------------------------------------------------------------

    FUNCTION "nor"  ( L,R : std_logic_vector ) RETURN std_logic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_logic_vector ( 1 to L'length ) IS L;
        ALIAS RV : std_logic_vector ( 1 to R'length ) IS R;
        VARIABLE result : std_logic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        if ( l'length /= r'length ) then
            assert false
            report "Arguments of overloaded 'nor' operator are not of the same length"
            severity FAILURE;
        else
            for i in result'range loop
                result(i) := not_table(or_table (LV(i), RV(i)));
            end loop;
        end if;
        return result;
    end "nor";

    ---------------------------------------------------------------------

    FUNCTION "xor"  ( L,R : std_logic_vector ) RETURN std_logic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_logic_vector ( 1 to L'length ) IS L;
        ALIAS RV : std_logic_vector ( 1 to R'length ) IS R;
        VARIABLE result : std_logic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        if ( l'length /= r'length ) then
            assert false
            report "Arguments of overloaded 'xor' operator are not of the same length"
            severity FAILURE;
        else
            for i in result'range loop
                result(i) := xor_table (LV(i), RV(i));
            end loop;
        end if;
        return result;
    end "xor";

    ---------------------------------------------------------------------

    FUNCTION "not"  ( l : std_logic_vector ) RETURN std_logic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_logic_vector ( 1 to L'length ) IS L;
        VARIABLE result : std_logic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        for i in result'range loop
            result(i) := not_table( LV(i) );
        end loop;
        return result;
    end;

    ---------------------------------------------------------------------
    -- Vectorized Overloaded Logical Operators (unresolved vectors)
    ---------------------------------------------------------------------

    FUNCTION "and"  ( L,R : std_ulogic_vector ) RETURN std_ulogic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_ulogic_vector ( 1 to L'length ) IS L;
        ALIAS RV : std_ulogic_vector ( 1 to R'length ) IS R;
        VARIABLE result : std_ulogic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        if ( L'length /= R'length ) then
            assert false
            report "Arguments of overloaded 'and' operator are not of the same length"
            severity FAILURE;
        else
            for i in result'range loop
                result(i) := and_table (LV(i), RV(i));
            end loop;
        end if;
        return result;
    end "and";

    ---------------------------------------------------------------------

    FUNCTION "nand" ( L,R : std_ulogic_vector ) RETURN std_ulogic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_ulogic_vector ( 1 to L'length ) IS L;
        ALIAS RV : std_ulogic_vector ( 1 to R'length ) IS R;
        VARIABLE result : std_ulogic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        if ( L'length /= R'length ) then
            assert false
            report "Arguments of overloaded 'nand' operator are not of the same length"
            severity FAILURE;
        else
            for i in result'range loop
                result(i) := not_table(and_table (LV(i), RV(i)));
            end loop;
        end if;
        return result;
    end "nand";

    ---------------------------------------------------------------------

    FUNCTION "or"  ( L,R : std_ulogic_vector ) RETURN std_ulogic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_ulogic_vector ( 1 to L'length ) IS L;
        ALIAS RV : std_ulogic_vector ( 1 to R'length ) IS R;
        VARIABLE result : std_ulogic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        if ( L'length /= R'length ) then
            assert false
            report "Arguments of overloaded 'or' operator are not of the same length"
            severity FAILURE;
        else
            for i in result'range loop
                result(i) := or_table (LV(i), RV(i));
            end loop;
        end if;
        return result;
    end "or";

    ---------------------------------------------------------------------

    FUNCTION "nor"  ( L,R : std_ulogic_vector ) RETURN std_ulogic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_ulogic_vector ( 1 to L'length ) IS L;
        ALIAS RV : std_ulogic_vector ( 1 to R'length ) IS R;
        VARIABLE result : std_ulogic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        if ( l'length /= r'length ) then
            assert false
            report "Arguments of overloaded 'nor' operator are not of the same length"
            severity FAILURE;
        else
            for i in result'range loop
                result(i) := not_table(or_table (LV(i), RV(i)));
            end loop;
        end if;
        return result;
    end "nor";

    ---------------------------------------------------------------------

    FUNCTION "xor"  ( L,R : std_ulogic_vector ) RETURN std_ulogic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_ulogic_vector ( 1 to L'length ) IS L;
        ALIAS RV : std_ulogic_vector ( 1 to R'length ) IS R;
        VARIABLE result : std_ulogic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        if ( l'length /= r'length ) then
            assert false
            report "Arguments of overloaded 'xor' operator are not of the same length"
            severity FAILURE;
        else
            for i in result'range loop
                result(i) := xor_table (LV(i), RV(i));
            end loop;
        end if;
        return result;
    end "xor";

    ---------------------------------------------------------------------

    FUNCTION "not"  ( l : std_ulogic_vector ) RETURN std_ulogic_vector IS
        -- Note : Implementations may use variables instead of the aliases
        ALIAS LV : std_ulogic_vector ( 1 to L'length ) IS L;
        VARIABLE result : std_ulogic_vector ( 1 to L'length ) := (Others => 'X');
    begin
        for i in result'range loop
            result(i) := not_table( LV(i) );
        end loop;
        return result;
    end;

    -------------------------------------------------------------------    
    -- Conversion Tables
    -------------------------------------------------------------------    
    TYPE logic_X01_table is array (std_ulogic'low to std_ulogic'high) of X01;
    TYPE logic_X01Z_table is array (std_ulogic'low to std_ulogic'high) of X01Z;
    TYPE logic_UX01_table is array (std_ulogic'low to std_ulogic'high) of UX01;
    ----------------------------------------------------------
    -- Table name : cvt_to_X01
    --
    -- Parameters :
    --        in  :: std_ulogic  -- some logic value
    -- Returns    :  X01         -- state value of logic value
    -- Purpose    :  to convert state-strength to state only
    --                  
    -- Example    : if (cvt_to_X01 (input_signal) = '1' ) then ...
    --
    ----------------------------------------------------------
    CONSTANT cvt_to_X01 : logic_X01_table := (
                         'X',  -- 'U'
                         'X',  -- 'X'
                         '0',  -- '0'
                         '1',  -- '1'
                         'X',  -- 'Z'
                         'X',  -- 'W'
                         '0',  -- 'L'
                         '1',  -- 'H'
                         'X'   -- '-'
                        );

    ----------------------------------------------------------
    -- Table name : cvt_to_X01Z
    --
    -- Parameters :
    --        in  :: std_ulogic  -- some logic value
    -- Returns    :  X01Z        -- state value of logic value
    -- Purpose    :  to convert state-strength to state only
    --                  
    -- Example    : if (cvt_to_X01Z (input_signal) = '1' ) then ...
    --
    ----------------------------------------------------------
    CONSTANT cvt_to_X01Z : logic_X01Z_table := (
                         'X',  -- 'U'
                         'X',  -- 'X'
                         '0',  -- '0'
                         '1',  -- '1'
                         'Z',  -- 'Z'
                         'X',  -- 'W'
                         '0',  -- 'L'
                         '1',  -- 'H'
                         'Z'   -- '-'
                        );

    ----------------------------------------------------------
    -- Table name : cvt_to_UX01
    --
    -- Parameters :
    --        in  :: std_ulogic  -- some logic value
    -- Returns    :  UX01        -- state value of logic value
    -- Purpose    :  to convert state-strength to state only
    --                  
    -- Example    : if (cvt_to_UX01 (input_signal) = '1' ) then ...
    --
    ----------------------------------------------------------
    CONSTANT cvt_to_UX01 : logic_UX01_table := (
                         'U',  -- 'U'
                         'X',  -- 'X'
                         '0',  -- '0'
                         '1',  -- '1'
                         'X',  -- 'Z'
                         'X',  -- 'W'
                         '0',  -- 'L'
                         '1',  -- 'H'
                         'X'   -- '-'
                        );
    
    -------------------------------------------------------------------    
    -- Conversion Functions
    -------------------------------------------------------------------    
    FUNCTION Convert_to_X01  ( s : std_logic_vector ) RETURN  std_logic_vector IS
        ALIAS SV : std_logic_vector ( 1 to s'length ) IS s;
        VARIABLE result : std_logic_vector ( 1 to s'length ) := (Others => 'X');
    BEGIN
        for i in result'range loop
            result(i) := cvt_to_X01 (SV(i));
        end loop;
        return result;
    END;

    FUNCTION Convert_to_X01  ( s : std_ulogic_vector ) RETURN  std_ulogic_vector IS
        ALIAS SV : std_ulogic_vector ( 1 to s'length ) IS s;
        VARIABLE result : std_ulogic_vector ( 1 to s'length ) := (Others => 'X');
    BEGIN
        for i in result'range loop
            result(i) := cvt_to_X01 (SV(i));
        end loop;
        return result;
    END;

    FUNCTION Convert_to_X01  ( s : std_ulogic ) RETURN  X01 IS
    BEGIN
        return (cvt_to_X01(s));
    END;

    FUNCTION Convert_to_X01  ( b : bit_vector ) RETURN  std_logic_vector IS
        ALIAS BV : bit_vector ( 1 to b'length ) IS b;
        VARIABLE result : std_logic_vector ( 1 to b'length ) := (Others => 'X');
    BEGIN
        for i in result'range loop
            case BV(i) is
                when '0' => result(i) := '0';
                when '1' => result(i) := '1';
            end case;
        end loop;
        return result;
    END;

    FUNCTION Convert_to_X01  ( b : bit_vector ) RETURN  std_ulogic_vector IS
        ALIAS BV : bit_vector ( 1 to b'length ) IS b;
        VARIABLE result : std_ulogic_vector ( 1 to b'length ) := (Others => 'X');
    BEGIN
        for i in result'range loop
            case BV(i) is
                when '0' => result(i) := '0';
                when '1' => result(i) := '1';
            end case;
        end loop;
        return result;
    END;

    FUNCTION Convert_to_X01  ( b : bit ) RETURN  X01 IS
    BEGIN
            case b is
                when '0' => return('0');
                when '1' => return('1');              
            end case;
    END;

    -------------------------------------------------------------------    
    FUNCTION Convert_to_X01Z  ( s : std_logic_vector ) RETURN  std_logic_vector IS
        ALIAS SV : std_logic_vector ( 1 to s'length ) IS s;
        VARIABLE result : std_logic_vector ( 1 to s'length ) := (Others => 'X');
    BEGIN
        for i in result'range loop
            result(i) := cvt_to_X01Z (SV(i));
        end loop;
        return result;
    END;

    FUNCTION Convert_to_X01Z  ( s : std_ulogic_vector ) RETURN  std_ulogic_vector IS
        ALIAS SV : std_ulogic_vector ( 1 to s'length ) IS s;
        VARIABLE result : std_ulogic_vector ( 1 to s'length ) := (Others => 'X');
    BEGIN
        for i in result'range loop
            result(i) := cvt_to_X01Z (SV(i));
        end loop;
        return result;
    END;

    FUNCTION Convert_to_X01Z  ( s : std_ulogic ) RETURN  X01Z IS
    BEGIN
        return (cvt_to_X01Z(s));
    END;

    FUNCTION Convert_to_X01Z  ( b : bit_vector ) RETURN  std_logic_vector IS
        ALIAS BV : bit_vector ( 1 to b'length ) IS b;
        VARIABLE result : std_logic_vector ( 1 to b'length ) := (Others => 'X');
    BEGIN
        for i in result'range loop
            case BV(i) is
                when '0' => result(i) := '0';
                when '1' => result(i) := '1';
            end case;
        end loop;
        return result;
    END;

    FUNCTION Convert_to_X01Z  ( b : bit_vector ) RETURN  std_ulogic_vector IS
        ALIAS BV : bit_vector ( 1 to b'length ) IS b;
        VARIABLE result : std_ulogic_vector ( 1 to b'length ) := (Others => 'X');
    BEGIN
        for i in result'range loop
            case BV(i) is
                when '0' => result(i) := '0';
                when '1' => result(i) := '1';
            end case;
        end loop;
        return result;
    END;

    FUNCTION Convert_to_X01Z  ( b : bit ) RETURN  X01Z IS
    BEGIN
            case b is
                when '0' => return('0');
                when '1' => return('1');              
            end case;
    END;
    -------------------------------------------------------------------    
    FUNCTION Convert_to_Bit ( s : std_logic_vector ) RETURN Bit_vector IS
        ALIAS SV : std_logic_vector ( 1 to s'length ) IS s;
        VARIABLE result : bit_vector ( 1 to s'length ) := (Others => '0');
    BEGIN
        for i in result'range loop
            case SV(i) is
                when '0' | 'L' => result(i) := '0';
                when '1' | 'H' => result(i) := '1';
                when others => assert false
                               report "CONVERT_TO_BIT ( s : std_logic_vector ) "&
                               ":: Unable to convert non { 0,1 } elements to Bit type"
                               severity FAILURE;
            end case;
        end loop;
        return result;
    END;

    FUNCTION Convert_to_Bit ( s : std_ulogic_vector ) RETURN Bit_vector IS
        ALIAS SV : std_ulogic_vector ( 1 to s'length ) IS s;
        VARIABLE result : bit_vector ( 1 to s'length ) := (Others => '0');
    BEGIN
        for i in result'range loop
            case SV(i) is
                when '0' | 'L' => result(i) := '0';
                when '1' | 'H' => result(i) := '1';
                when others => assert false
                               report "CONVERT_TO_BIT ( s : std_ulogic_vector ) "&
                               ":: Unable to convert non { 0,1 } elements to Bit type"
                               severity FAILURE;
            end case;
        end loop;
        return result;
    END;

    FUNCTION Convert_to_Bit ( s : std_ulogic ) RETURN Bit IS
    BEGIN
            case s is
                when '0' | 'L' => return ('0');
                when '1' | 'H' => return ('1');
                when others => assert false
                               report "CONVERT_TO_BIT ( s : std_logic_vector ) "&
                               ":: Unable to convert non { 0,1 } elements to Bit type"
                               severity FAILURE;
                               return ('0');
            end case;
    END;

    -------------------------------------------------------------------    
    FUNCTION Convert_to_UX01 ( s : std_logic_vector ) RETURN std_logic_vector IS
        ALIAS SV : std_logic_vector ( 1 to s'length ) IS s;
        VARIABLE result : std_logic_vector ( 1 to s'length ) := (Others => 'U');
    BEGIN
        for i in result'range loop
            result(i) := cvt_to_UX01 (SV(i));
        end loop;
        return result;
    END;

    FUNCTION Convert_to_UX01 ( s : std_ulogic_vector ) RETURN std_ulogic_vector IS
        ALIAS SV : std_ulogic_vector ( 1 to s'length ) IS s;
        VARIABLE result : std_ulogic_vector ( 1 to s'length ) := (Others => 'U');
    BEGIN
        for i in result'range loop
            result(i) := cvt_to_UX01 (SV(i));
        end loop;
        return result;
    END;

    FUNCTION Convert_to_UX01 ( s : std_ulogic ) RETURN UX01 IS
    BEGIN
        return (cvt_to_UX01(s));
    END;

    -------------------------------------------------------------------    
    -- Edge Detection
    -------------------------------------------------------------------    
    FUNCTION rising_edge  (SIGNAL s : std_ulogic) RETURN boolean is
    begin
        return (s'event and (convert_to_X01(s) = '1') and 
                            (convert_to_X01(s'last_value) = '0'));
    end;
--    
    FUNCTION falling_edge (SIGNAL s : std_ulogic) RETURN boolean is
    begin
        return (s'event and (convert_to_X01(s) = '0') and 
                            (convert_to_X01(s'last_value) = '1'));
    end;

END Std_Logic_1164;
