-- Filename : SN54xx38_timing.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
library STD_PACK;
  use STD_PACK.SIMFLAG.all;     -- Global simulation options
  use STD_PACK.TTL_TIMING.all;  -- TTL derating coefficients

package SN54xx38_TIMING is

  type Speed_Type is (MIN,TYP,MAX);

  type Prop_Times is record
    TPZL : Time;
    TPLZ : Time;
  end record;

  type Delay_Type is array (Tech_Type,Speed_Type) of Prop_Times;

  ----------- data book times ----------------
  -- NOTE:  The default load capacitance for each SN54xx38 output is 45 pf.

  constant DATASHEET_TIMES : Delay_Type :=
  (
   -- STD family
    (
      --  TPZL     TPLZ
       (  3.9 ns,   4.9 ns), -- MIN
       (  11.0 ns, 14.0 ns), -- TYP
       (  18.0 ns, 22.0 ns)  -- MAX
     ),
  
   -- S family
    (
      --  TPZL     TPLZ
       (  2.3 ns,  3.5 ns), -- MIN
       (  6.5 ns, 10.0 ns), -- TYP
       (  6.5 ns, 10.0 ns)  -- MAX
     ),
  
   -- LS family
    (
      --  TPZL     TPLZ
       (  6.3 ns,   7.0 ns), -- MIN
       (  18.0 ns, 20.0 ns), -- TYP
       (  28.0 ns, 32.0 ns)  -- MAX
     )
  );
  
  ----------------------function declarations--------------------

  function CALC_CURVE ( X :  Real; COEFFS :Four_Coeffs ) return Real;

  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times;

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOAD  : Real) return Prop_Times;

end SN54xx38_TIMING;

-- ======================================================================

package body SN54xx38_TIMING is

------------------------------------------------------------
  function CALC_CURVE ( X : Real; COEFFS : Four_Coeffs) return Real is
    variable O : Real;

  begin
    O := COEFFS(4)*(X)**3 + COEFFS(3)*(X)**2 + COEFFS(2)*(X) + COEFFS(1);
    return O;
  end CALC_CURVE;
  
------------------------------------------------------------

  function GET_TIMING (GEN_FAMILY      : Tech_Type;
                       GEN_TIME_MODE   : Time_Modes;
                       GEN_VCC         : Real;
                       GEN_TA          : Real;
                       GEN_DFACTOR     : Real) return Prop_Times is

    variable KVLZ,KVZL  : Real;        -- voltage derating factors
    variable KTLZ,KTZL  : Real;        -- temperature derating factors
    variable DERATINGLZ : Real;        -- composite derating factors
    variable DERATINGZL : Real;
    variable MTIME      : Prop_Times;

  begin

    -- Check for SIMFLAG options out-of-range 

    assert not ( GEN_TA < -55.0 or GEN_TA > 125.0 ) report
        "Temperature out of range" severity error;
    assert not ( GEN_VCC < 4.5 or GEN_VCC > 5.5 ) report
        "Voltage out of range" severity error;
    assert not ( GEN_DFACTOR < 0.0 ) report
        "Negative DERATE_FACTOR not allowed" severity error;

    -- Calculate derating factors 
    -- Note:  LZ factors use LH coefficients
    --        ZL factors use HL coefficients

    KTLZ := CALC_CURVE(GEN_TA,    TLH_COEFFICIENT(GEN_FAMILY));
    KTZL := CALC_CURVE(GEN_TA,    THL_COEFFICIENT(GEN_FAMILY));
    KVLZ := CALC_CURVE(GEN_VCC,   VLH_COEFFICIENT(GEN_FAMILY));
    KVZL := CALC_CURVE(GEN_VCC,   VHL_COEFFICIENT(GEN_FAMILY));

    -- select delay values

    case GEN_TIME_MODE is
      when MINIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MIN);
      when TYPICAL        => MTIME := DATASHEET_TIMES(GEN_FAMILY,TYP);
      when MAXIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MAX);
    end case;

    -- composite derating factors

    DERATINGLZ := KVLZ * KTLZ * (GEN_DFACTOR /100.0);
    DERATINGZL := KVZL * KTZL * (GEN_DFACTOR /100.0);

    -- apply derating factors to propagation delays

    MTIME.TPLZ := MTIME.TPLZ * DERATINGLZ;
    MTIME.TPZL := MTIME.TPZL * DERATINGZL;

    return(MTIME);
                                                  
  end GET_TIMING;
  
------------------------------------------------------------
  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOAD  : Real) return Prop_Times is
  
    -- The times specified in the TTL Data Book assume a 45 pf capacitive load.
    -- Derating factors below are used to normalize the times in accordance with
    -- the capacitive load assumed in the data book.

    variable KCLZ_45pf     : Real;    -- load capacitance derating factors
    variable KCZL_45pf     : Real;
    
    variable MTIME    : Prop_Times;
    
  begin
  
    -- Note:  LZ factors use LH coefficients
    --        ZL factors use HL coefficients

    KCLZ_45pf := CALC_CURVE (GEN_CLOAD, CLH_COEFFICIENT(GEN_FAMILY)) 
               / CALC_CURVE (45.0, CLH_COEFFICIENT(GEN_FAMILY));

    KCZL_45pf := CALC_CURVE (GEN_CLOAD, CHL_COEFFICIENT(GEN_FAMILY))
               / CALC_CURVE (45.0, CHL_COEFFICIENT(GEN_FAMILY));
    
    MTIME.TPLZ := CHIP_TIMES.TPLZ * KCLZ_45pf;
    MTIME.TPZL := CHIP_TIMES.TPZL * KCZL_45pf;
    
    return(MTIME);
  
  end BACK_ANNOTATE;
  
------------------------------------------------------------

end SN54xx38_TIMING;
