-- Filename : SN54xx368A_timing.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
library STD_PACK;
  use STD_PACK.SIMFLAG.all;     -- Global simulation options
  use STD_PACK.TTL_TIMING.all;  -- TTL derating coefficients

package SN54xx368A_TIMING is

  type Speed_Type is (MIN,TYP,MAX);

  type Prop_Times is record
    TPLH : Time;
    TPHL : Time;
    TPZH : Time;
    TPZL : Time;
    TPHZ : Time;
    TPLZ : Time;
  end record;

  type Delay_Type is array (Tech_Type,Speed_Type) of Prop_Times;

  --data book times
  -- NOTE:  The default load capacitance for each SN54xx368A output is 50 pf.
  --        However, the TPHZ and TPLZ parameters were specified only
  --        at 5 pf.  The values of TPHZ and TPLZ at 50 pf were
  --        obtained according to the capacitance derating curves for
  --        STD and LS families (located in the TTL_TIMING package).


  constant DATASHEET_TIMES : Delay_Type :=
  (
   -- STD family
    ( 
      --  TPLH    TPHL     TPZH     TPZL     TPHZ      TPLZ
       (  3.7 ns,  3.5 ns, 7.7 ns,   8.1 ns,  2.4 ns,  5.9 ns), -- MIN
       ( 10.6 ns, 10.0 ns, 21.9 ns, 23.1 ns,  6.9 ns, 16.9 ns), -- TYP
       ( 17.0 ns, 16.0 ns, 35.0 ns, 37.0 ns, 11.0 ns, 27.0 ns)  -- MAX
     ),
  
   -- S family
    ( others => (others => 0.0 ns)),
  
   -- LS family
    ( 
      --  TPLH    TPHL     TPZH     TPZL     TPHZ      TPLZ
       (  2.5 ns,  4.2 ns,  6.3 ns,  9.8 ns,  6.8 ns,  7.4 ns), -- MIN
       (  7.0 ns, 12.0 ns, 18.0 ns, 28.0 ns, 19.4 ns, 21.2 ns), -- TYP
       ( 15.0 ns, 18.0 ns, 35.0 ns, 45.0 ns, 32.0 ns, 35.0 ns)  -- MAX
     )
  );
  
  ----------------------function declarations--------------------

  function CALC_CURVE ( X :  Real; COEFFS :Four_Coeffs ) return Real;

  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times;

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOAD  : Real) return Prop_Times;

end SN54xx368A_TIMING;

-- ======================================================================

package body SN54xx368A_TIMING is

-- ------------------------------------------------------------
  function CALC_CURVE ( X : Real; COEFFS : Four_Coeffs ) return Real is

    variable O : Real;

  begin
    O := COEFFS(4)*(X)**3 + COEFFS(3)*(X)**2 + COEFFS(2)*(X) + COEFFS(1);
    return O;
  end CALC_CURVE;

-- ------------------------------------------------------------
  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times is

    variable KVLH,KVHL       : Real;   -- voltage derating factors
    variable KTLH,KTHL       : Real;   -- temperature derating factors

    variable DERATINGLH      : Real;   -- composite derating factors
    variable DERATINGHL      : Real;

    variable MTIME           : Prop_Times;
  
  begin
  
    -- Check for SIMFLAG options out-of-range

    assert not (GEN_TA < -55.0 or GEN_TA > 125.0) report 
      "Temperature out of range" severity error;
    assert not (GEN_VCC < 4.5 or GEN_VCC > 5.5 ) report 
      "Voltage out of range" severity error;
    assert not (GEN_DFACTOR < 0.0 ) report 
       "Negative GEN_DFACTOR not allowed" severity error;

    -- check illegal family selection

    assert not (GEN_FAMILY = S_FAMILY) report
      "S family not available for this part" severity error;

    -- calculate derating factors

    KTLH := CALC_CURVE (GEN_TA,   TLH_COEFFICIENT(GEN_FAMILY));
    KTHL := CALC_CURVE (GEN_TA,   THL_COEFFICIENT(GEN_FAMILY));
    KVLH := CALC_CURVE (GEN_VCC,  VLH_COEFFICIENT(GEN_FAMILY));
    KVHL := CALC_CURVE (GEN_VCC,  VHL_COEFFICIENT(GEN_FAMILY));

    -- select delay values

    case GEN_TIME_MODE is
      when MINIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MIN);
      when TYPICAL        => MTIME := DATASHEET_TIMES(GEN_FAMILY,TYP);
      when MAXIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MAX);
    end case;

    -- composite derating factors

    DERATINGLH := KVLH * KTLH * (GEN_DFACTOR / 100.0);
    DERATINGHL := KVHL * KTHL * (GEN_DFACTOR / 100.0);

    -- apply derating factors to propagation delays
    -- note:  LH factors are used also for ZH, LZ
    --        HL factors are used also for ZL, HZ

    MTIME.TPLH := MTIME.TPLH * DERATINGLH;
    MTIME.TPZH := MTIME.TPZH * DERATINGLH;
    MTIME.TPLZ := MTIME.TPLZ * DERATINGLH;

    MTIME.TPHL := MTIME.TPHL * DERATINGHL;
    MTIME.TPZL := MTIME.TPZL * DERATINGHL;
    MTIME.TPHZ := MTIME.TPHZ * DERATINGHL;

    return(MTIME);

  end GET_TIMING;

-- ----------------------------------------------------------

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOAD  : Real) return Prop_Times is

    -- The times specified in the TTL Data Book assume various capacitive loads.
    -- Derating factors below are used to normalize the times in accordance with
    -- the capacitive load assumed in the data book.

    variable KCLH_5pf,  KCHL_5pf   : Real;  -- 5 pf loading
    variable KCLH_15pf, KCHL_15pf  : Real;  -- 15 pf loading
    variable KCLH_45pf, KCHL_45pf  : Real;  -- 45 pf loading
    variable KCLH_50pf, KCHL_50pf  : Real;  -- 50 pf loading

    variable MTIME           : Prop_Times;

  begin

    -- default curves are normalized around 15 pf load

    KCLH_15pf := CALC_CURVE (GEN_CLOAD, CLH_COEFFICIENT(GEN_FAMILY));
    KCHL_15pf := CALC_CURVE (GEN_CLOAD, CHL_COEFFICIENT(GEN_FAMILY));

    -- normalize around 5 pf load

    KCLH_5pf := KCLH_15pf / CALC_CURVE (5.0, CLH_COEFFICIENT(GEN_FAMILY));
    KCHL_5pf := KCHL_15pf / CALC_CURVE (5.0, CHL_COEFFICIENT(GEN_FAMILY));

    -- normalize around 45 pf load

    KCLH_45pf := KCLH_15pf / CALC_CURVE (45.0, CLH_COEFFICIENT(GEN_FAMILY));
    KCHL_45pf := KCHL_15pf / CALC_CURVE (45.0, CHL_COEFFICIENT(GEN_FAMILY));

    -- normalize around 50 pf load

    KCLH_50pf := KCLH_15pf / CALC_CURVE (50.0, CLH_COEFFICIENT(GEN_FAMILY));
    KCHL_50pf := KCHL_15pf / CALC_CURVE (50.0, CHL_COEFFICIENT(GEN_FAMILY));

    -- apply derating factors to propagation delays
    -- note:  LH factors are used also for ZH, LZ
    --        HL factors are used also for ZL, HZ

    case GEN_FAMILY is
      when STD_FAMILY  => 
        MTIME.TPLH  := CHIP_TIMES.TPLH * KCLH_50pf;
        MTIME.TPHL  := CHIP_TIMES.TPHL * KCHL_50pf;
        MTIME.TPZH  := CHIP_TIMES.TPZH * KCLH_50pf;
        MTIME.TPZL  := CHIP_TIMES.TPZL * KCHL_50pf;
        MTIME.TPHZ  := CHIP_TIMES.TPHZ * KCHL_5pf;
        MTIME.TPLZ  := CHIP_TIMES.TPLZ * KCLH_5pf;
      when S_FAMILY =>
        assert FALSE report "S Family not available" severity error;
      when LS_FAMILY =>
        MTIME.TPLH  := CHIP_TIMES.TPLH * KCLH_45pf;
        MTIME.TPHL  := CHIP_TIMES.TPHL * KCHL_45pf;
        MTIME.TPZH  := CHIP_TIMES.TPZH * KCLH_45pf;
        MTIME.TPZL  := CHIP_TIMES.TPZL * KCHL_45pf;
        MTIME.TPHZ  := CHIP_TIMES.TPHZ * KCHL_5pf;
        MTIME.TPLZ  := CHIP_TIMES.TPLZ * KCLH_5pf;
      end case;

    return(MTIME);

  end BACK_ANNOTATE;

------------------------------------------------------------

end SN54xx368A_TIMING;
