-- Filename : SN54xx280_timing.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
library STD_PACK;
  use STD_PACK.SIMFLAG.all;     -- Global simulation options
  use STD_PACK.TTL_TIMING.all;  -- TTL derating coefficients

package SN54xx280_TIMING is

  type Speed_Type is (MIN,TYP,MAX);

  type Prop_Times is record
    TPLH_EVEN : Time;
    TPHL_EVEN : Time;
    TPLH_ODD  : Time;
    TPHL_ODD  : Time;
  end record;

  type Delay_Type is array (Tech_Type,Speed_Type) of Prop_Times;

  type Cload_Out is record  -- capacitive loadings on output pins
    PIN_EVEN  : Real ;
    PIN_ODD : Real ;
  end record;

  ----------- data book times ----------------

  constant DATASHEET_TIMES : Delay_Type :=
  (
   -- STD family
    ( others => (others => 0.0 ns)),

   -- S family
    (
      --  TPLH_EVEN  TPHL_EVEN  TPHL_ODD  TPHL_ODD
       (  4.9 ns,    4.0 ns,    4.9 ns,   4.0 ns), -- MIN
       ( 14.0 ns,   11.5 ns,   14.0 ns,  11.5 ns), -- TYP
       ( 21.0 ns,   18.0 ns,   21.0 ns,  18.0 ns)  -- MAX
    ),
   -- LS family
    (
      --  TPLH_EVEN  TPHL_EVEN  TPHL_ODD  TPHL_ODD
       ( 11.6 ns,   10.2 ns,   8.1  ns,  10.9 ns), -- MIN
       ( 33.0 ns,   29.0 ns,   23.0 ns,  31.0 ns), -- TYP
       ( 50.0 ns,   45.0 ns,   35.0 ns,  50.0 ns)  -- MAX
    ) 
  );

  -------------- function declarations -----------------------

  function CALC_CURVE ( X :  Real; COEFFS :Four_Coeffs ) return Real;
  
  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times;

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times;

end SN54xx280_TIMING;

-- ======================================================================

package body SN54xx280_TIMING is

------------------------------------------------------------

  function CALC_CURVE ( X : Real; COEFFS : Four_Coeffs) return Real is
    variable O : Real;

  begin
    O := COEFFS(4)*(X)**3 + COEFFS(3)*(X)**2 + COEFFS(2)*(X) + COEFFS(1);
    return O;
  end CALC_CURVE;

------------------------------------------------------------

  function GET_TIMING (GEN_FAMILY      : Tech_Type;
                       GEN_TIME_MODE   : Time_Modes;
                       GEN_VCC         : Real;
                       GEN_TA          : Real;
                       GEN_DFACTOR     : Real) return Prop_Times is

    variable KVLH,KVHL       : Real;        -- voltage derating factors
    variable KTLH,KTHL       : Real;        -- temperature derating factors
    variable DERATINGLH      : Real;        -- composite derating factors
    variable DERATINGHL      : Real;
    variable MTIME           : Prop_Times;  

  begin

    -- Check for SIMFLAG options out-of-range

    assert not ( GEN_TA < -55.0 or GEN_TA > 125.0 ) report
        "Temperature out of range" severity error;
    assert not ( GEN_VCC < 4.5 or GEN_VCC > 5.5 ) report
        "Voltage out of range" severity error;
    assert not ( GEN_DFACTOR < 0.0 ) report
        "Negative DERATE_FACTOR not allowed" severity error;

    -- Check for illegal family selection

    assert not (GEN_FAMILY = STD_FAMILY) report
        "STD family not available for this part." severity error;

    -- Calculate derating factors

    KTLH := CALC_CURVE(GEN_TA,  TLH_COEFFICIENT(GEN_FAMILY));
    KTHL := CALC_CURVE(GEN_TA,  THL_COEFFICIENT(GEN_FAMILY));
    KVLH := CALC_CURVE(GEN_VCC, VLH_COEFFICIENT(GEN_FAMILY));
    KVHL := CALC_CURVE(GEN_VCC, VHL_COEFFICIENT(GEN_FAMILY));

    -- select delay values

    case GEN_TIME_MODE is
      when MINIMUM  => MTIME := DATASHEET_TIMES(GEN_FAMILY,MIN);
      when TYPICAL  => MTIME := DATASHEET_TIMES(GEN_FAMILY,TYP);
      when MAXIMUM  => MTIME := DATASHEET_TIMES(GEN_FAMILY,MAX);
    end case;

    -- composite derating factors

    DERATINGLH := KVLH * KTLH * (GEN_DFACTOR /100.0);
    DERATINGHL := KVHL * KTHL * (GEN_DFACTOR /100.0);

    -- apply derating factors to propagation delays

    MTIME.TPLH_EVEN := MTIME.TPLH_EVEN * DERATINGLH;
    MTIME.TPHL_EVEN := MTIME.TPHL_EVEN * DERATINGHL;
    MTIME.TPLH_ODD  := MTIME.TPLH_ODD  * DERATINGLH;
    MTIME.TPHL_ODD  := MTIME.TPHL_ODD  * DERATINGHL;

    return(MTIME);

  end GET_TIMING;

------------------------------------------------------------

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times is

    variable KCLH     : Cload_Out;  -- load capacitance derating factors
    variable KCHL     : Cload_Out;

    variable MTIME    : Prop_Times;

  begin

    KCLH.PIN_EVEN := CALC_CURVE(GEN_CLOADS.PIN_EVEN,CLH_COEFFICIENT(GEN_FAMILY));
    KCLH.PIN_ODD  := CALC_CURVE(GEN_CLOADS.PIN_ODD,CLH_COEFFICIENT(GEN_FAMILY));

    KCHL.PIN_EVEN := CALC_CURVE(GEN_CLOADS.PIN_EVEN,CHL_COEFFICIENT(GEN_FAMILY));
    KCHL.PIN_ODD  := CALC_CURVE(GEN_CLOADS.PIN_ODD,CHL_COEFFICIENT(GEN_FAMILY));

    -- apply derating factors to propagation delays

    MTIME.TPLH_EVEN := CHIP_TIMES.TPLH_EVEN * KCLH.PIN_EVEN;
    MTIME.TPHL_EVEN := CHIP_TIMES.TPHL_EVEN * KCHL.PIN_EVEN;
    MTIME.TPLH_ODD  := CHIP_TIMES.TPLH_ODD  * KCLH.PIN_ODD;
    MTIME.TPHL_ODD  := CHIP_TIMES.TPHL_ODD  * KCHL.PIN_ODD;

    return(MTIME);

  end BACK_ANNOTATE;

------------------------------------------------------------

end SN54xx280_TIMING;
