-- Filename : SN54xx251_timing.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
library STD_PACK;
  use STD_PACK.SIMFLAG.all;     -- Global simulation options
  use STD_PACK.TTL_TIMING.all;  -- TTL derating coefficients
  use STD_PACK.MISC_FUNC.all;   -- Miscellaneous

package SN54xx251_TIMING is

  type Speed_Type is (MIN,TYP,MAX);

  type Prop_Times is record
    TPLH_BS_Y   : Time;
    TPHL_BS_Y   : Time;
    TPLH_BS_W   : Time;
    TPHL_BS_W   : Time;
    TPLH_D_Y    : Time;
    TPHL_D_Y    : Time;
    TPLH_D_W    : Time;
    TPHL_D_W    : Time;
    TPZH_G_Y    : Time;
    TPZL_G_Y    : Time;
    TPZH_G_W    : Time;
    TPZL_G_W    : Time;
    TPHZ_G_Y    : Time;
    TPLZ_G_Y    : Time;
    TPHZ_G_W    : Time;
    TPLZ_G_W    : Time;
  end record;

  type Delay_Type is array (Tech_Type,Speed_Type) of Prop_Times;

  type Cload_Out is record
    PIN_Y : Real;
    PIN_W : Real;
  end record;

  ----------- data book times ----------------

  constant DATASHEET_TIMES : Delay_Type :=
  (
   -- STD family
    (
      --  minimum times
      (
         -- the following times assume 50 pf load
         TPLH_BS_Y   => 10.2 ns,
         TPHL_BS_Y   =>  9.8 ns,
         TPLH_BS_W   =>  7.0 ns,
         TPHL_BS_W   =>  7.4 ns,
         TPLH_D_Y    =>  6.0 ns,
         TPHL_D_Y    =>  6.3 ns,
         TPLH_D_W    =>  3.5 ns,
         TPHL_D_W    =>  3.2 ns,
         TPZH_G_Y    =>  6.0 ns,
         TPZL_G_Y    =>  9.1 ns,
         TPZH_G_W    =>  6.0 ns,
         TPZL_G_W    =>  8.4 ns,
         -- the following times assume 5 pf load
         TPHZ_G_Y    =>  1.8 ns,
         TPLZ_G_Y    =>  5.3 ns,
         TPHZ_G_W    =>  1.8 ns,
         TPLZ_G_W    =>  5.3 ns
       ),

      --  typical times
      (
         -- the following times assume 50 pf load
         TPLH_BS_Y   => 29.0 ns,
         TPHL_BS_Y   => 28.0 ns,
         TPLH_BS_W   => 20.0 ns,
         TPHL_BS_W   => 21.0 ns,
         TPLH_D_Y    => 17.0 ns,
         TPHL_D_Y    => 18.0 ns,
         TPLH_D_W    => 10.0 ns,
         TPHL_D_W    =>  9.0 ns,
         TPZH_G_Y    => 17.0 ns,
         TPZL_G_Y    => 26.0 ns,
         TPZH_G_W    => 17.0 ns,
         TPZL_G_W    => 24.0 ns,
         -- the following times assume 5 pf load
         TPHZ_G_Y    =>  5.0 ns,
         TPLZ_G_Y    => 15.0 ns,
         TPHZ_G_W    =>  5.0 ns,
         TPLZ_G_W    => 15.0 ns
       ),

      --  maximum times
      (
         -- the following times assume 50 pf load
         TPLH_BS_Y   => 45.0 ns,
         TPHL_BS_Y   => 45.0 ns,
         TPLH_BS_W   => 33.0 ns,
         TPHL_BS_W   => 33.0 ns,
         TPLH_D_Y    => 28.0 ns,
         TPHL_D_Y    => 28.0 ns,
         TPLH_D_W    => 15.0 ns,
         TPHL_D_W    => 15.0 ns,
         TPZH_G_Y    => 27.0 ns,
         TPZL_G_Y    => 40.0 ns,
         TPZH_G_W    => 27.0 ns,
         TPZL_G_W    => 40.0 ns,
         -- the following times assume 5 pf load
         TPHZ_G_Y    =>  8.0 ns,
         TPLZ_G_Y    => 23.0 ns,
         TPHZ_G_W    =>  8.0 ns,
         TPLZ_G_W    => 23.0 ns
       )
    ),

   -- S family
    (
      --  minimum times
      (
         -- the following times assume 15 pf load
         TPLH_BS_Y   => 4.2 ns,
         TPHL_BS_Y   => 4.6 ns,
         TPLH_BS_W   => 3.5 ns,
         TPHL_BS_W   => 3.2 ns,
         TPLH_D_Y    => 2.8 ns,
         TPHL_D_Y    => 2.8 ns,
         TPLH_D_W    => 1.6 ns,
         TPHL_D_W    => 1.6 ns,
         -- the following times assume 50 pf load
         TPZH_G_Y    => 4.6 ns,
         TPZL_G_Y    => 4.9 ns,
         TPZH_G_W    => 4.6 ns,
         TPZL_G_W    => 4.9 ns,
         -- the following times assume 5 pf load
         TPHZ_G_Y    => 1.9 ns,
         TPLZ_G_Y    => 3.2 ns,
         TPHZ_G_W    => 1.9 ns,
         TPLZ_G_W    => 3.2 ns
       ),

      --  typical times
      (
         -- the following times assume 15 pf load
         TPLH_BS_Y   => 12.0 ns,
         TPHL_BS_Y   => 13.0 ns,
         TPLH_BS_W   => 10.0 ns,
         TPHL_BS_W   =>  9.0 ns,
         TPLH_D_Y    =>  8.0 ns,
         TPHL_D_Y    =>  8.0 ns,
         TPLH_D_W    =>  4.5 ns,
         TPHL_D_W    =>  4.5 ns,
         -- the following times assume 50 pf load
         TPZH_G_Y    => 13.0 ns,
         TPZL_G_Y    => 14.0 ns,
         TPZH_G_W    => 13.0 ns,
         TPZL_G_W    => 14.0 ns,
         -- the following times assume 5 pf load
         TPHZ_G_Y    =>  5.5 ns,
         TPLZ_G_Y    =>  9.0 ns,
         TPHZ_G_W    =>  5.5 ns,
         TPLZ_G_W    =>  9.0 ns
       ),

      --  maximum times
      (
         -- the following times assume 15 pf load
         TPLH_BS_Y   => 18.0 ns,
         TPHL_BS_Y   => 19.5 ns,
         TPLH_BS_W   => 15.0 ns,
         TPHL_BS_W   => 13.5 ns,
         TPLH_D_Y    => 12.0 ns,
         TPHL_D_Y    => 12.0 ns,
         TPLH_D_W    =>  7.0 ns,
         TPHL_D_W    =>  7.0 ns,
         -- the following times assume 50 pf load
         TPZH_G_Y    => 19.5 ns,
         TPZL_G_Y    => 21.0 ns,
         TPZH_G_W    => 19.5 ns,
         TPZL_G_W    => 21.0 ns,
         -- the following times assume 5 pf load
         TPHZ_G_Y    =>  8.5 ns,
         TPLZ_G_Y    => 14.0 ns,
         TPHZ_G_W    =>  8.5 ns,
         TPLZ_G_W    => 14.0 ns
       )
    ),

   -- LS family
    (
      --  minimum times
      (
         -- the following times assume 15 pf load
         TPLH_BS_Y   => 10.2 ns,
         TPHL_BS_Y   =>  9.8 ns,
         TPLH_BS_W   =>  7.0 ns,
         TPHL_BS_W   =>  7.4 ns,
         TPLH_D_Y    =>  6.0 ns,
         TPHL_D_Y    =>  6.3 ns,
         TPLH_D_W    =>  3.5 ns,
         TPHL_D_W    =>  3.2 ns,
         TPZH_G_Y    => 10.5 ns,
         TPZL_G_Y    =>  9.1 ns,
         TPZH_G_W    =>  6.0 ns,
         TPZL_G_W    =>  8.4 ns,
         -- the following times assume 5 pf load
         TPHZ_G_Y    => 10.5 ns,
         TPLZ_G_Y    =>  5.3 ns,
         TPHZ_G_W    => 13.0 ns,
         TPLZ_G_W    =>  5.3 ns
       ),

      --  typical times
      (
         -- the following times assume 15 pf load
         TPLH_BS_Y   => 29.0 ns,
         TPHL_BS_Y   => 28.0 ns,
         TPLH_BS_W   => 20.0 ns,
         TPHL_BS_W   => 21.0 ns,
         TPLH_D_Y    => 17.0 ns,
         TPHL_D_Y    => 18.0 ns,
         TPLH_D_W    => 10.0 ns,
         TPHL_D_W    =>  9.0 ns,
         TPZH_G_Y    => 30.0 ns,
         TPZL_G_Y    => 26.0 ns,
         TPZH_G_W    => 17.0 ns,
         TPZL_G_W    => 24.0 ns,
         -- the following times assume 5 pf load
         TPHZ_G_Y    => 30.0 ns,
         TPLZ_G_Y    => 15.0 ns,
         TPHZ_G_W    => 37.0 ns,
         TPLZ_G_W    => 15.0 ns
       ),

      --  maximum times
      (
         -- the following times assume 15 pf load
         TPLH_BS_Y   => 45.0 ns,
         TPHL_BS_Y   => 45.0 ns,
         TPLH_BS_W   => 33.0 ns,
         TPHL_BS_W   => 33.0 ns,
         TPLH_D_Y    => 28.0 ns,
         TPHL_D_Y    => 28.0 ns,
         TPLH_D_W    => 15.0 ns,
         TPHL_D_W    => 15.0 ns,
         TPZH_G_Y    => 45.0 ns,
         TPZL_G_Y    => 40.0 ns,
         TPZH_G_W    => 27.0 ns,
         TPZL_G_W    => 40.0 ns,
         -- the following times assume 5 pf load
         TPHZ_G_Y    => 45.0 ns,
         TPLZ_G_Y    => 25.0 ns,
         TPHZ_G_W    => 55.0 ns,
         TPLZ_G_W    => 25.0 ns
       )
    )
  );

  ----------------------function declarations--------------------
  
  function CALC_CURVE ( X :  Real; COEFFS :Four_Coeffs ) return Real;
  
  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times;

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times;

end SN54xx251_TIMING;

-- ======================================================================

package body SN54xx251_TIMING is

------------------------------------------------------------

  function CALC_CURVE ( X : Real; COEFFS : Four_Coeffs) return Real is
    variable O : Real;

  begin
    O := COEFFS(4)*(X)**3 + COEFFS(3)*(X)**2 + COEFFS(2)*(X) + COEFFS(1);
    return O;
  end CALC_CURVE;

------------------------------------------------------------

  function GET_TIMING (GEN_FAMILY      : Tech_Type;
                       GEN_TIME_MODE   : Time_Modes;
                       GEN_VCC         : Real;
                       GEN_TA          : Real;
                       GEN_DFACTOR     : Real) return Prop_Times is

    variable KVLH,KVHL       : Real;        -- voltage derating factors
    variable KTLH,KTHL       : Real;        -- temperature derating factors
    variable DERATINGLH      : Real;        -- composite derating factors
    variable DERATINGHL      : Real;
    variable MTIME           : Prop_Times;  

  begin

    -- Check for SIMFLAG options out-of-range

    assert not ( GEN_TA < -55.0 or GEN_TA > 125.0 ) report
        "Temperature out of range" severity error;
    assert not ( GEN_VCC < 4.5 or GEN_VCC > 5.5 ) report
        "Voltage out of range" severity error;
    assert not ( GEN_DFACTOR < 0.0 ) report
        "Negative DERATE_FACTOR not allowed" severity error;


    -- Calculate derating factors

    KTLH := CALC_CURVE(GEN_TA,    TLH_COEFFICIENT(GEN_FAMILY));
    KTHL := CALC_CURVE(GEN_TA,    THL_COEFFICIENT(GEN_FAMILY));
    KVLH := CALC_CURVE(GEN_VCC,   VLH_COEFFICIENT(GEN_FAMILY));
    KVHL := CALC_CURVE(GEN_VCC,   VHL_COEFFICIENT(GEN_FAMILY));

    -- select delay values

    case GEN_TIME_MODE is
      when MINIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MIN);
      when TYPICAL        => MTIME := DATASHEET_TIMES(GEN_FAMILY,TYP);
      when MAXIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MAX);
    end case;

    -- composite derating factors

    DERATINGLH := KVLH * KTLH * (GEN_DFACTOR /100.0);
    DERATINGHL := KVHL * KTHL * (GEN_DFACTOR /100.0);

    -- apply derating factors to propagation delays
    -- note:  LH factors are applied to ZH, LZ
    --        HL factors are applied to ZL, HZ

    MTIME.TPLH_BS_Y   := MTIME.TPLH_BS_Y * DERATINGLH;
    MTIME.TPHL_BS_Y   := MTIME.TPHL_BS_Y * DERATINGHL;
    MTIME.TPLH_BS_W   := MTIME.TPLH_BS_W * DERATINGLH;
    MTIME.TPHL_BS_W   := MTIME.TPHL_BS_W * DERATINGHL;
    MTIME.TPLH_D_Y    := MTIME.TPLH_D_Y  * DERATINGLH;
    MTIME.TPHL_D_Y    := MTIME.TPHL_D_Y  * DERATINGHL;
    MTIME.TPLH_D_W    := MTIME.TPLH_D_W  * DERATINGLH;
    MTIME.TPHL_D_W    := MTIME.TPHL_D_W  * DERATINGHL;
    MTIME.TPZH_G_Y    := MTIME.TPZH_G_Y  * DERATINGLH;
    MTIME.TPZL_G_Y    := MTIME.TPZL_G_Y  * DERATINGHL;
    MTIME.TPZH_G_W    := MTIME.TPZH_G_W  * DERATINGLH;
    MTIME.TPZL_G_W    := MTIME.TPZL_G_W  * DERATINGHL;
    MTIME.TPHZ_G_Y    := MTIME.TPHZ_G_Y  * DERATINGHL;
    MTIME.TPLZ_G_Y    := MTIME.TPLZ_G_Y  * DERATINGLH;
    MTIME.TPHZ_G_W    := MTIME.TPHZ_G_W  * DERATINGHL;
    MTIME.TPLZ_G_W    := MTIME.TPLZ_G_W  * DERATINGLH;

    return(MTIME);

  end GET_TIMING;

------------------------------------------------------------

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times is

    -- The times specified in the TTL Data Book assume various capacitive loads.
    -- Derating factors below are used to normalize the times in accordance with
    -- the capacitive load assumed in the data book.

    variable KCLH_Y_5pf,  KCHL_Y_5pf  : Real;  -- 5 pf loading
    variable KCLH_W_5pf,  KCHL_W_5pf  : Real;

    variable KCLH_Y_15pf, KCHL_Y_15pf  : Real; -- 15 pf loading
    variable KCLH_W_15pf, KCHL_W_15pf  : Real;

    variable KCLH_Y_50pf, KCHL_Y_50pf  : Real; -- 50 pf loading
    variable KCLH_W_50pf, KCHL_W_50pf  : Real;

    -- note:  LH factors also apply to ZH, LZ
    --        HL factors also apply to ZL, HZ

    variable MTIME    : Prop_Times;

  begin

    -- default curves are normalized around 15 pf load

    KCLH_Y_15pf := CALC_CURVE (GEN_CLOADS.PIN_Y, CLH_COEFFICIENT(GEN_FAMILY));
    KCHL_Y_15pf := CALC_CURVE (GEN_CLOADS.PIN_Y, CHL_COEFFICIENT(GEN_FAMILY));
    KCLH_W_15pf := CALC_CURVE (GEN_CLOADS.PIN_W, CLH_COEFFICIENT(GEN_FAMILY));
    KCHL_W_15pf := CALC_CURVE (GEN_CLOADS.PIN_W, CHL_COEFFICIENT(GEN_FAMILY));

    -- normalize around 5 pf load

    KCLH_Y_5pf := KCLH_Y_15pf / CALC_CURVE (5.0, CLH_COEFFICIENT(GEN_FAMILY));
    KCHL_Y_5pf := KCHL_Y_15pf / CALC_CURVE (5.0, CHL_COEFFICIENT(GEN_FAMILY));
    KCLH_W_5pf := KCLH_W_15pf / CALC_CURVE (5.0, CLH_COEFFICIENT(GEN_FAMILY));
    KCHL_W_5pf := KCHL_W_15pf / CALC_CURVE (5.0, CHL_COEFFICIENT(GEN_FAMILY));

    -- normalize around 50 pf load

    KCLH_Y_50pf := KCLH_Y_15pf / CALC_CURVE (50.0, CLH_COEFFICIENT(GEN_FAMILY));
    KCHL_Y_50pf := KCHL_Y_15pf / CALC_CURVE (50.0, CHL_COEFFICIENT(GEN_FAMILY));
    KCLH_W_50pf := KCLH_W_15pf / CALC_CURVE (50.0, CLH_COEFFICIENT(GEN_FAMILY));
    KCHL_W_50pf := KCHL_W_15pf / CALC_CURVE (50.0, CHL_COEFFICIENT(GEN_FAMILY));


    case GEN_FAMILY is
      when STD_FAMILY  => 
        MTIME.TPLH_BS_Y   := CHIP_TIMES.TPLH_BS_Y * KCLH_Y_50pf;
        MTIME.TPHL_BS_Y   := CHIP_TIMES.TPHL_BS_Y * KCHL_Y_50pf;
        MTIME.TPLH_BS_W   := CHIP_TIMES.TPLH_BS_W * KCLH_W_50pf;
        MTIME.TPHL_BS_W   := CHIP_TIMES.TPHL_BS_W * KCHL_W_50pf;
        MTIME.TPLH_D_Y    := CHIP_TIMES.TPLH_D_Y  * KCLH_Y_50pf;
        MTIME.TPHL_D_Y    := CHIP_TIMES.TPHL_D_Y  * KCHL_Y_50pf;
        MTIME.TPLH_D_W    := CHIP_TIMES.TPLH_D_W  * KCLH_W_50pf;
        MTIME.TPHL_D_W    := CHIP_TIMES.TPHL_D_W  * KCHL_W_50pf;
        MTIME.TPZH_G_Y    := CHIP_TIMES.TPZH_G_Y  * KCLH_Y_50pf;
        MTIME.TPZL_G_Y    := CHIP_TIMES.TPZL_G_Y  * KCHL_Y_50pf;
        MTIME.TPZH_G_W    := CHIP_TIMES.TPZH_G_W  * KCLH_W_50pf;
        MTIME.TPZL_G_W    := CHIP_TIMES.TPZL_G_W  * KCHL_W_50pf;
        MTIME.TPHZ_G_Y    := CHIP_TIMES.TPHZ_G_Y  * KCHL_Y_5pf;
        MTIME.TPLZ_G_Y    := CHIP_TIMES.TPLZ_G_Y  * KCLH_Y_5pf;
        MTIME.TPHZ_G_W    := CHIP_TIMES.TPHZ_G_W  * KCHL_W_5pf;
        MTIME.TPLZ_G_W    := CHIP_TIMES.TPLZ_G_W  * KCLH_W_5pf;

      when S_FAMILY =>
        MTIME.TPLH_BS_Y   := CHIP_TIMES.TPLH_BS_Y * KCLH_Y_15pf;
        MTIME.TPHL_BS_Y   := CHIP_TIMES.TPHL_BS_Y * KCHL_Y_15pf;
        MTIME.TPLH_BS_W   := CHIP_TIMES.TPLH_BS_W * KCLH_W_15pf;
        MTIME.TPHL_BS_W   := CHIP_TIMES.TPHL_BS_W * KCHL_W_15pf;
        MTIME.TPLH_D_Y    := CHIP_TIMES.TPLH_D_Y  * KCLH_Y_15pf;
        MTIME.TPHL_D_Y    := CHIP_TIMES.TPHL_D_Y  * KCHL_Y_15pf;
        MTIME.TPLH_D_W    := CHIP_TIMES.TPLH_D_W  * KCLH_W_15pf;
        MTIME.TPHL_D_W    := CHIP_TIMES.TPHL_D_W  * KCHL_W_15pf;
        MTIME.TPZH_G_Y    := CHIP_TIMES.TPZH_G_Y  * KCLH_Y_50pf;
        MTIME.TPZL_G_Y    := CHIP_TIMES.TPZL_G_Y  * KCHL_Y_50pf;
        MTIME.TPZH_G_W    := CHIP_TIMES.TPZH_G_W  * KCLH_W_50pf;
        MTIME.TPZL_G_W    := CHIP_TIMES.TPZL_G_W  * KCHL_W_50pf;
        MTIME.TPHZ_G_Y    := CHIP_TIMES.TPHZ_G_Y  * KCHL_Y_5pf;
        MTIME.TPLZ_G_Y    := CHIP_TIMES.TPLZ_G_Y  * KCLH_Y_5pf;
        MTIME.TPHZ_G_W    := CHIP_TIMES.TPHZ_G_W  * KCHL_W_5pf;
        MTIME.TPLZ_G_W    := CHIP_TIMES.TPLZ_G_W  * KCLH_W_5pf;

      when LS_FAMILY => 
        MTIME.TPLH_BS_Y   := CHIP_TIMES.TPLH_BS_Y * KCLH_Y_15pf;
        MTIME.TPHL_BS_Y   := CHIP_TIMES.TPHL_BS_Y * KCHL_Y_15pf;
        MTIME.TPLH_BS_W   := CHIP_TIMES.TPLH_BS_W * KCLH_W_15pf;
        MTIME.TPHL_BS_W   := CHIP_TIMES.TPHL_BS_W * KCHL_W_15pf;
        MTIME.TPLH_D_Y    := CHIP_TIMES.TPLH_D_Y  * KCLH_Y_15pf;
        MTIME.TPHL_D_Y    := CHIP_TIMES.TPHL_D_Y  * KCHL_Y_15pf;
        MTIME.TPLH_D_W    := CHIP_TIMES.TPLH_D_W  * KCLH_W_15pf;
        MTIME.TPHL_D_W    := CHIP_TIMES.TPHL_D_W  * KCHL_W_15pf;
        MTIME.TPZH_G_Y    := CHIP_TIMES.TPZH_G_Y  * KCLH_Y_15pf;
        MTIME.TPZL_G_Y    := CHIP_TIMES.TPZL_G_Y  * KCHL_Y_15pf;
        MTIME.TPZH_G_W    := CHIP_TIMES.TPZH_G_W  * KCLH_W_15pf;
        MTIME.TPZL_G_W    := CHIP_TIMES.TPZL_G_W  * KCHL_W_15pf;
        MTIME.TPHZ_G_Y    := CHIP_TIMES.TPHZ_G_Y  * KCHL_Y_5pf;
        MTIME.TPLZ_G_Y    := CHIP_TIMES.TPLZ_G_Y  * KCLH_Y_5pf;
        MTIME.TPHZ_G_W    := CHIP_TIMES.TPHZ_G_W  * KCHL_W_5pf;
        MTIME.TPLZ_G_W    := CHIP_TIMES.TPLZ_G_W  * KCLH_W_5pf;
    end case;
    



    return(MTIME);

  end BACK_ANNOTATE;

------------------------------------------------------------

end SN54xx251_TIMING;



