-- Filename : SN54xx251.vhd
--
-- Version 2.1
--
--
--                                   NOTICE
--
--      LICENSEE acknowledges that it has read and executed the
--      Institute for Technology Development, VHDL Standard Component
--      Library Software Standard License Agreement.  Licensee may not
--      sublicense, assign or transfer the Software, the Documentation or its
--      rights under the Agreement.
--      
--      The Agreement authorizes Licensee to use at any one time only
--      one copy of the Software (either the original Software or the
--      Archival Copy but never both.)  Licensee may make one (1) copy
--      of the Software (the "Archival Copy") and one (1) copy of the
--      Documentation for archival purposes.  Licensee may not otherwise copy,
--      modify, transfer or disclose in any manner the Software or
--      Documentation.  Licensee shall store the Archival Copy in a secure
--      location to which access shall be restricted.  Licensee shall maintain
--      a written record of the data on which the Archival Copy is created and
--      its location.
--      
--      Licensee expressly recognizes that the Software is the proprietary
--      property of ITD.  Licensee warrants that it will not take any action
--      which would result in the impairment or loss of proprietary rights of
--      ITD in the Software.  Licensee warrants that the Software will
--      not be incorporated into any Licensee product.  Licensee warrants 
--      that it will not decompile, disassemble or reverse engineer the 
--      Software or attempt to do so for other than internal applications.
--      Licensee warrants that it will never divulge to any person without the
--      prior written consent of ITD the Software source code or Software
--      object code.
--      
--      The Software is provided "AS IS" WITHOUT WARRANTY of any kind, either
--      express or implied, including but not limited to implied warranties of
--      merchantability and fitness for a particular purpose.
--      
--      RIGHT TO ACCESS OF THIS SOFTWARE IS DENIED IF THIS NOTICE IS REMOVED.
--      
--
--
--        ------------------------------------------------------------------

library IEEE;                        -- Logic system
  use IEEE.STD_LOGIC_1164.all;       -- Defines logic types, operators, functions

library STD_PACK;                    -- Standard library for all models
  use STD_PACK.MISC_FUNC.all;        -- Miscellaneous functions
  use STD_PACK.TIME_FUNC.all;        -- Timing functions
  use STD_PACK.TTL_TIMING.all;       -- TTL derating coefficients

library SN54xx251;
  use SN54xx251.SN54xx251_TIMING.all;       -- SN54xx251 timing module

entity SN54xx251 is

  generic 
  (
    GEN_PROP_TIMES : Prop_Times ;
    GEN_FAMILY : Tech_Type ;

    TWD_G   : Time;
    TWD_A   : Time;
    TWD_B   : Time;
    TWD_C   : Time;
    TWD_D0  : Time;
    TWD_D1  : Time;
    TWD_D2  : Time;
    TWD_D3  : Time;
    TWD_D4  : Time;
    TWD_D5  : Time;
    TWD_D6  : Time;
    TWD_D7  : Time;

    CL_Y : Real;
    CL_W : Real;

    REF : String
  );

  port 
  (
    Y       : out Std_Ulogic;
    W       : out Std_Ulogic;
    G       : in Std_Ulogic;
    A       : in Std_Ulogic;
    B       : in Std_Ulogic;
    C       : in Std_Ulogic;
    D0      : in Std_Ulogic;
    D1      : in Std_Ulogic;
    D2      : in Std_Ulogic;
    D3      : in Std_Ulogic;
    D4      : in Std_Ulogic;
    D5      : in Std_Ulogic;
    D6      : in Std_Ulogic;
    D7      : in Std_Ulogic
  );

end SN54xx251;

-- ======================================================================

architecture behavioral of SN54xx251 is

  constant CL_OUT : Cload_Out := (
    PIN_Y => CL_Y,
    PIN_W => CL_W
  );

  constant MODEL_TIMES : Prop_Times := 
    BACK_ANNOTATE(GEN_PROP_TIMES,GEN_FAMILY,CL_OUT) ;

  -- Local signal declarations

  signal GIN        : UX01;
  signal SEL        : UX01_Vector(2 downto 0);
  signal DATA       : UX01_Vector(7 downto 0);
  signal YOUT, WOUT : UX01Z;

begin

  -- assign pin values to local signals after wire delay

  GIN     <= transport CONVERT_TO_UX01(G)   after TWD_G; 
  SEL(0)  <= transport CONVERT_TO_UX01(A)   after TWD_A;
  SEL(1)  <= transport CONVERT_TO_UX01(B)   after TWD_B;
  SEL(2)  <= transport CONVERT_TO_UX01(C)   after TWD_C;
  DATA(0) <= transport CONVERT_TO_UX01(D0)  after TWD_D0;
  DATA(1) <= transport CONVERT_TO_UX01(D1)  after TWD_D1;
  DATA(2) <= transport CONVERT_TO_UX01(D2)  after TWD_D2;
  DATA(3) <= transport CONVERT_TO_UX01(D3)  after TWD_D3;
  DATA(4) <= transport CONVERT_TO_UX01(D4)  after TWD_D4;
  DATA(5) <= transport CONVERT_TO_UX01(D5)  after TWD_D5;
  DATA(6) <= transport CONVERT_TO_UX01(D6)  after TWD_D6;
  DATA(7) <= transport CONVERT_TO_UX01(D7)  after TWD_D7;

  -- Assign local values to output pins

  Y <= YOUT;
  W <= WOUT;

  -- Check for invalid inputs on control input ports

  assert not (G'event and ((G = 'U') or (G = 'X') or (G = 'Z')
    or (G = 'W') or (G = '-'))) report "{" & ref & "} " & 
    STD_ULOGIC_TO_CHAR(G) & " state on G input" severity warning;
    
    assert not (A'event and ((A = 'U') or (A = 'X') or (A = 'Z') 
      or (A = 'W') or (G = '-'))) report "{" & ref & "} " & 
      STD_ULOGIC_TO_CHAR(A) & " state on A input" severity warning;
    
    assert not (B'event and ((B = 'U') or (B = 'X') or (B = 'Z') 
      or (B = 'W') or (B = '-'))) report "{" & ref & "} " & 
      STD_ULOGIC_TO_CHAR(B) & " state on B input" severity warning;
    
    assert not (C'event and ((C = 'U') or (C = 'X') or (C = 'Z') 
      or (C = 'W') or (C = '-'))) report "{" & ref & "} " & 
      STD_ULOGIC_TO_CHAR(C) & " state on C input" severity warning;
    
------------------------------------------------------------------------------

MUX_Y : process(GIN,SEL,DATA)

  variable Y_VALUE, W_VALUE : Std_Ulogic;    
begin
 
  case GIN is
    when '1' => -- disabled
      Y_VALUE := 'Z';
      W_VALUE := 'Z';
    when 'X' =>
      Y_VALUE := 'X';
      W_VALUE := 'X';
    when 'U' =>
      Y_VALUE := 'U';
      W_VALUE := 'U';
    when '0' =>  -- device is enabled
      if (UNKNOWN_ON_BUS(SEL)) then -- propagate unknown state
        if (SEL(0) = 'U' or SEL(1) = 'U' or SEL(2) = 'U') then
           Y_VALUE := 'U';
           W_VALUE := 'U';
        else
           Y_VALUE := 'X';
           W_VALUE := 'X';
        end if;
      else
        Y_VALUE :=     DATA(BITVECTOR_TO_INT(SEL)); -- mux the data lines
        W_VALUE := not DATA(BITVECTOR_TO_INT(SEL));
      end if;

  end case; -- GIN
  
  -- assign output after appropriate delay


  if (Y_VALUE = 'X' or Y_VALUE = 'U') then
    YOUT <= Y_VALUE;  -- propagate unknowns instantly
    WOUT <= W_VALUE;
  elsif (SEL'event) then
    YOUT <= transport Y_VALUE after 
        TP_DELAY(Y_VALUE,MODEL_TIMES.TPLH_BS_Y,MODEL_TIMES.TPHL_BS_Y);
    WOUT <= transport W_VALUE after 
        TP_DELAY(W_VALUE,MODEL_TIMES.TPLH_BS_W,MODEL_TIMES.TPHL_BS_W);
  elsif (DATA'event) then
    YOUT <= transport Y_VALUE after 
        TP_DELAY(Y_VALUE,MODEL_TIMES.TPLH_D_Y,MODEL_TIMES.TPHL_D_Y);
    WOUT <= transport W_VALUE after 
        TP_DELAY(W_VALUE,MODEL_TIMES.TPLH_D_W,MODEL_TIMES.TPHL_D_W);
  elsif (GIN'event) then
    case GIN is
      when '0' =>  -- enable mux
        YOUT <= transport Y_VALUE after 
            TP_DELAY(Y_VALUE,MODEL_TIMES.TPZH_G_Y,MODEL_TIMES.TPZL_G_Y);
        WOUT <= transport W_VALUE after 
            TP_DELAY(W_VALUE,MODEL_TIMES.TPZH_G_W,MODEL_TIMES.TPZL_G_W);
      when '1' =>  -- tristate outputs
        YOUT <= transport 'Z' after 
            TP_DELAY(YOUT,MODEL_TIMES.TPHZ_G_Y,MODEL_TIMES.TPLZ_G_Y);
        WOUT <= transport 'Z' after 
            TP_DELAY(WOUT,MODEL_TIMES.TPHZ_G_W,MODEL_TIMES.TPLZ_G_W);
      when others =>
        YOUT <= Y_VALUE;
        WOUT <= W_VALUE;
    end case;
  end if;    
end process;

end BEHAVIORAL;
    
