-- Filename : wgenerator_SN54xx20.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
  use STD.TEXTIO.all;

library WAVES_STANDARD;
  use WAVES_STANDARD.WAVES_SYSTEM.all;
  use WAVES_STANDARD.WAVES_INTERFACE.all;

library SN54xx20;
  use SN54xx20.WAVES_DEVICE.all;
  use SN54xx20.WAVES_OBJECTS.all;
  use SN54xx20.WAVES_FRAMES.all;
  use SN54xx20.WAVES_UTILITIES.all;

package WGENERATOR_SN54xx20 is
  procedure SN54xx20_TEST
  (
    signal CONNECT   : inout Waves_Port_List
  );

  procedure SN54xx20_TEST
  (
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time 
  );

end WGENERATOR_SN54xx20;

-- ======================================================================

package body WGENERATOR_SN54xx20 is

    constant O_PINS : Pinset := NEW_PINSET(( TP_Y1, TP_Y2));
    constant I_PINS : Pinset := ALL_PINS and not O_PINS;
    constant EMPTY_EVENT : Integer := -1;

  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs) are contained within
  --  the procedure, and are analyzed with the VHDL code.  The test vectors
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After 95 ns (95% of period):  Expected outputs set to appropriate values
  --
  --   After 100 ns (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
  procedure SN54xx20_TEST(
    signal CONNECT   : inout Waves_Port_List) is

    constant PERIOD          : Time       := 100 ns;
    constant UNCERTAINTY_PCT : Real       := 0.95;
    constant TDELAY          : Event_Time := ETIME (UNCERTAINTY_PCT * PERIOD);

    constant FSA : Frame_Set_Array := 
      NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, 
                                             TIMED_DELAY, 0, 0, 0);

    variable TD1 : Time_Data := NEW_TIME_DATA ( FSA );
    variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1);

    --        Order of signals:
    --
    --        (Y1, Y2,
    --         A1, B1, C1, D1, A2, B2, C2, D2)
    --

    variable ALL_STATES_PART1 : Wtime_Slice_List (1 to 11) := (
      (  "UU"
     & "UUUUUUUU", T1),

      (  "00"
     & "11111111", T1), -- reset to normal

      (  "UU"
     & "U111UUUU", T1), -- 9 states 

      (  "XU"
     & "X111UUUU", T1),

      (  "1U"
     & "0111UUUU", T1),

      (  "0U"
     & "1111UUUU", T1),

      (  "XU"
     & "Z111UUUU", T1),

      (  "XU"
     & "W111UUUU", T1),

      (  "1U"
     & "L111UUUU", T1),

      (  "0U"
     & "H111UUUU", T1),

      (  "XU"
     & "-111UUUU", T1)
      );



    variable ALL_STATES_PART2 : Wtime_Slice_List (1 to 9) := (
      (  "UU"
     & "UUUUU111", T1), -- 9 states 

      (  "UX"
     & "UUUUX111", T1),

      (  "U1"
     & "UUUU0111", T1),

      (  "U0"
     & "UUUU1111", T1),

      (  "UX"
     & "UUUUZ111", T1),

      (  "UX"
     & "UUUUW111", T1),

      (  "U1"
     & "UUUUL111", T1),

      (  "U0"
     & "UUUUH111", T1),

      (  "UX"
     & "UUUU-111", T1)
      );



    variable TIMING_PARAMS : Wtime_Slice_List (1 to 4) := (
      (  "00"
     & "11111111", T1), -- reset to normal

      (  "00"
     & "11111111", T1), -- timing parameters

      (  "11"
     & "00000000", T1), -- tplh

      (  "00"
     & "11111111", T1) -- tphl
      );



    variable TRUTH_TABLE : Wtime_Slice_List (1 to 10) := (
      (  "00"
     & "11111111", T1), -- reset to normal

      (  "00"
     & "11111111", T1), -- truth table

      (  "1U"
     & "0UUUUUUU", T1),

      (  "1U"
     & "U0UUUUUU", T1),

      (  "1U"
     & "UU0UUUUU", T1),

      (  "1U"
     & "UUU0UUUU", T1),

      (  "U1"
     & "UUUU0UUU", T1),

      (  "U1"
     & "UUUUU0UU", T1),

      (  "U1"
     & "UUUUUU0U", T1),

      (  "U1"
     & "UUUUUUU0", T1)
      );



  begin
    assert FALSE
      report "ALL_STATES_PART1 vectors"
        severity note;
      for I in ALL_STATES_PART1' range loop
        APPLY(CONNECT,ALL_STATES_PART1(I).CODES,ALL_STATES_PART1(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "ALL_STATES_PART2 vectors"
        severity note;
      for I in ALL_STATES_PART2' range loop
        APPLY(CONNECT,ALL_STATES_PART2(I).CODES,ALL_STATES_PART2(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "TIMING_PARAMS vectors"
        severity note;
      for I in TIMING_PARAMS' range loop
        APPLY(CONNECT,TIMING_PARAMS(I).CODES,TIMING_PARAMS(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "TRUTH_TABLE vectors"
        severity note;
      for I in TRUTH_TABLE' range loop
        APPLY(CONNECT,TRUTH_TABLE(I).CODES,TRUTH_TABLE(I).WTIME,ALL_PINS);
      end loop;

    end;

  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs, along with the period
  --  of each slice) are read in from an external file.  The test vectors 
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After <START_TOLERANCE>:  Expected outputs set to appropriate values
  --
  --   After <FS_TIME - END_TOLERANCE>: Get next set of inputs and outputs
  --
  --   After <FS_TIME> (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
    procedure SN54xx20_Test
    (
      signal CONNECT            : inout Waves_Port_List;
      constant INPUT_FILE_NAME  : String;
      constant START_TOLERANCE  : Time;
      constant END_TOLERANCE    : Time ) is

      variable TOLERANCE : Time := START_TOLERANCE + END_TOLERANCE;

      constant FSA : Frame_Set_Array :=
        NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY (OUTPUT_FRAMES, O_PINS);

      constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, TIMED_DELAY, 0, 0, 0);

      variable TD : Time_Data := NEW_TIME_DATA (FSA);

      constant FSA_DONT_CARE : Frame_Set_Array :=
        NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY (DONT_CARE_FRAMES, O_PINS);

      variable TD_DONT_CARE : Time_Data := NEW_TIME_DATA (FSA_DONT_CARE);

      file INPUT_FILE : Text is in INPUT_FILE_NAME;

      -- the file slice must be allocated
      variable SN54xx20_FILE_SLICE : File_Slice := NEW_FILE_SLICE;

    begin
      loop
        READ_FILE_SLICE(INPUT_FILE, SN54xx20_FILE_SLICE);
        exit when SN54xx20_FILE_SLICE.END_OF_FILE;
        if SN54xx20_FILE_SLICE.FS_TIME > TOLERANCE then
          APPLY(CONNECT, SN54xx20_FILE_SLICE.CODES.all, 
                DELAY(START_TOLERANCE), TD_DONT_CARE);
          APPLY(CONNECT, SN54xx20_FILE_SLICE.CODES.all, 
                DELAY(SN54xx20_FILE_SLICE.FS_TIME - TOLERANCE), TD);
          APPLY(CONNECT, SN54xx20_FILE_SLICE.CODES.all, 
                DELAY(END_TOLERANCE), TD_DONT_CARE);
        else
          APPLY(CONNECT, SN54xx20_FILE_SLICE.CODES.all, 
                DELAY(SN54xx20_FILE_SLICE.FS_TIME), TD);
        end if;
      end loop;
    end;
end WGENERATOR_SN54xx20;
