-- Filename : wgenerator_SN54xx175.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
  use STD.TEXTIO.all;

library WAVES_STANDARD;
  use WAVES_STANDARD.WAVES_SYSTEM.all;
  use WAVES_STANDARD.WAVES_INTERFACE.all;

library SN54xx175;
  use SN54xx175.WAVES_DEVICE.all;
  use SN54xx175.WAVES_OBJECTS.all;
  use SN54xx175.WAVES_FRAMES.all;
  use SN54xx175.WAVES_UTILITIES.all;

package WGENERATOR_SN54xx175 is
  procedure SN54xx175_TEST
  (
    signal CONNECT   : inout Waves_Port_List
  );

  procedure SN54xx175_TEST(
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time );

end WGENERATOR_SN54xx175;

-- ======================================================================

package body WGENERATOR_SN54xx175 is

  constant I_PINS : Pinset := NEW_PINSET
  ( 
    (TP_CLK, TP_D1, TP_D2, TP_D3, TP_D4, TP_CLR)
  );

  constant O_PINS      : Pinset  := ALL_PINS and not I_PINS;
  constant EMPTY_EVENT : Integer := -1;

  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs) are contained within
  --  the procedure, and are analyzed with the VHDL code.  The test vectors
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After 95 ns (95% of period):  Expected outputs set to appropriate values
  --
  --   After 100 ns (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
  procedure SN54xx175_TEST(
    signal CONNECT   : inout Waves_Port_List) is

    constant PERIOD          : Time       := 100 ns;
    constant UNCERTAINTY_PCT : Real       := 0.95;
    constant TDELAY          : Event_Time := ETIME (UNCERTAINTY_PCT * PERIOD);

    constant FSA : Frame_Set_Array := 
      NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, TIMED_DELAY, 0, 0, 0);

    variable TD1 : Time_Data := NEW_TIME_DATA ( FSA );
    variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1);

    constant Pulse      : Time       := 1 ns;
    constant PulseDelay : Event_Time := ETIME (UNCERTAINTY_PCT * Pulse);
         
    constant FSA_pulse : FRAME_SET_ARRAY := 
      NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(PulseDelay), O_PINS);
                              
    variable TD_PULSE : TIME_DATA := NEW_TIME_DATA ( FSA_PULSE);
    variable TP  : WAVE_TIMING := ((PULSE, DT_BASIS), TD_PULSE);

    --        Order of signals:
    --
    --        (Q1, QB1, Q2, QB2, Q3, QB3, Q4, QB4,
    --         CLK, D1, D2, D3, D4, CLR)
    --

    variable TRUTH_TABLE : Wtime_Slice_List (1 to 18) := (
      (  "UUUUUUUU"
     & "UUUUUU", T1),

      (  "01010101"
     & "-----0", T1), -- clear

      (  "01010101"
     & "01---1", T1), -- D1 data high

      (  "10XXXXXX"
     & "11---1", T1), -- clock

      (  "10XXXXXX"
     & "00---1", T1), -- D1 data low

      (  "01XXXXXX"
     & "10---1", T1), -- clock

      (  "01XXXXXX"
     & "0-1--1", T1), -- D2 data high

      (  "XX10XXXX"
     & "1-1--1", T1), -- clock

      (  "XX10XXXX"
     & "0-0--1", T1), -- D2 data low

      (  "XX01XXXX"
     & "1-0--1", T1), -- clock

      (  "XX01XXXX"
     & "0--1-1", T1), -- D3 data high

      (  "XXXX10XX"
     & "1--1-1", T1), -- clock

      (  "XXXX10XX"
     & "0--0-1", T1), -- D3 data low

      (  "XXXX01XX"
     & "1--0-1", T1), -- clock

      (  "XXXX01XX"
     & "0---11", T1), -- D4 data high

      (  "XXXXXX10"
     & "1---11", T1), -- clock

      (  "XXXXXX10"
     & "0---01", T1), -- D4 data low

      (  "XXXXXX01"
     & "1---01", T1) -- clock
      );



    variable PULSE_WIDTH_VIOLATIONS : Wtime_Slice_List (1 to 10) := (
      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1), 

      (  "XXXXXXXX"
     & "P11111", T1), -- clock high pulse violation

      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1),

      (  "10101010"
     & "111111", T1), -- clock in data

      (  "XXXXXXXX"
     & "N11111", T1), -- clock low pulse violation

      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1), 

      (  "XXXXXXXX"
     & "01111N", T1) -- clear pulse violation
      );



    variable SETUP_VIOLATIONS : Wtime_Slice_List (1 to 16) := (
      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1), 

      (  "01010101"
     & "001111", TP), -- change D1

      (  "XX101010"
     & "101111", T1), -- D1 setup violation

      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1), --

      (  "01010101"
     & "010111", TP), -- change D2

      (  "10XX1010"
     & "110111", T1), -- D2 setup violation

      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1), --

      (  "01010101"
     & "011011", TP), -- change D3

      (  "1010XX10"
     & "111011", T1), -- D3 setup violation

      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1), --

      (  "01010101"
     & "011101", TP), -- change D4

      (  "101010XX"
     & "111101", T1) -- D4 setup violation
      );



    variable HOLD_VIOLATIONS : Wtime_Slice_List (1 to 16) := (
      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1),

      (  "01010101"
     & "111111", TP), -- clock

      (  "XX101010"
     & "101111", T1), -- D1 hold violation

      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1),

      (  "01010101"
     & "111111", TP), -- clock

      (  "10XX1010"
     & "110111", T1), -- D2 hold violation

      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1),

      (  "01010101"
     & "111111", TP), -- clock

      (  "1010XX10"
     & "111011", T1), -- D3 hold violation

      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1),

      (  "01010101"
     & "111111", TP), -- clock

      (  "101010XX"
     & "111101", T1) -- D4 hold violation
      );



    variable UNKNOWN_ON_INPUTS : Wtime_Slice_List (1 to 6) := (
      (  "01010101"
     & "010100", T1), -- clear

      (  "01010101"
     & "010101", T1),

      (  "10011001"
     & "110101", T1), -- clock data

      (  "XX01XX01"
     & "01010X", T1), -- 'X' on clear

      (  "01010101"
     & "011110", T1), -- clear

      (  "01010101"
     & "011111", T1)
      );



  begin
    assert FALSE
      report "TRUTH_TABLE vectors"
        severity note;
      for I in TRUTH_TABLE' range loop
        APPLY(CONNECT,TRUTH_TABLE(I).CODES,TRUTH_TABLE(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "PULSE_WIDTH_VIOLATIONS vectors"
        severity note;
      for I in PULSE_WIDTH_VIOLATIONS' range loop
        APPLY(CONNECT,PULSE_WIDTH_VIOLATIONS(I).CODES,PULSE_WIDTH_VIOLATIONS(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "SETUP_VIOLATIONS vectors"
        severity note;
      for I in SETUP_VIOLATIONS' range loop
        APPLY(CONNECT,SETUP_VIOLATIONS(I).CODES,SETUP_VIOLATIONS(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "HOLD_VIOLATIONS vectors"
        severity note;
      for I in HOLD_VIOLATIONS' range loop
        APPLY(CONNECT,HOLD_VIOLATIONS(I).CODES,HOLD_VIOLATIONS(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "UNKNOWN_ON_INPUTS vectors"
        severity note;
      for I in UNKNOWN_ON_INPUTS' range loop
        APPLY(CONNECT,UNKNOWN_ON_INPUTS(I).CODES,UNKNOWN_ON_INPUTS(I).WTIME,ALL_PINS);
      end loop;

    end;

  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs, along with the period
  --  of each slice) are read in from an external file.  The test vectors 
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After <START_TOLERANCE>:  Expected outputs set to appropriate values
  --
  --   After <FS_TIME - END_TOLERANCE>: Get next set of inputs and outputs
  --
  --   After <FS_TIME> (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
  procedure SN54xx175_Test(
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time ) is


    variable TOLERANCE : Time := START_TOLERANCE + END_TOLERANCE;

    constant FSA : Frame_Set_Array :=
      NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY (OUTPUT_FRAMES, O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, 
                                             TIMED_DELAY, 0, 0, 0);

    variable TD : Time_Data := NEW_TIME_DATA (FSA);

    constant FSA_DONT_CARE : Frame_Set_Array :=
      NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY (DONT_CARE_FRAMES, O_PINS);

    variable TD_DONT_CARE : Time_Data := NEW_TIME_DATA (FSA_DONT_CARE);

    file INPUT_FILE : Text is in INPUT_FILE_NAME;

    -- the file slice must be allocated
    variable SN54xx175_FILE_SLICE : File_Slice := NEW_FILE_SLICE;

  begin
    loop
      READ_FILE_SLICE(INPUT_FILE, SN54xx175_FILE_SLICE);
      exit when SN54xx175_FILE_SLICE.END_OF_FILE;
      if SN54xx175_FILE_SLICE.FS_TIME > TOLERANCE then
        APPLY(CONNECT, SN54xx175_FILE_SLICE.CODES.all, DELAY(START_TOLERANCE), TD_DONT_CARE);
        APPLY(CONNECT, SN54xx175_FILE_SLICE.CODES.all, DELAY(SN54xx175_FILE_SLICE.FS_TIME - TOLERANCE), TD);
        APPLY(CONNECT, SN54xx175_FILE_SLICE.CODES.all, DELAY(END_TOLERANCE), TD_DONT_CARE);
       else
         APPLY(CONNECT, SN54xx175_FILE_SLICE.CODES.all, DELAY(SN54xx175_FILE_SLICE.FS_TIME), TD);
      end if;
    end loop;
  end;
end WGENERATOR_SN54xx175;
