-- Filename : SN54xx175_timing.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
library STD_PACK;
  use STD_PACK.SIMFLAG.all;    -- Global simulation options
  use STD_PACK.TTL_TIMING.all; -- TTL derating coefficients

package SN54xx175_TIMING is

  type Mode_Type is (UNINIT,ERROR,CLR_FF,LATCH);
  type Speed_Type is (MIN,TYP,MAX);

  type Prop_Times is record
    TPLH_CLEAR_QB : Time;
    TPHL_CLEAR_Q  : Time;
    TPLH_CLOCK_Q  : Time;
    TPHL_CLOCK_Q  : Time;
    TPLH_CLOCK_QB : Time;
    TPHL_CLOCK_QB : Time;
  end record;

  type Cload_Out is record  -- capacitive loading on output pins
    PIN_Q  : Real;
    PIN_QB : Real;
  end record;

  type Delay_Type is array (Tech_Type,Speed_Type) of Prop_Times;

  type Constraint_Type is array (Tech_Type) of Time;

----------- data book times ----------------

  constant DATASHEET_TIMES : Delay_Type :=
  (
   -- STD family
    (  --  minimum times
       ( TPLH_CLEAR_QB => 5.6 ns,
         TPHL_CLEAR_Q  => 8.1 ns,
         TPLH_CLOCK_Q  => 7.0 ns,
         TPHL_CLOCK_Q  => 8.4 ns,
         TPLH_CLOCK_QB => 7.0 ns,
         TPHL_CLOCK_QB => 8.4 ns
       ),

       --  typical times
       ( TPLH_CLEAR_QB => 16.0 ns,
         TPHL_CLEAR_Q  => 23.0 ns,
         TPLH_CLOCK_Q  => 20.0 ns,
         TPHL_CLOCK_Q  => 24.0 ns,
         TPLH_CLOCK_QB => 20.0 ns,
         TPHL_CLOCK_QB => 24.0 ns
       ),
       --  maximum times
       ( TPLH_CLEAR_QB => 25.0 ns,
         TPHL_CLEAR_Q  => 35.0 ns,
         TPLH_CLOCK_Q  => 30.0 ns,
         TPHL_CLOCK_Q  => 35.0 ns,
         TPLH_CLOCK_QB => 30.0 ns,
         TPHL_CLOCK_QB => 35.0 ns
       )
    ),

   -- S family
    (  --  minimum times
       ( TPLH_CLEAR_QB => 3.5 ns,
         TPHL_CLEAR_Q  => 4.6 ns,
         TPLH_CLOCK_Q  => 2.8 ns,
         TPHL_CLOCK_Q  => 4.0 ns,
         TPLH_CLOCK_QB => 2.8 ns,
         TPHL_CLOCK_QB => 4.0 ns
       ),

       --  typical times
       ( TPLH_CLEAR_QB => 10.0 ns,
         TPHL_CLEAR_Q  => 13.0 ns,
         TPLH_CLOCK_Q  =>  8.0 ns,
         TPHL_CLOCK_Q  => 11.5 ns,
         TPLH_CLOCK_QB =>  8.0 ns,
         TPHL_CLOCK_QB => 11.5 ns
       ),
       --  maximum times
       ( TPLH_CLEAR_QB => 15.0 ns,
         TPHL_CLEAR_Q  => 22.0 ns,
         TPLH_CLOCK_Q  => 12.0 ns,
         TPHL_CLOCK_Q  => 17.0 ns,
         TPLH_CLOCK_QB => 12.0 ns,
         TPHL_CLOCK_QB => 17.0 ns
       )
    ),

   -- LS family
    (  --  minimum times
       ( TPLH_CLEAR_QB => 7.0 ns,
         TPHL_CLEAR_Q  => 7.0 ns,
         TPLH_CLOCK_Q  => 4.6 ns,
         TPHL_CLOCK_Q  => 5.6 ns,
         TPLH_CLOCK_QB => 4.6 ns,
         TPHL_CLOCK_QB => 5.6 ns
       ),

       --  typical times
       ( TPLH_CLEAR_QB => 20.0 ns,
         TPHL_CLEAR_Q  => 20.0 ns,
         TPLH_CLOCK_Q  => 13.0 ns,
         TPHL_CLOCK_Q  => 16.0 ns,
         TPLH_CLOCK_QB => 13.0 ns,
         TPHL_CLOCK_QB => 16.0 ns
       ),
       --  maximum times
       ( TPLH_CLEAR_QB => 30.0 ns,
         TPHL_CLEAR_Q  => 30.0 ns,
         TPLH_CLOCK_Q  => 25.0 ns,
         TPHL_CLOCK_Q  => 25.0 ns,
         TPLH_CLOCK_QB => 25.0 ns,
         TPHL_CLOCK_QB => 25.0 ns
       )
    )
  );

--                                         STD      S        LS
  constant TW_CLK   : Constraint_Type := (20.0 ns,  7.0 ns, 20.0 ns);
  constant TW_CLR   : Constraint_Type := (20.0 ns, 10.0 ns, 20.0 ns);
  constant TSU_DATA : Constraint_Type := (20.0 ns,  5.0 ns, 20.0 ns);
  constant TSU_CLR  : Constraint_Type := (25.0 ns,  5.0 ns, 25.0 ns);
  constant TH       : Constraint_Type := ( 5.0 ns,  3.0 ns, 5.0 ns);

  ----------------------function declarations--------------------

  function CALC_CURVE ( X :  Real; COEFFS :Four_Coeffs ) return Real;

  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times;

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times;

end SN54xx175_TIMING;

-- ======================================================================

package body SN54xx175_TIMING is

-- ------------------------------------------------------------
  function CALC_CURVE ( X : Real; COEFFS : Four_Coeffs ) return Real is

    variable O : Real;

  begin
    O := COEFFS(4)*(X)**3 + COEFFS(3)*(X)**2 + COEFFS(2)*(X) + COEFFS(1);
    return O;
  end CALC_CURVE;

-- ------------------------------------------------------------
  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times is

    variable KVLH,KVHL       : Real;   -- voltage derating factors
    variable KTLH,KTHL       : Real;   -- temperature derating factors

    variable DERATINGLH      : Real;   -- composite derating factors
    variable DERATINGHL      : Real;

    variable MTIME           : Prop_Times;
  
  begin

    -- Check for SIMFLAG options out-of-range 

    assert not (GEN_TA < -55.0 or GEN_TA > 125.0) report 
      "Temperature out of range" severity error;
    assert not (GEN_VCC < 4.5 or GEN_VCC > 5.5 ) report 
      "Voltage out of range" severity error;
    assert not (GEN_DFACTOR < 0.0 ) report 
       "Negative GEN_DFACTOR not allowed" severity error;

    -- calculate derating factors

    KTLH := CALC_CURVE (GEN_TA,   TLH_COEFFICIENT(GEN_FAMILY));
    KTHL := CALC_CURVE (GEN_TA,   THL_COEFFICIENT(GEN_FAMILY));
    KVLH := CALC_CURVE (GEN_VCC,  VLH_COEFFICIENT(GEN_FAMILY));
    KVHL := CALC_CURVE (GEN_VCC,  VHL_COEFFICIENT(GEN_FAMILY));

    -- select delay values

    case GEN_TIME_MODE is
      when MINIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MIN);
      when TYPICAL        => MTIME := DATASHEET_TIMES(GEN_FAMILY,TYP);
      when MAXIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MAX);
    end case;

    -- composite derating factors

    DERATINGLH := KVLH * KTLH * (GEN_DFACTOR / 100.0);
    DERATINGHL := KVHL * KTHL * (GEN_DFACTOR / 100.0);

    -- apply derating factors to propagation delays

    MTIME.TPLH_CLEAR_QB := MTIME.TPLH_CLEAR_QB * DERATINGLH;
    MTIME.TPHL_CLEAR_Q  := MTIME.TPHL_CLEAR_Q  * DERATINGHL;
    MTIME.TPLH_CLOCK_Q  := MTIME.TPLH_CLOCK_Q  * DERATINGLH;
    MTIME.TPHL_CLOCK_Q  := MTIME.TPHL_CLOCK_Q  * DERATINGHL;
    MTIME.TPLH_CLOCK_QB := MTIME.TPLH_CLOCK_QB * DERATINGLH;
    MTIME.TPHL_CLOCK_QB := MTIME.TPHL_CLOCK_QB * DERATINGHL;

    return(MTIME);

  end GET_TIMING;

-- ----------------------------------------------------------
  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times is

    variable KCLH  : Cload_Out;   -- load capacitance derating factors
    variable KCHL  : Cload_Out;

    variable MTIME : Prop_Times;

  begin

    KCLH.PIN_Q  := CALC_CURVE (GEN_CLOADS.PIN_Q,CLH_COEFFICIENT(GEN_FAMILY));
    KCLH.PIN_QB := CALC_CURVE (GEN_CLOADS.PIN_QB,CLH_COEFFICIENT(GEN_FAMILY));

    KCHL.PIN_Q  := CALC_CURVE (GEN_CLOADS.PIN_Q,CHL_COEFFICIENT(GEN_FAMILY));
    KCHL.PIN_QB := CALC_CURVE (GEN_CLOADS.PIN_QB,CHL_COEFFICIENT(GEN_FAMILY));

    MTIME.TPLH_CLEAR_QB := CHIP_TIMES.TPLH_CLEAR_QB * KCLH.PIN_QB;
    MTIME.TPHL_CLEAR_Q  := CHIP_TIMES.TPHL_CLEAR_Q  * KCHL.PIN_Q;
    MTIME.TPLH_CLOCK_Q  := CHIP_TIMES.TPLH_CLOCK_Q  * KCLH.PIN_Q;
    MTIME.TPHL_CLOCK_Q  := CHIP_TIMES.TPHL_CLOCK_Q  * KCHL.PIN_Q;
    MTIME.TPLH_CLOCK_QB := CHIP_TIMES.TPLH_CLOCK_QB * KCLH.PIN_QB;
    MTIME.TPHL_CLOCK_QB := CHIP_TIMES.TPHL_CLOCK_QB * KCHL.PIN_QB;

  return(MTIME);

end BACK_ANNOTATE;

------------------------------------------------------------

end SN54xx175_TIMING;
