-- Filename : SN54xx175.vhd
--
-- Version 2.1
--
--
--                                   NOTICE
--
--      LICENSEE acknowledges that it has read and executed the
--      Institute for Technology Development, VHDL Standard Component
--      Library Software Standard License Agreement.  Licensee may not
--      sublicense, assign or transfer the Software, the Documentation or its
--      rights under the Agreement.
--      
--      The Agreement authorizes Licensee to use at any one time only
--      one copy of the Software (either the original Software or the
--      Archival Copy but never both.)  Licensee may make one (1) copy
--      of the Software (the "Archival Copy") and one (1) copy of the
--      Documentation for archival purposes.  Licensee may not otherwise copy,
--      modify, transfer or disclose in any manner the Software or
--      Documentation.  Licensee shall store the Archival Copy in a secure
--      location to which access shall be restricted.  Licensee shall maintain
--      a written record of the data on which the Archival Copy is created and
--      its location.
--      
--      Licensee expressly recognizes that the Software is the proprietary
--      property of ITD.  Licensee warrants that it will not take any action
--      which would result in the impairment or loss of proprietary rights of
--      ITD in the Software.  Licensee warrants that the Software will
--      not be incorporated into any Licensee product.  Licensee warrants 
--      that it will not decompile, disassemble or reverse engineer the 
--      Software or attempt to do so for other than internal applications.
--      Licensee warrants that it will never divulge to any person without the
--      prior written consent of ITD the Software source code or Software
--      object code.
--      
--      The Software is provided "AS IS" WITHOUT WARRANTY of any kind, either
--      express or implied, including but not limited to implied warranties of
--      merchantability and fitness for a particular purpose.
--      
--      RIGHT TO ACCESS OF THIS SOFTWARE IS DENIED IF THIS NOTICE IS REMOVED.
--      
-- 
--
--        ------------------------------------------------------------------

  use STD.TEXTIO.all;

library IEEE;                        -- Logic system
  use IEEE.STD_LOGIC_1164.all;    -- Defines logic types, operators, functions

library STD_PACK;                   -- Standard library for all models
  use STD_PACK.MISC_FUNC.all;       -- Miscellaneous functions package
  use STD_PACK.TIME_FUNC.all;       -- Timing functions package
  use STD_PACK.TTL_TIMING.all;      -- TTL derating coefficients

library SN54xx175;
  use SN54xx175.SN54xx175_TIMING.all;    -- timing module

entity SN54xx175 is
  generic                         -- Recommended operating conditions
  (
    GEN_PROP_TIMES  : Prop_Times ;
    GEN_XGEN        : Boolean ;
    GEN_TIMING_MESG : Boolean ;
    GEN_FAMILY      : Tech_Type ;

    GEN_TW_CLK     : Time ;
    GEN_TW_CLR     : Time ;
    GEN_TSU_DATA   : Time ;
    GEN_TSU_CLR    : Time ;
    GEN_TH         : Time ;

    TWD_CLK : Time ;
    TWD_D   : Time ;
    TWD_CLR : Time ;

    CL_Q    : Real ;
    CL_QB   : Real ;

    REF     : String
  );

  port
  (
    Q         : out Std_Ulogic;
    QB        : out Std_Ulogic;
    CLK       : in Std_Ulogic;
    D         : in Std_Ulogic;
    CLR       : in Std_Ulogic
   );

end SN54xx175;

-- ======================================================================

architecture BEHAVIORAL of SN54xx175 is

  constant CL_OUT : Cload_Out := (
    PIN_Q  => CL_Q,
    PIN_QB => CL_QB
  );

  constant MODEL_TIMES : Prop_Times := 
    BACK_ANNOTATE(GEN_PROP_TIMES,GEN_FAMILY,CL_OUT) ;

  -- Local signal declarations

  signal CLOCK, CLEAR, DATA : UX01  := 'U';
  signal QOUT, QBOUT        : UX01Z := 'U';

  signal VIOLATION    : Resolved_Boolean := FALSE;
  signal CLEAR_last   : Time := 0 ns;
  signal CLOCK_last   : Time := 0 ns;

begin

  -- assign pin values to local signals after wire delay

  CLOCK    <= transport CONVERT_TO_UX01(CLK) after TWD_CLK;
  DATA     <= transport CONVERT_TO_UX01(D)   after TWD_D;
  CLEAR    <= transport CONVERT_TO_UX01(CLR) after TWD_CLR;

  -- Assign local values to output pins

  Q  <= QOUT;
  QB <= QBOUT;

  -- Check for invalid states on control inputs

  assert not (CLK'event and BIT_UNKNOWN(CLK))
    report "{" & REF & "} " & STD_ULOGIC_TO_CHAR(CLK) &
    " state on CLK input" severity warning;

  assert not (CLR'event and BIT_UNKNOWN(CLR))
    report "{" & REF & "} " & STD_ULOGIC_TO_CHAR(CLR) &
    " state on CLR input" severity warning;

---------------------------------------------------------------------------------  

  LAST_CLOCK : process (CLOCK)
  begin
    CLOCK_LAST  <= NOW;
  end process LAST_CLOCK;

---------------------------------------------------------------------------------

  LAST_CLEAR : process (CLEAR)
  begin
    CLEAR_LAST  <= NOW;
  end process LAST_CLEAR;

-------------------------------------------------------------------------

CLK_PULSE_CHECK : process
 
  variable CLK_VIOLATE : Boolean; 
  variable TIMESTRING  : Line;

begin

  wait on CLOCK until ( (CLEAR = '1') and (RISING_EDGE(CLOCK) or FALLING_EDGE(CLOCK)));

  if CLOCK = '0' then  -- clock high pulse check
    CLK_VIOLATE := (NOW - CLOCK_LAST) < GEN_TW_CLK;
    write (TIMESTRING, NOW);
    assert not (GEN_TIMING_MESG and CLK_VIOLATE)
      report "{" & REF & "} pulse check warning: CLK - HIGH at time "
      & TIMESTRING.all severity warning;
    deallocate (TIMESTRING);

  else  -- clock low pulse check
    CLK_VIOLATE :=  (NOW - CLOCK_LAST) < GEN_TW_CLK;
    write (TIMESTRING, NOW);
    assert not (GEN_TIMING_MESG and CLK_VIOLATE)
      report "{" & REF & "} pulse check warning: CLK - LOW at time "
      & TIMESTRING.all severity warning;
    deallocate (TIMESTRING);
  end if;

  VIOLATION <= CLK_VIOLATE and GEN_XGEN;
  wait for 0 ns;
  VIOLATION <= FALSE;

end process CLK_PULSE_CHECK;

-------------------------------------------------------------------------

CLR_PULSE_CHECK : process

  variable CLR_VIOLATE : Boolean; 
  variable TIMESTRING  : Line;

begin

  wait until (RISING_EDGE(CLEAR));

  CLR_VIOLATE := (NOW - CLEAR_LAST) < GEN_TW_CLR;
  write (TIMESTRING, NOW);
  assert not (GEN_TIMING_MESG and CLR_VIOLATE)
    report "{" & REF & "} pulse check warning: CLR - LOW at time"
    & TIMESTRING.all severity warning;
  deallocate (TIMESTRING);

  VIOLATION <= CLR_VIOLATE and GEN_XGEN;
  wait for 0 ns;
  VIOLATION <= FALSE;

end process CLR_PULSE_CHECK;

-------------------------------------------------------------------------

SETUP_CHECKS : process

  variable SETUP_DATA_VIOLATE  : Boolean; 
  variable SETUP_CLEAR_VIOLATE  : Boolean; 
  variable TIMESTRING          : Line;

begin

  wait on CLOCK until ((CLEAR = '1') and RISING_EDGE(CLOCK));

  write (TIMESTRING, NOW);

  SETUP_DATA_VIOLATE := not DATA'stable(GEN_TSU_DATA);
  assert not (GEN_TIMING_MESG and SETUP_DATA_VIOLATE)
    report "{" & REF & "} setup check warning: D at time"
    & TIMESTRING.all severity warning;

  SETUP_CLEAR_VIOLATE := not CLEAR'stable(GEN_TSU_CLR);
  assert not (GEN_TIMING_MESG and SETUP_CLEAR_VIOLATE)
    report "{" & REF & "} SETUP check warning: CLR at time "
    & TIMESTRING.all severity warning;

  deallocate (TIMESTRING);

  VIOLATION  <= GEN_XGEN and (SETUP_DATA_VIOLATE or SETUP_CLEAR_VIOLATE);
  wait for 0 ns;
  VIOLATION <= FALSE;

end process SETUP_CHECKS;

-------------------------------------------------------------------------

D_HOLD_CHECK : process

  variable HOLD_DATA_VIOLATE : Boolean;
  variable TIMESTRING        : Line;

begin

  wait on DATA until ( (CLEAR = '1') and (CLOCK'last_value = '0') 
    and (CLOCK = '1'));

  HOLD_DATA_VIOLATE := not CLOCK'stable (GEN_TH);
  write (TIMESTRING, NOW);
  assert not (GEN_TIMING_MESG and HOLD_DATA_VIOLATE)
    report "{" & REF & "} hold check warning: D at time "
    & TIMESTRING.all severity warning;

  deallocate (TIMESTRING);

  VIOLATION <= HOLD_DATA_VIOLATE and GEN_XGEN;
  wait for 0 ns;
  VIOLATION <= FALSE;

end process D_HOLD_CHECK;

-------------------------------------------------------------------------

DFF : process ( CLEAR, CLOCK, VIOLATION)

  variable FFMODE : Mode_Type;

begin

  FFMODE := UNINIT;    -- powerup unitialized
  if (CLEAR = 'U' or CLOCK = 'U' or DATA = 'U') then
    FFMODE := UNINIT;
  elsif (VIOLATION) then  -- a timing violation has occurred
    FFMODE := ERROR;
  elsif (CLEAR = '0') then
    FFMODE := CLR_FF;
  elsif (CLEAR = '1') then
   FFMODE := LATCH;
  elsif (CLEAR = 'X' and QOUT /= '0') then
    FFMODE := ERROR;
  end if;
      
  case FFMODE is
    when UNINIT => -- flip-flop is uninitialized
      QOUT  <= 'U';
      QBOUT <= 'U';

    when ERROR =>  -- a timing violation occurred
      QOUT  <= 'X';
      QBOUT <= 'X';
  
    when CLR_FF => 
      QOUT  <= transport '0' after MODEL_TIMES.TPHL_CLEAR_Q;
      QBOUT <= transport '1' after MODEL_TIMES.TPLH_CLEAR_QB;

    when LATCH =>
      if (RISING_EDGE(CLOCK)) then
        case DATA is 
          when '0' =>
            QOUT  <= transport '0' after MODEL_TIMES.TPHL_CLOCK_Q;
            QBOUT <= transport '1' after MODEL_TIMES.TPLH_CLOCK_QB;
          when '1' =>
            QOUT  <= transport '1' after MODEL_TIMES.TPLH_CLOCK_Q;
            QBOUT <= transport '0' after MODEL_TIMES.TPHL_CLOCK_QB;
          when 'U' =>
            QOUT  <= 'U';
            QBOUT <= 'U';
          when 'X' =>
            QOUT  <= 'X';
            QBOUT <= 'X';
        end case;  -- DATA
          
      elsif (CLOCK = 'X' and DATA /= QOUT) then
        QOUT <= 'X';
        QBOUT <= 'X';
      end if;

    end case; -- MODE
  
end process DFF;

-------------------------------------------------------------------------
    
end BEHAVIORAL;
    
