-- Filename : wgenerator_SN54xx161A.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
  use STD.TEXTIO.all;

library WAVES_STANDARD;
  use WAVES_STANDARD.WAVES_SYSTEM.all;
  use WAVES_STANDARD.WAVES_INTERFACE.all;

library SN54xx161A;
  use SN54xx161A.WAVES_DEVICE.all;
  use SN54xx161A.WAVES_OBJECTS.all;
  use SN54xx161A.WAVES_FRAMES.all;
  use SN54xx161A.WAVES_UTILITIES.all;

package WGENERATOR_SN54xx161A is
  procedure SN54xx161A_TEST
  (
    signal CONNECT   : inout Waves_Port_List
  );

  procedure SN54xx161A_TEST
  (
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time 
  );

end WGENERATOR_SN54xx161A;

-- ======================================================================

package body WGENERATOR_SN54xx161A is

  constant O_PINS : Pinset := NEW_PINSET
  (
    (TP_QA, TP_QB, TP_QC, TP_QD, TP_RC)
  );

  constant I_PINS      : Pinset  := ALL_PINS and not O_PINS;
  constant EMPTY_EVENT : Integer := -1;

  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs) are contained within
  --  the procedure, and are analyzed with the VHDL code.  The test vectors
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After 95 ns (95% of period):  Expected outputs set to appropriate values
  --
  --   After 100 ns (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
  procedure SN54xx161A_TEST
  (
    signal CONNECT   : inout Waves_Port_List) is

    constant PERIOD          : Time       := 100 ns;
    constant UNCERTAINTY_PCT : Real       := 0.95;
    constant TDELAY          : Event_Time := ETIME (UNCERTAINTY_PCT * PERIOD);

    constant FSA : Frame_Set_Array := 
      NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS);

    constant DT_BASIS : Delay_Time_Basis :=(FALSE, 0 ns, 0 ns, 
                                            TIMED_DELAY, 0, 0, 0);

    variable TD1 : Time_Data  := NEW_TIME_DATA ( FSA );
    variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1);

    constant PULSE      : Time       := 1 ns;
    constant PULSEDELAY : Event_Time := ETIME (UNCERTAINTY_PCT * PULSE);
         
    constant FSA_PULSE : FRAME_SET_ARRAY := 
      NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(PULSEDELAY), O_PINS);
                              
    variable TD_PULSE : TIME_DATA   := NEW_TIME_DATA ( FSA_PULSE);
    variable TP       : WAVE_TIMING := ((PULSE, DT_BASIS), TD_PULSE);
                                      

    --        Order of signals:
    --
    --        (QA, QB, QC, QD, RC,
    --         CLK, A, B, C, D, ENP, ENT, LOAD, CLR)
    --

    variable TRUTH_TABLE : Wtime_Slice_List (1 to 27) := (
      (  "UUUUU"
     & "UUUUUUUUU", T1), -- initial

      (  "00000"
     & "000110010", T1), -- clear

      (  "00000"
     & "000110001", T1), -- load 12 into counter

      (  "00110"
     & "100110001", T1), -- clock

      (  "00110"
     & "000111111", T1), -- starts counting

      (  "10110"
     & "100111111", T1), -- 13 

      (  "10110"
     & "000111111", T1),

      (  "01110"
     & "100111111", T1), -- 14

      (  "01110"
     & "000111111", T1),

      (  "11111"
     & "100111111", T1), -- 15 (RCO high)

      (  "11111"
     & "000111111", T1),

      (  "11110"
     & "000111011", T1), -- negate ENT(tphl)

      (  "11111"
     & "000111111", T1), -- assert ENT(tplh)

      (  "00000"
     & "100111111", T1), -- 0

      (  "00000"
     & "000111111", T1),

      (  "10000"
     & "100111111", T1), -- 1

      (  "10000"
     & "000111111", T1),

      (  "01000"
     & "100111111", T1), -- 2

      (  "01000"
     & "000110111", T1), -- negate ENP

      (  "01000"
     & "100110111", T1), -- stops counting

      (  "01000"
     & "000111111", T1), -- assert ENP

      (  "11000"
     & "100111111", T1), -- 3

      (  "11000"
     & "000111011", T1), -- negate ENT

      (  "11000"
     & "100111011", T1), -- stops counting

      (  "11000"
     & "000111111", T1), -- assert ENT

      (  "00100"
     & "100111111", T1), -- 4

      (  "00100"
     & "000111111", T1)
      );



    variable PULSE_WIDTH_VIOLATIONS : Wtime_Slice_List (1 to 11) := (
      (  "00000"
     & "011110010", T1), -- clear

      (  "00000"
     & "011110011", T1),

      (  "XXXXX"
     & "P11110011", T1), -- TW_CLK_HI violation

      (  "00000"
     & "111110010", T1), -- clear

      (  "00000"
     & "111110011", T1),

      (  "XXXXX"
     & "N11110011", T1), -- TW_CLK_LO violation

      (  "00000"
     & "111110010", T1), -- clear

      (  "00000"
     & "111110011", T1),

      (  "XXXXX"
     & "11111001N", T1), -- TW_CLR violation

      (  "00000"
     & "111110010", T1), -- clear

      (  "00000"
     & "111110011", T1) 
      );



    variable SETUP_VIOLATIONS : Wtime_Slice_List (1 to 28) := (
      (  "00000"
     & "000000000", T1), -- clear, assert LOAD

      (  "00000"
     & "000000001", T1),

      (  "00000"
     & "010000001", TP), -- change A data

      (  "XXXXX"
     & "110000001", T1), -- A setup violation

      (  "00000"
     & "000000000", T1), -- clear

      (  "00000"
     & "000000001", T1),

      (  "00000"
     & "001000001", TP), -- change B data

      (  "XXXXX"
     & "101000001", T1), -- B setup violation

      (  "00000"
     & "000000000", T1), -- clear

      (  "00000"
     & "000000001", T1),

      (  "00000"
     & "000100001", TP), -- change C data

      (  "XXXXX"
     & "100100001", T1), -- C setup violation

      (  "00000"
     & "000000000", T1), -- clear

      (  "00000"
     & "000000001", T1),

      (  "00000"
     & "000010001", TP), -- change D data

      (  "XXXXX"
     & "100010001", T1), -- D setup violation

      (  "00000"
     & "000000000", T1), -- clear

      (  "00000"
     & "000000011", T1), -- negate LOAD

      (  "00000"
     & "000001011", TP), -- assert ENP

      (  "XXXXX"
     & "100001011", T1), -- ENP setup violation

      (  "00000"
     & "000000010", T1), -- clear

      (  "00000"
     & "000000011", T1),

      (  "00000"
     & "000000111", TP), -- assert ENT

      (  "XXXXX"
     & "100000111", T1), -- ENT setup violation

      (  "00000"
     & "000000010", T1), -- clear 

      (  "00000"
     & "000000011", T1),

      (  "00000"
     & "000000001", TP), -- assert LOAD

      (  "XXXXX"
     & "100000001", T1) -- LOAD setup violation
      );



    variable HOLD_VIOLATIONS : Wtime_Slice_List (1 to 24) := (
      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "000001101", T1), -- assert LOAD

      (  "00000"
     & "100001101", TP), -- clock

      (  "XXXXX"
     & "110001101", T1), -- A hold violation

      (  "00000"
     & "000001100", T1), -- clear

      (  "00000"
     & "000001101", T1),

      (  "00000"
     & "100001101", TP), -- clock

      (  "XXXXX"
     & "101001101", T1), -- B hold violation

      (  "00000"
     & "000001100", T1), -- clear

      (  "00000"
     & "000001101", T1),

      (  "00000"
     & "100001101", TP), -- clock

      (  "XXXXX"
     & "100101101", T1), -- C hold violation

      (  "00000"
     & "000001100", T1), -- clear

      (  "00000"
     & "000001101", T1),

      (  "00000"
     & "100001101", TP), -- clock

      (  "XXXXX"
     & "100011101", T1), -- D hold violation

      (  "00000"
     & "000001110", T1), -- clear, negate LOAD

      (  "00000"
     & "000001111", T1),

      (  "00000"
     & "100001111", TP), -- clock

      (  "XXXXX"
     & "100000111", T1), -- ENP hold violation

      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "000001111", T1),

      (  "00000"
     & "100001111", TP), -- clock

      (  "XXXXX"
     & "100001011", T1) -- ENT hold violation
      );



    variable ALL_STATES_CLOCK : Wtime_Slice_List (1 to 18) := (
      (  "00000"
     & "000001110", T1), -- clear

      (  "UUUUU"
     & "U00001111", T1), -- U

      (  "00000"
     & "000001110", T1), -- clear

      (  "X0000"
     & "X00001111", T1), -- X

      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "000001111", T1), -- 0

      (  "00000"
     & "000001110", T1), -- clear

      (  "10000"
     & "100001111", T1), -- 1

      (  "00000"
     & "000001110", T1), -- clear

      (  "X0000"
     & "Z00001111", T1), -- Z

      (  "00000"
     & "000001110", T1), -- clear

      (  "X0000"
     & "W00001111", T1), -- W

      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "L00001111", T1), -- L

      (  "00000"
     & "000001110", T1), -- clear

      (  "10000"
     & "H00001111", T1), -- H

      (  "00000"
     & "000001110", T1), -- clear

      (  "X0000"
     & "-00001111", T1) -- -
      );



    variable ALL_STATES_ENABLE : Wtime_Slice_List (1 to 26) := (
      (  "00000"
     & "000001110", T1), -- clear

      (  "UUUUU"
     & "00000U111", T1), -- U

      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "00000X111", T1), -- X

      (  "X0000"
     & "10000X111", T1), -- clock

      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "000000111", T1), -- 0

      (  "00000"
     & "100000111", T1), -- clock

      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "000001111", T1), -- 1

      (  "10000"
     & "100001111", T1), -- clock

      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "00000Z111", T1), -- Z

      (  "X0000"
     & "10000Z111", T1), -- clock

      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "00000W111", T1), -- W

      (  "X0000"
     & "10000W111", T1), -- clock

      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "00000L111", T1), -- L

      (  "00000"
     & "10000L111", T1), -- clock

      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "00000H111", T1), -- H

      (  "10000"
     & "10000H111", T1), -- clock

      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "00000-111", T1), -- -

      (  "X0000"
     & "10000-111", T1) -- clock
      );



    variable ALL_STATES_LOAD : Wtime_Slice_List (1 to 23) := (
      (  "00000"
     & "001011110", T1), -- clear

      (  "UUUUU"
     & "0010111U1", T1), -- U

      (  "00000"
     & "001011110", T1), -- clear

      (  "00000"
     & "0010111X1", T1), -- X

      (  "0X0X0"
     & "1010111X1", T1), -- clock

      (  "00000"
     & "001011110", T1), -- clear

      (  "00000"
     & "001011101", T1), -- 0 (load data)

      (  "01010"
     & "101011101", T1), -- clock

      (  "00000"
     & "001011110", T1), -- clear

      (  "00000"
     & "001011111", T1), -- 1

      (  "10000"
     & "101011111", T1), -- clock

      (  "00000"
     & "001011110", T1), -- clear

      (  "00000"
     & "0010111Z1", T1), -- Z

      (  "0X0X0"
     & "1010111Z1", T1), -- clock

      (  "00000"
     & "001011110", T1), -- clear

      (  "00000"
     & "0010111L1", T1), -- L (load data)

      (  "01010"
     & "1010111L1", T1), -- clock

      (  "00000"
     & "001011110", T1), -- clear

      (  "00000"
     & "0010111H1", T1), -- H

      (  "10000"
     & "1010111H1", T1), -- clock

      (  "00000"
     & "001011110", T1), -- clear

      (  "00000"
     & "0010111-1", T1), -- -

      (  "0X0X0"
     & "1010111-1", T1) -- clock
      );



    variable ILLEGAL_STATES_ON_CLEAR  : Wtime_Slice_List (1 to 6) := (
      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "00000111X", T1), -- clear = X, no effect

      (  "00000"
     & "011111111", T1), 

      (  "00000"
     & "011111101", T1), -- load 15

      (  "11111"
     & "111111101", T1), -- clock

      (  "XXXXX"
     & "01111111X", T1) -- clear = X, outputs = X
      );



    variable LOAD_ILLEGAL_DATA : Wtime_Slice_List (1 to 13) := (
      (  "00000"
     & "000001110", T1), -- clear

      (  "00000"
     & "0X0001111", T1), -- unknown on A

      (  "00000"
     & "0X0001101", T1), -- load

      (  "X0000"
     & "1X0001101", T1), -- clock

      (  "X0000"
     & "00X001111", T1), -- unknown on B

      (  "X0000"
     & "00X001101", T1), -- load

      (  "0X000"
     & "10X001101", T1), -- clock

      (  "0X000"
     & "000X01111", T1), -- unknown on C

      (  "0X000"
     & "000X01101", T1), -- load

      (  "00X00"
     & "100X01101", T1), -- clock

      (  "00X00"
     & "0000X1111", T1), -- unknown on D

      (  "00X00"
     & "0000X1101", T1), -- load

      (  "000X0"
     & "1000X1101", T1) -- clock
      );



    variable UNKNOWN_ON_CLOCK : Wtime_Slice_List (1 to 11) := (
      (  "00000"
     & "000001110", T1), -- clear

      (  "X0000"
     & "X00001111", T1), 

      (  "X0000"
     & "100001111", T1), 

      (  "X0000"
     & "000001111", T1), 

      (  "XX000"
     & "100001111", T1), 

      (  "XX000"
     & "000001111", T1), 

      (  "XXX00"
     & "100001111", T1), 

      (  "XXX00"
     & "000001111", T1), 

      (  "XXXXX"
     & "100001111", T1), 

      (  "XXXXX"
     & "000001111", T1), 

      (  "XXXXX"
     & "100001111", T1) 
      );



  begin
    assert FALSE
      report "TRUTH_TABLE vectors"
        severity note;
      for I in TRUTH_TABLE' range loop
        APPLY(CONNECT,TRUTH_TABLE(I).CODES,TRUTH_TABLE(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "PULSE_WIDTH_VIOLATIONS vectors"
        severity note;
      for I in PULSE_WIDTH_VIOLATIONS' range loop
        APPLY(CONNECT,PULSE_WIDTH_VIOLATIONS(I).CODES,PULSE_WIDTH_VIOLATIONS(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "SETUP_VIOLATIONS vectors"
        severity note;
      for I in SETUP_VIOLATIONS' range loop
        APPLY(CONNECT,SETUP_VIOLATIONS(I).CODES,SETUP_VIOLATIONS(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "HOLD_VIOLATIONS vectors"
        severity note;
      for I in HOLD_VIOLATIONS' range loop
        APPLY(CONNECT,HOLD_VIOLATIONS(I).CODES,HOLD_VIOLATIONS(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "ALL_STATES_CLOCK vectors"
        severity note;
      for I in ALL_STATES_CLOCK' range loop
        APPLY(CONNECT,ALL_STATES_CLOCK(I).CODES,ALL_STATES_CLOCK(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "ALL_STATES_ENABLE vectors"
        severity note;
      for I in ALL_STATES_ENABLE' range loop
        APPLY(CONNECT,ALL_STATES_ENABLE(I).CODES,ALL_STATES_ENABLE(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "ALL_STATES_LOAD vectors"
        severity note;
      for I in ALL_STATES_LOAD' range loop
        APPLY(CONNECT,ALL_STATES_LOAD(I).CODES,ALL_STATES_LOAD(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "ILLEGAL_STATES_ON_CLEAR  vectors"
        severity note;
      for I in ILLEGAL_STATES_ON_CLEAR ' range loop
        APPLY(CONNECT,ILLEGAL_STATES_ON_CLEAR (I).CODES,ILLEGAL_STATES_ON_CLEAR (I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "LOAD_ILLEGAL_DATA vectors"
        severity note;
      for I in LOAD_ILLEGAL_DATA' range loop
        APPLY(CONNECT,LOAD_ILLEGAL_DATA(I).CODES,LOAD_ILLEGAL_DATA(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "UNKNOWN_ON_CLOCK vectors"
        severity note;
      for I in UNKNOWN_ON_CLOCK' range loop
        APPLY(CONNECT,UNKNOWN_ON_CLOCK(I).CODES,UNKNOWN_ON_CLOCK(I).WTIME,ALL_PINS);
      end loop;

    end;



  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs, along with the period
  --  of each slice) are read in from an external file.  The test vectors 
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After <START_TOLERANCE>:  Expected outputs set to appropriate values
  --
  --   After <FS_TIME - END_TOLERANCE>: Get next set of inputs and outputs
  --
  --   After <FS_TIME> (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
  procedure SN54xx161A_Test
  (
    signal CONNECT   : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time) is

    variable TOLERANCE : Time := START_TOLERANCE + END_TOLERANCE;

    constant FSA : Frame_Set_Array :=
      NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY (OUTPUT_FRAMES, O_PINS);

    constant DT_BASIS : Delay_Time_Basis :=(FALSE, 0 ns, 0 ns, 
                                            TIMED_DELAY, 0, 0, 0);

    variable TD : Time_Data := NEW_TIME_DATA (FSA);

    constant FSA_DONT_CARE : Frame_Set_Array :=
      NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY (DONT_CARE_FRAMES, O_PINS);

    variable TD_DONT_CARE : Time_Data := NEW_TIME_DATA (FSA_DONT_CARE);

    file INPUT_FILE : Text is in INPUT_FILE_NAME;

    -- the file slice must be allocated
    variable SN54xx161A_FILE_SLICE : File_Slice := NEW_FILE_SLICE;

  begin
    loop
      READ_FILE_SLICE(INPUT_FILE, SN54xx161A_FILE_SLICE);
      exit when SN54xx161A_FILE_SLICE.END_OF_FILE;
      if SN54xx161A_FILE_SLICE.FS_TIME > TOLERANCE then
        APPLY(CONNECT, SN54xx161A_FILE_SLICE.CODES.all,
              DELAY(START_TOLERANCE), TD_DONT_CARE);
        APPLY(CONNECT, SN54xx161A_FILE_SLICE.CODES.all,
              DELAY(SN54xx161A_FILE_SLICE.FS_TIME - TOLERANCE), TD);
        APPLY(CONNECT, SN54xx161A_FILE_SLICE.CODES.all,
              DELAY(END_TOLERANCE), TD_DONT_CARE);
      else
        APPLY(CONNECT, SN54xx161A_FILE_SLICE.CODES.all,
              DELAY(SN54xx161A_FILE_SLICE.FS_TIME), TD);
      end if;
    end loop;
  end;

end WGENERATOR_SN54xx161A; 
