-- Filename : SN54xx161A_timing.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
library STD_PACK;
  use STD_PACK.SIMFLAG.all;
  use STD_PACK.TTL_TIMING.all;

package SN54xx161A_TIMING is

  type Mode_Type is (ERROR,CLR_ERROR,LOAD_ERROR,CLR_CNTR,LD_CNTR,CNT,HOLD);
  type Speed_Type is (MIN,TYP,MAX);

  type Prop_Times is record
    TPLH_CLK_RCO   : Time;
    TPHL_CLK_RCO   : Time;
    TPLH_CLK_QA    : Time;
    TPLH_CLK_QB    : Time;
    TPLH_CLK_QC    : Time;
    TPLH_CLK_QD    : Time;
    TPHL_CLK_QA    : Time;
    TPHL_CLK_QB    : Time;
    TPHL_CLK_QC    : Time;
    TPHL_CLK_QD    : Time;
    TPLH_ENT_RCO   : Time;
    TPHL_ENT_RCO   : Time;
    TPHL_CLR_QA    : Time;
    TPHL_CLR_QB    : Time;
    TPHL_CLR_QC    : Time;
    TPHL_CLR_QD    : Time;
--  -----------------------------------------------------------------------
--  note : TPHL_CLR_RCO is not specified in the TTL Data Book.  
--         The delay for this parameter was calculated as follows:
--         TPHL_CLR_RCO = TPHL_CLR_Q + (TPHL_CLK_RCO - TPHL_CLK_Q)
--  ----------------------------------------------------------------------- 
    TPHL_CLR_RCO   : Time;
  end record;

  type Delay_Type is array (Tech_Type,Speed_Type) of Prop_Times;

  type Cload_Out is record  -- capacitive loadings on output pins
    PIN_QA  : Real ;
    PIN_QB  : Real ;
    PIN_QC  : Real ;
    PIN_QD  : Real ;
    PIN_RCO : Real ;
  end record;

  type Constraint_Type is array (Tech_Type) of Time;

----------- data book times ----------------

  constant DATASHEET_TIMES : Delay_Type :=
  (

   -- STD family
    ( others => (others => 0.0 ns )),

   -- S family
    ( others => (others => 0.0 ns )),

   -- LS family
    (  --  minimum times
       ( TPLH_CLK_RCO   => 7.0 ns,
         TPHL_CLK_RCO   => 6.3 ns,
         TPLH_CLK_QA    => 4.6 ns,
         TPLH_CLK_QB    => 4.6 ns,
         TPLH_CLK_QC    => 4.6 ns,
         TPLH_CLK_QD    => 4.6 ns,
         TPHL_CLK_QA    => 6.3 ns,
         TPHL_CLK_QB    => 6.3 ns,
         TPHL_CLK_QC    => 6.3 ns,
         TPHL_CLK_QD    => 6.3 ns,
         TPLH_ENT_RCO   => 3.2 ns,
         TPHL_ENT_RCO   => 3.2 ns,
         TPHL_CLR_QA    => 7.0 ns,
         TPHL_CLR_QB    => 7.0 ns,
         TPHL_CLR_QC    => 7.0 ns,
         TPHL_CLR_QD    => 7.0 ns,
         TPHL_CLR_RCO   => 7.0 ns
       ),
      --  typical times
       ( TPLH_CLK_RCO   => 20.0 ns,
         TPHL_CLK_RCO   => 18.0 ns,
         TPLH_CLK_QA    => 13.0 ns,
         TPLH_CLK_QB    => 13.0 ns,
         TPLH_CLK_QC    => 13.0 ns,
         TPLH_CLK_QD    => 13.0 ns,
         TPHL_CLK_QA    => 18.0 ns,
         TPHL_CLK_QB    => 18.0 ns,
         TPHL_CLK_QC    => 18.0 ns,
         TPHL_CLK_QD    => 18.0 ns,
         TPLH_ENT_RCO   =>  9.0 ns,
         TPHL_ENT_RCO   =>  9.0 ns,
         TPHL_CLR_QA    => 20.0 ns,
         TPHL_CLR_QB    => 20.0 ns,
         TPHL_CLR_QC    => 20.0 ns,
         TPHL_CLR_QD    => 20.0 ns,
         TPHL_CLR_RCO   => 20.0 ns
       ),
      --  maximum times
       ( TPLH_CLK_RCO   => 35.0 ns,
         TPHL_CLK_RCO   => 35.0 ns,
         TPLH_CLK_QA    => 24.0 ns,
         TPLH_CLK_QB    => 24.0 ns,
         TPLH_CLK_QC    => 24.0 ns,
         TPLH_CLK_QD    => 24.0 ns,
         TPHL_CLK_QA    => 27.0 ns,
         TPHL_CLK_QB    => 27.0 ns,
         TPHL_CLK_QC    => 27.0 ns,
         TPHL_CLK_QD    => 27.0 ns,
         TPLH_ENT_RCO   => 14.0 ns,
         TPHL_ENT_RCO   => 14.0 ns,
         TPHL_CLR_QA    => 28.0 ns,
         TPHL_CLR_QB    => 28.0 ns,
         TPHL_CLR_QC    => 28.0 ns,
         TPHL_CLR_QD    => 28.0 ns,
         TPHL_CLR_RCO   => 28.0 ns
       )
    )
  );

-- note:  The data book specifies only TW_CLK, assumed to refer to the
--        high clock pulse.  The maximum frequency of 25 MHz corresponds 
--        to a clock period of 40 ns is obtained.  The clock-low pulse
--        width is then TW_CLK_LO = (40.0 ns - 25.0 ns) = 15.0 ns

--                                      STD     S       LS
  constant TW_CLK_HI : Constraint_Type := (0.0 ns, 0.0 ns, 25.0 ns);
  constant TW_CLK_LO : Constraint_Type := (0.0 ns, 0.0 ns, 15.0 ns);
  constant TW_CLR    : Constraint_Type := (0.0 ns, 0.0 ns, 20.0 ns);
  constant TSU       : Constraint_Type := (0.0 ns, 0.0 ns, 20.0 ns);
  constant TH        : Constraint_Type := (0.0 ns, 0.0 ns, 3.0 ns);

  -------------- function declarations -----------------------

  function CALC_CURVE ( X : Real; COEFFS : Four_Coeffs ) return Real;

  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times;

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times;
    
end SN54xx161A_TIMING;

-- ======================================================================

package body SN54xx161A_TIMING is

-- ------------------------------------------------------------
  function CALC_CURVE ( X : Real; COEFFS : Four_Coeffs ) return Real is

    variable O : Real;

  begin
    O := COEFFS(4)*(X)**3 + COEFFS(3)*(X)**2 + COEFFS(2)*(X) + COEFFS(1);
    return O;
  end CALC_CURVE;

-- --------------------GET_TIMING----------------------------------------
  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times is

    variable KVLH,KVHL       : Real;   -- voltage derating factors
    variable KTLH,KTHL       : Real;   -- temperature derating factors

    variable DERATINGLH      : Real;   -- composite derating factors
    variable DERATINGHL      : Real;

    variable MTIME           : Prop_Times;

  begin

    -- Check for SIMFLAG options out-of-range 

    assert not (GEN_TA < -55.0 or GEN_TA > 125.0) report 
      "Temperature out of range" severity error;
    assert not (GEN_VCC < 4.5 or GEN_VCC > 5.5 ) report 
      "Voltage out of range" severity error;
    assert not (GEN_DFACTOR < 0.0 ) report 
       "Negative GEN_DFACTOR not allowed" severity error;

    assert (GEN_FAMILY = LS_FAMILY) report 
        "Only the LS family is available" severity error;


    -- calculate derating factors
    KTLH := CALC_CURVE (GEN_TA,   TLH_COEFFICIENT(GEN_FAMILY));
    KTHL := CALC_CURVE (GEN_TA,   THL_COEFFICIENT(GEN_FAMILY));
    KVLH := CALC_CURVE (GEN_VCC,  VLH_COEFFICIENT(GEN_FAMILY));
    KVHL := CALC_CURVE (GEN_VCC,  VHL_COEFFICIENT(GEN_FAMILY));

    -- select delay values
    case GEN_TIME_MODE is
      when MINIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MIN);
      when TYPICAL        => MTIME := DATASHEET_TIMES(GEN_FAMILY,TYP);
      when MAXIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MAX);
    end case;

    -- composite derating factors
    DERATINGLH := KVLH * KTLH * (GEN_DFACTOR / 100.0);
    DERATINGHL := KVHL * KTHL * (GEN_DFACTOR / 100.0);

    -- apply derating factors to propagation delays

    MTIME.TPLH_CLK_RCO := MTIME.TPLH_CLK_RCO * DERATINGLH;
    MTIME.TPHL_CLK_RCO := MTIME.TPHL_CLK_RCO * DERATINGHL;
    MTIME.TPLH_CLK_QA  := MTIME.TPLH_CLK_QA  * DERATINGLH;
    MTIME.TPLH_CLK_QB  := MTIME.TPLH_CLK_QB  * DERATINGLH;
    MTIME.TPLH_CLK_QC  := MTIME.TPLH_CLK_QC  * DERATINGLH;
    MTIME.TPLH_CLK_QD  := MTIME.TPLH_CLK_QD  * DERATINGLH;
    MTIME.TPHL_CLK_QA  := MTIME.TPHL_CLK_QA  * DERATINGHL;
    MTIME.TPHL_CLK_QB  := MTIME.TPHL_CLK_QB  * DERATINGHL;
    MTIME.TPHL_CLK_QC  := MTIME.TPHL_CLK_QC  * DERATINGHL;
    MTIME.TPHL_CLK_QD  := MTIME.TPHL_CLK_QD  * DERATINGHL;
    MTIME.TPLH_ENT_RCO := MTIME.TPLH_ENT_RCO * DERATINGLH;
    MTIME.TPHL_ENT_RCO := MTIME.TPHL_ENT_RCO * DERATINGHL;
    MTIME.TPHL_CLR_QA  := MTIME.TPHL_CLR_QA  * DERATINGHL;
    MTIME.TPHL_CLR_QB  := MTIME.TPHL_CLR_QB  * DERATINGHL;
    MTIME.TPHL_CLR_QC  := MTIME.TPHL_CLR_QC  * DERATINGHL;
    MTIME.TPHL_CLR_QD  := MTIME.TPHL_CLR_QD  * DERATINGHL;
    MTIME.TPHL_CLR_RCO := MTIME.TPHL_CLR_RCO * DERATINGHL;

    return(MTIME);

  end GET_TIMING;

-- ----------------------------------------------------------
  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times is

    variable KCLH  : Cload_Out;   -- load capacitance derating factors
    variable KCHL  : Cload_Out;

    variable MTIME : Prop_Times;

  begin

    KCLH.PIN_QA  := CALC_CURVE (GEN_CLOADS.PIN_QA, CLH_COEFFICIENT(GEN_FAMILY));
    KCLH.PIN_QB  := CALC_CURVE (GEN_CLOADS.PIN_QB, CLH_COEFFICIENT(GEN_FAMILY));
    KCLH.PIN_QC  := CALC_CURVE (GEN_CLOADS.PIN_QC, CLH_COEFFICIENT(GEN_FAMILY));
    KCLH.PIN_QD  := CALC_CURVE (GEN_CLOADS.PIN_QD, CLH_COEFFICIENT(GEN_FAMILY));
    KCLH.PIN_RCO := CALC_CURVE (GEN_CLOADS.PIN_RCO,CLH_COEFFICIENT(GEN_FAMILY));

    KCHL.PIN_QA  := CALC_CURVE (GEN_CLOADS.PIN_QA, CHL_COEFFICIENT(GEN_FAMILY));
    KCHL.PIN_QB  := CALC_CURVE (GEN_CLOADS.PIN_QB, CHL_COEFFICIENT(GEN_FAMILY));
    KCHL.PIN_QC  := CALC_CURVE (GEN_CLOADS.PIN_QC, CHL_COEFFICIENT(GEN_FAMILY));
    KCHL.PIN_QD  := CALC_CURVE (GEN_CLOADS.PIN_QD, CHL_COEFFICIENT(GEN_FAMILY));
    KCHL.PIN_RCO := CALC_CURVE (GEN_CLOADS.PIN_RCO,CHL_COEFFICIENT(GEN_FAMILY));

    MTIME.TPLH_CLK_RCO := CHIP_TIMES.TPLH_CLK_RCO * KCLH.PIN_RCO;
    MTIME.TPHL_CLK_RCO := CHIP_TIMES.TPHL_CLK_RCO * KCHL.PIN_RCO;
    MTIME.TPLH_CLK_QA  := CHIP_TIMES.TPLH_CLK_QA  * KCLH.PIN_QA;
    MTIME.TPLH_CLK_QB  := CHIP_TIMES.TPLH_CLK_QB  * KCLH.PIN_QB;
    MTIME.TPLH_CLK_QC  := CHIP_TIMES.TPLH_CLK_QC  * KCLH.PIN_QC;
    MTIME.TPLH_CLK_QD  := CHIP_TIMES.TPLH_CLK_QD  * KCLH.PIN_QD;
    MTIME.TPHL_CLK_QA  := CHIP_TIMES.TPHL_CLK_QA  * KCHL.PIN_QA;
    MTIME.TPHL_CLK_QB  := CHIP_TIMES.TPHL_CLK_QB  * KCHL.PIN_QB;
    MTIME.TPHL_CLK_QC  := CHIP_TIMES.TPHL_CLK_QC  * KCHL.PIN_QC;
    MTIME.TPHL_CLK_QD  := CHIP_TIMES.TPHL_CLK_QD  * KCHL.PIN_QD;
    MTIME.TPLH_ENT_RCO := CHIP_TIMES.TPLH_ENT_RCO * KCLH.PIN_RCO;
    MTIME.TPHL_ENT_RCO := CHIP_TIMES.TPHL_ENT_RCO * KCHL.PIN_RCO;
    MTIME.TPHL_CLR_QA  := CHIP_TIMES.TPHL_CLR_QA  * KCHL.PIN_QA;
    MTIME.TPHL_CLR_QB  := CHIP_TIMES.TPHL_CLR_QB  * KCHL.PIN_QB;
    MTIME.TPHL_CLR_QC  := CHIP_TIMES.TPHL_CLR_QC  * KCHL.PIN_QC;
    MTIME.TPHL_CLR_QD  := CHIP_TIMES.TPHL_CLR_QD  * KCHL.PIN_QD;
    MTIME.TPHL_CLR_RCO := CHIP_TIMES.TPHL_CLR_RCO * KCHL.PIN_RCO;

  return(MTIME);

end BACK_ANNOTATE;

------------------------------------------------------------
end SN54xx161A_TIMING;
 
