-- Filename : SN54xx161A.vhd
--
-- Version 2.1
--
--
--                                   NOTICE
--
--      LICENSEE acknowledges that it has read and executed the
--      Institute for Technology Development, VHDL Standard Component
--      Library Software Standard License Agreement.  Licensee may not
--      sublicense, assign or transfer the Software, the Documentation or its
--      rights under the Agreement.
--      
--      The Agreement authorizes Licensee to use at any one time only
--      one copy of the Software (either the original Software or the
--      Archival Copy but never both.)  Licensee may make one (1) copy
--      of the Software (the "Archival Copy") and one (1) copy of the
--      Documentation for archival purposes.  Licensee may not otherwise copy,
--      modify, transfer or disclose in any manner the Software or
--      Documentation.  Licensee shall store the Archival Copy in a secure
--      location to which access shall be restricted.  Licensee shall maintain
--      a written record of the data on which the Archival Copy is created and
--      its location.
--      
--      Licensee expressly recognizes that the Software is the proprietary
--      property of ITD.  Licensee warrants that it will not take any action
--      which would result in the impairment or loss of proprietary rights of
--      ITD in the Software.  Licensee warrants that the Software will
--      not be incorporated into any Licensee product.  Licensee warrants 
--      that it will not decompile, disassemble or reverse engineer the 
--      Software or attempt to do so for other than internal applications.
--      Licensee warrants that it will never divulge to any person without the
--      prior written consent of ITD the Software source code or Software
--      object code.
--      
--      The Software is provided "AS IS" WITHOUT WARRANTY of any kind, either
--      express or implied, including but not limited to implied warranties of
--      merchantability and fitness for a particular purpose.
--      
--      RIGHT TO ACCESS OF THIS SOFTWARE IS DENIED IF THIS NOTICE IS REMOVED.
--      
--
--        ------------------------------------------------------------------

  use STD.TEXTIO.all;

library IEEE;                        -- Logic system
  use IEEE.STD_LOGIC_1164.all;  -- Defines logic types, operators, functions

library STD_PACK;                 -- Standard library for all models
  use STD_PACK.TIME_FUNC.all;     -- Timing violation functions package
  use STD_PACK.MISC_FUNC.all;     -- Miscellaneous functions
  use STD_PACK.TTL_TIMING.all;

library SN54xx161A;
  use SN54xx161A.SN54xx161A_TIMING.all;  -- SN54xx161A timing module

entity SN54xx161A is

  generic      -- values are for nominal operating conditions
  (
    GEN_PROP_TIMES  : Prop_Times ;
    GEN_XGEN        : Boolean ;
    GEN_TIMING_MESG : Boolean ;
    GEN_FAMILY      : Tech_Type ;

    GEN_TW_CLK_HI : Time ;
    GEN_TW_CLK_LO : Time ;
    GEN_TW_CLR    : Time ;
    GEN_TSU       : Time ;
    GEN_TH        : Time ;

    TWD_CLR  : Time ;
    TWD_CLK  : Time ;
    TWD_A    : Time ;
    TWD_B    : Time ;
    TWD_C    : Time ;
    TWD_D    : Time ;
    TWD_ENP  :Time ;
    TWD_LOAD : Time ;
    TWD_ENT  : Time ;

    CL_QA    : Real ;
    CL_QB    : Real ;
    CL_QC    : Real ;
    CL_QD    : Real ;
    CL_RCO   : Real ;

    REF     : String
  );

  port 
  (
    QA        : out Std_Ulogic;
    QB        : out Std_Ulogic;
    QC        : out Std_Ulogic;
    QD        : out Std_Ulogic;
    RC        : out Std_Ulogic;

    CLR       : in Std_Ulogic;
    CLK       : in Std_Ulogic;
    A         : in Std_Ulogic;
    B         : in Std_Ulogic;
    C         : in Std_Ulogic;
    D         : in Std_Ulogic;
    ENP       : in Std_Ulogic;
    LOAD      : in Std_Ulogic;
    ENT       : in Std_Ulogic
  );

end SN54xx161A;

-- ======================================================================

architecture behavioral of SN54xx161A is

  constant CL_OUT : Cload_Out := (
    PIN_QA => CL_QA,
    PIN_QB => CL_QB,
    PIN_QC => CL_QC,
    PIN_QD => CL_QD,
    PIN_RCO => CL_RCO
  );

  constant MODEL_TIMES : Prop_Times := 
    BACK_ANNOTATE(GEN_PROP_TIMES,GEN_FAMILY,CL_OUT) ;

  -- Local signal declarations

  signal CLOCK,CLEAR,ENABLEP     : UX01       := 'U';
  signal LOADIN,ENABLET          : UX01       := 'U';
  signal AIN,BIN,CIN,DIN         : UX01       := 'U';
  signal QOUTA,QOUTB,QOUTC,QOUTD : UX01Z      := 'U';

  -----------------------------------------------------------------
  -- two internal signals, RC_CLOCK and RC_ENT, are used to 
  -- implement the delay path to output RCO.  This "structural"
  -- implementation is necessary to produce the correct propagation
  -- delay from input ENT or CLOCK to output RCO.  Consult the 1988 
  -- TI Data Book, page 2-499 for a structural diagram.
  -----------------------------------------------------------------
  signal RC_CLOCK, RC_ENT : Std_Ulogic := 'U';

  -- Global signals
  signal VIOLATION    : Resolved_Boolean := FALSE;
  signal CLEAR_LAST   : Time := 0 ns;
  signal CLOCK_LAST   : Time := 0 ns;

begin

  -- assign pin values to internal signals after wire delay

  CLEAR    <= transport CONVERT_TO_UX01(CLR)  after TWD_CLR;
  CLOCK    <= transport CONVERT_TO_UX01(CLK)  after TWD_CLK;
  AIN      <= transport CONVERT_TO_UX01(A)    after TWD_A;
  BIN      <= transport CONVERT_TO_UX01(B)    after TWD_B;
  CIN      <= transport CONVERT_TO_UX01(C)    after TWD_C;
  DIN      <= transport CONVERT_TO_UX01(D)    after TWD_D;
  LOADIN   <= transport CONVERT_TO_UX01(LOAD) after TWD_LOAD;
  ENABLEP  <= transport CONVERT_TO_UX01(ENP)  after TWD_ENP;
  ENABLET  <= transport CONVERT_TO_UX01(ENT)  after TWD_ENT;
  RC_ENT   <= transport ENABLET after TP_DELAY(ENABLET,MODEL_TIMES.TPLH_ENT_RCO,
                                                       MODEL_TIMES.TPHL_ENT_RCO);

  -- Assign internal values to output pins
 
  QA <=  QOUTA;
  QB <=  QOUTB;
  QC <=  QOUTC;
  QD <=  QOUTD;

  -- Check for invalid inputs on control input ports

  assert not (CLK'event and BIT_UNKNOWN(CLK))
    report "{" & ref & "} " & STD_ULOGIC_to_char(CLK) & 
    " state on CLK input" severity warning;

  assert not (CLR'event and BIT_UNKNOWN(CLR))
    report "{" & ref & "} " & STD_ULOGIC_TO_CHAR(CLR) & 
    " state on CLR input" severity warning;

  assert not (ENP'event and  BIT_UNKNOWN(ENP))
    report "{" & ref & "} " & STD_ULOGIC_TO_CHAR(ENP) & 
    " state on ENP input" severity warning;

  assert not (LOAD'event and BIT_UNKNOWN(LOAD))
    report "{" & ref & "} " & STD_ULOGIC_to_char(LOAD) & 
    " state on LOAD input" severity warning;

  assert not (ENT'event and BIT_UNKNOWN(ENT))
    report "{" & ref & "} " & STD_ULOGIC_to_char(ENT) & 
    " state on ENT input" severity warning;

-------------------------------------------------------------------------
  LAST_CLEAR : process (CLEAR)
  begin
    CLEAR_LAST  <= NOW;
  end process LAST_CLEAR;
-------------------------------------------------------------------------
  LAST_CLOCK : process (CLOCK)
  begin
    CLOCK_LAST  <= NOW;
  end process LAST_CLOCK;
-------------------------------------------------------------------------

  CLK_PULSE_CHECK : process

    variable CLK_VIOLATE : Boolean;
    variable TIMESTRING : Line;

  begin
    wait on CLOCK until (CLEAR = '1' and 
      (RISING_EDGE(CLOCK) or FALLING_EDGE(CLOCK)));

    if CLOCK = '0' then
      CLK_VIOLATE := (NOW - CLOCK_LAST) < GEN_TW_CLK_HI;
    write (TIMESTRING, NOW);
    assert not(GEN_TIMING_MESG and CLK_VIOLATE)
      report "{" & ref & "} PULSE CHECK WARNING: CLK - HIGH at time"
      & TIMESTRING.all severity warning;
    deallocate (TIMESTRING);

    else
      CLK_VIOLATE := (NOW - CLOCK_LAST) < GEN_TW_CLK_LO;
      write (TIMESTRING, NOW);
      assert not(GEN_TIMING_MESG and CLK_VIOLATE)
        report "{" & ref & "} PULSE CHECK WARNING: CLK - LOW at time "
        & TIMESTRING.all severity warning;
      deallocate (TIMESTRING);
    end if;

    VIOLATION <= CLK_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process CLK_PULSE_CHECK;

-------------------------------------------------------------------------

  CLR_PULSE_CHECK : process

    variable CLR_VIOLATE : Boolean;
    variable TIMESTRING : Line;

  begin
    wait until RISING_EDGE (CLEAR);

    CLR_VIOLATE := (NOW - CLEAR_LAST) < GEN_TW_CLR;
    write (TIMESTRING, NOW);
    assert not(GEN_TIMING_MESG and CLR_VIOLATE)
      report "{" & ref & "} PULSE CHECK WARNING: CLR - LOW at time "
      & TIMESTRING.all severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= CLR_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process CLR_PULSE_CHECK;

  -------------------------------------------------------------------------

  SETUP_CHECKS : process

    variable SETUP_AIN_VIOLATE  : Boolean;
    variable SETUP_BIN_VIOLATE  : Boolean;
    variable SETUP_CIN_VIOLATE  : Boolean;
    variable SETUP_DIN_VIOLATE  : Boolean;
    variable SETUP_ENP_VIOLATE  : Boolean;
    variable SETUP_ENT_VIOLATE  : Boolean;
    variable SETUP_LOAD_VIOLATE : Boolean;
    variable TIMESTRING : Line;

  begin
    wait on CLOCK until (CLEAR = '1' and RISING_EDGE(CLOCK));

    write (TIMESTRING, NOW);

    if LOADIN = '0' then
      SETUP_AIN_VIOLATE := not AIN'stable(GEN_TSU);
      assert not(GEN_TIMING_MESG and SETUP_AIN_VIOLATE)
        report "{" & REF & "} setup check warning: A at time " & TIMESTRING.all
        severity warning;

      SETUP_BIN_VIOLATE := not BIN'stable(GEN_TSU);
      assert not(GEN_TIMING_MESG and SETUP_BIN_VIOLATE)
        report "{" & REF & "} setup check warning: B at time " & TIMESTRING.all
        severity warning;

      SETUP_CIN_VIOLATE := not CIN'stable(GEN_TSU);
      assert not(GEN_TIMING_MESG and SETUP_CIN_VIOLATE)
        report "{" & REF & "} setup check warning: C at time " & TIMESTRING.all
        severity warning;

      SETUP_DIN_VIOLATE := not DIN'stable(GEN_TSU);
      assert not(GEN_TIMING_MESG and SETUP_DIN_VIOLATE)
        report "{" & REF & "} setup check warning: D at time " & TIMESTRING.all
        severity warning;
    end if;

    SETUP_ENP_VIOLATE := not ENABLEP'stable(GEN_TSU);
    assert not(GEN_TIMING_MESG and SETUP_ENP_VIOLATE)
      report "{" & REF & "} setup check warning: ENP at time " & TIMESTRING.all
      severity warning;

    SETUP_ENT_VIOLATE := not ENABLET'stable(GEN_TSU);
    assert not(GEN_TIMING_MESG and SETUP_ENT_VIOLATE)
      report "{" & REF & "} setup check warning: ENT at time " & TIMESTRING.all
      severity warning;

    SETUP_LOAD_VIOLATE := not LOADIN'stable(GEN_TSU);
    assert not(GEN_TIMING_MESG and SETUP_LOAD_VIOLATE)
      report "{" & REF & "} setup check warning: LOAD at time " & TIMESTRING.all
      severity warning;

    deallocate (TIMESTRING);

    VIOLATION <= GEN_XGEN and (SETUP_AIN_VIOLATE or
                               SETUP_BIN_VIOLATE or
                               SETUP_CIN_VIOLATE or
                               SETUP_DIN_VIOLATE or
                               SETUP_ENP_VIOLATE or
                               SETUP_ENT_VIOLATE or
                               SETUP_LOAD_VIOLATE );
    wait for 0 ns;
    VIOLATION  <= FALSE;

  end process SETUP_CHECKS;

-------------------------------------------------------------------------

  A_HOLD_CHECK : process

    variable HOLD_AIN_VIOLATE : Boolean;
    variable TIMESTRING : Line;

  begin
    wait on AIN until CLEAR = '1' and CLOCK'last_value = '0' and
      CLOCK = '1' and LOADIN = '0';

    HOLD_AIN_VIOLATE := not CLOCK'stable (GEN_TH);
    write (TIMESTRING, NOW);
    assert not(GEN_TIMING_MESG and HOLD_AIN_VIOLATE)
      report "{" & REF & "} hold check warning: A at time " & TIMESTRING.all
      severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= HOLD_AIN_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process A_HOLD_CHECK;

-------------------------------------------------------------------------

  B_HOLD_CHECK : process

    variable HOLD_BIN_VIOLATE : Boolean;
    variable TIMESTRING : Line;

  begin
    wait on BIN until CLEAR = '1' and CLOCK'last_value = '0' and
      CLOCK = '1' and LOADIN = '0';

    HOLD_BIN_VIOLATE := not CLOCK'stable (GEN_TH);
    write (TIMESTRING, NOW);
    assert not (GEN_TIMING_MESG and HOLD_BIN_VIOLATE)
      report "{" & REF & "} hold check warning: B at time " & TIMESTRING.all
      severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= HOLD_BIN_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process B_HOLD_CHECK;

  -------------------------------------------------------------------------`

  C_HOLD_CHECK : process

    variable HOLD_CIN_VIOLATE : Boolean;
    variable TIMESTRING : Line;

  begin
    wait on CIN until CLEAR = '1' and CLOCK'last_value = '0' and
      CLOCK = '1' and LOADIN = '0';

    HOLD_CIN_VIOLATE := not CLOCK'stable (GEN_TH);
    write (TIMESTRING, NOW);
    assert not (GEN_TIMING_MESG and HOLD_CIN_VIOLATE)
      report "{" & REF & "} hold check warning: C at time " & TIMESTRING.all
      severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= HOLD_CIN_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process C_HOLD_CHECK;

  -------------------------------------------------------------------------

  D_HOLD_CHECK : process

    variable HOLD_DIN_VIOLATE : Boolean;
    variable TIMESTRING : Line;

  begin
    wait on DIN until CLEAR = '1' and CLOCK'last_value = '0' and
      CLOCK = '1' and LOADIN = '0';

    HOLD_DIN_VIOLATE := not CLOCK'stable (GEN_TH);
    write (TIMESTRING, NOW);
    assert not(GEN_TIMING_MESG and HOLD_DIN_VIOLATE)
      report "{" & REF & "} hold check warning: D at time " & TIMESTRING.all
      severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= HOLD_DIN_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process D_HOLD_CHECK;

-------------------------------------------------------------------------

  ENP_HOLD_CHECK : process

    variable HOLD_ENP_VIOLATE : Boolean;
    variable TIMESTRING : Line;

  begin
    wait on ENABLEP until CLEAR = '1' and CLOCK'last_value = '0' and CLOCK = '1';

    HOLD_ENP_VIOLATE := not CLOCK'stable (GEN_TH);
    write (TIMESTRING, NOW);
    assert not(GEN_TIMING_MESG and HOLD_ENP_VIOLATE)
      report "{" & REF & "} hold check warning: ENP at time " & TIMESTRING.all
      severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= HOLD_ENP_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process ENP_HOLD_CHECK;

-------------------------------------------------------------------------

  ENT_HOLD_CHECK : process

    variable HOLD_ENT_VIOLATE :Boolean;
    variable TIMESTRING : Line;

  begin
    wait on ENABLET until CLEAR = '1' and CLOCK'last_value = '0' and CLOCK = '1';

    HOLD_ENT_VIOLATE := not CLOCK'stable (GEN_TH);
    write (TIMESTRING, NOW);
    assert not(GEN_TIMING_MESG and HOLD_ENT_VIOLATE)
      report "{" & REF & "} hold check warning: ENT at time " & TIMESTRING.all
      severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= HOLD_ENT_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process ENT_HOLD_CHECK;

-------------------------------------------------------------------------

  LOAD_HOLD_CHECK : process

    variable HOLD_LOAD_VIOLATE : Boolean;
    variable TIMESTRING : Line;

  begin
    wait on LOADIN until CLEAR = '1' and CLOCK'last_value = '0' and CLOCK = '1';

    HOLD_LOAD_VIOLATE := not CLOCK'stable (GEN_TH);
    write (TIMESTRING, NOW);
    assert not (GEN_TIMING_MESG and HOLD_LOAD_VIOLATE)
      report "{" & REF & "} hold check warning: LOAD at time " & TIMESTRING.all
      severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= HOLD_LOAD_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process LOAD_HOLD_CHECK;

  -------------------------------------------------------------------------

  COUNT : process (CLEAR, CLOCK, ENABLEP, ENABLET, LOADIN, VIOLATION)

    variable MODE      : Mode_Type;
    variable CARRY     : Std_Ulogic;
    variable RC        : Std_Ulogic;
    variable QSTATE    : Std_Ulogic_Vector (0 to 3);
    variable UNKNOWN   : Boolean := FALSE;

  begin

      -- select mode of operation

      -- 'U' state has highest priority!
      if (CLOCK = 'U' or ENABLEP = 'U' or ENABLET = 'U' or LOADIN = 'U' 
          or CLEAR = 'U' or AIN = 'U' or BIN = 'U' or CIN = 'U' or DIN = 'U') then
        QOUTA     <= 'U';
        QOUTB     <= 'U';
        QOUTC     <= 'U';
        QOUTD     <= 'U';
        RC_CLOCK  <= 'U';
        MODE := HOLD;
      elsif (VIOLATION) then -- a timing violation has occurred
        MODE := ERROR;
      elsif (CLEAR = '0') then
        MODE := CLR_CNTR;
      elsif (CLEAR = '1' and LOADIN = '0') then
        MODE := LD_CNTR;
      elsif(CLEAR = '1' and LOADIN = '1') then
        MODE := CNT;
      elsif (BIT_UNKNOWN(CLEAR)) then
        MODE := CLR_ERROR;
      elsif (BIT_UNKNOWN(LOADIN) and 
             (RISING_EDGE(CLOCK) or BIT_UNKNOWN(CLOCK))) then
        MODE := LOAD_ERROR;
      else
        MODE := HOLD;
      end if;
    
      -- operation of counter depends on current mode

      case MODE is
        -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
        when ERROR => -- a timing violation has occurred
          QOUTA     <= 'X';
          QOUTB     <= 'X';
          QOUTC     <= 'X';
          QOUTD     <= 'X';
          RC_CLOCK  <= 'X';
    
        -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
        when CLR_ERROR =>
        -- if a digit is already zero, don't set to 'X'
          if (QOUTA /= '0') then
            QOUTA <= 'X';
          end if;
    
          if (QOUTB /= '0') then
            QOUTB <= 'X';
          end if;
    
          if (QOUTC /= '0') then
            QOUTC <= 'X';
          end if;
    
          if (QOUTD /= '0') then
            QOUTD <= 'X';
          end if;

          if (RC_CLOCK /= '0') then
            RC_CLOCK <= 'X';
          end if;
    
        -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
        when LOAD_ERROR =>
          -- set to 'X' if data /= present output
          if (AIN /= QOUTA) then
            QOUTA <= 'X';
          end if;
    
          if (BIN /= QOUTB) then
            QOUTB <= 'X';
          end if;
    
          if (CIN /= QOUTC) then
            QOUTC <= 'X';
          end if;
    
          if (DIN /= QOUTD) then
            QOUTD <= 'X';
          end if;
    
        -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
        when CLR_CNTR =>
          QOUTA    <= transport '0' after MODEL_TIMES.TPHL_CLR_QA;
          QOUTB    <= transport '0' after MODEL_TIMES.TPHL_CLR_QB;
          QOUTC    <= transport '0' after MODEL_TIMES.TPHL_CLR_QC;
          QOUTD    <= transport '0' after MODEL_TIMES.TPHL_CLR_QD;
          RC_CLOCK <= transport '0' after MODEL_TIMES.TPHL_CLR_RCO;

        -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
        when LD_CNTR =>
          if (RISING_EDGE(CLOCK)) then
            if (not BIT_UNKNOWN(AIN)) then
              QOUTA <= transport AIN after TP_DELAY(AIN,
                       MODEL_TIMES.TPLH_CLK_QA,
                       MODEL_TIMES.TPHL_CLK_QA);
            else
              QOUTA <= 'X';
            end if;
    
            if (not BIT_UNKNOWN(BIN)) then
              QOUTB <= transport BIN after TP_DELAY(BIN,
                       MODEL_TIMES.TPLH_CLK_QB,
                       MODEL_TIMES.TPHL_CLK_QB);
            else
              QOUTB <= 'X';
            end if;
    
            if (not BIT_UNKNOWN(CIN)) then
              QOUTC <= transport CIN after TP_DELAY(CIN,
                       MODEL_TIMES.TPLH_CLK_QC,
                       MODEL_TIMES.TPHL_CLK_QC);
            else
              QOUTC <= 'X';
            end if;
    
            if (not BIT_UNKNOWN(DIN)) then
              QOUTD <= transport DIN after TP_DELAY(DIN,
                       MODEL_TIMES.TPLH_CLK_QD,
                       MODEL_TIMES.TPHL_CLK_QD);
            else
              QOUTD <= 'X';
            end if;
    
            RC := AIN and BIN and CIN and DIN;
            RC_CLOCK <= transport RC after TP_DELAY(RC,
                        MODEL_TIMES.TPLH_CLK_RCO,
                        MODEL_TIMES.TPHL_CLK_RCO);
    
          elsif (BIT_UNKNOWN(CLOCK)) then
            if (AIN /= QOUTA) then
              QOUTA <= 'X';
            end if;
    
            if (BIN /= QOUTB) then
              QOUTB <= 'X';
            end if;
    
            if (CIN /= QOUTC) then
              QOUTC <= 'X';
            end if;
   
            if (DIN /= QOUTD) then
              QOUTD <= 'X';
            end if;
    
            RC := QOUTA and QOUTB and QOUTC and QOUTD;
            RC_CLOCK <= transport RC after TP_DELAY(RC,
                        MODEL_TIMES.TPLH_CLK_RCO,
                        MODEL_TIMES.TPHL_CLK_RCO);
          end if;
    
        -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

        when CNT =>
          QSTATE(0) := QOUTD;
          QSTATE(1) := QOUTC;
          QSTATE(2) := QOUTB;
          QSTATE(3) := QOUTA;
    
          if (RISING_EDGE(CLOCK) and ENABLET = '1' and ENABLEP = '1') then
            -- set the carry to '1' to add one to the lsb
            CARRY := '1';
          elsif (CLOCK = 'X' or ENABLET = 'X' or ENABLEP = 'X') then
            -- set the carry to 'X' to propagate down
            CARRY := 'X';
          end if;
          
          -- let the carry propagate down through the other digits.

          for DIGIT_POS in 3 downto 0 loop
            case CARRY is
              when '0' =>
                if (QSTATE(DIGIT_POS) = 'X') then
                  CARRY := 'X';
                else
                  CARRY := '0';
                end if;
              when '1' =>
                CARRY := QSTATE(DIGIT_POS);
                QSTATE(DIGIT_POS) := not QSTATE(DIGIT_POS);
              when 'X' =>
                CARRY := CARRY and QSTATE(DIGIT_POS);
                QSTATE(DIGIT_POS) := 'X';
              when 'U' =>
                CARRY := CARRY and QSTATE(DIGIT_POS);
                QSTATE(DIGIT_POS) := 'U';                            
              when others =>
                null;
            end case;
          end loop;            

          if (ENABLEP = '0' or ENABLET = '0') then -- counter is disabled
            null;

          elsif (RISING_EDGE(CLOCK) or CLOCK = 'X') then
            QOUTA <= transport QSTATE(3) after TP_DELAY(QSTATE(3),
                     MODEL_TIMES.TPLH_CLK_QA,
                     MODEL_TIMES.TPHL_CLK_QA);
            QOUTB <= transport QSTATE(2) after TP_DELAY(QSTATE(2),
                     MODEL_TIMES.TPLH_CLK_QB,
                     MODEL_TIMES.TPHL_CLK_QB);
            QOUTC <= transport QSTATE(1) after TP_DELAY(QSTATE(1),
                     MODEL_TIMES.TPLH_CLK_QC,
                     MODEL_TIMES.TPHL_CLK_QC);
            QOUTD <= transport QSTATE(0) after TP_DELAY(QSTATE(0),
                     MODEL_TIMES.TPLH_CLK_QD,
                     MODEL_TIMES.TPHL_CLK_QD);

            RC := QSTATE(0) and QSTATE(1) and QSTATE(2) and QSTATE(3);
            RC_CLOCK <= transport RC after TP_DELAY(RC,
                        MODEL_TIMES.TPLH_CLK_RCO,
                        MODEL_TIMES.TPHL_CLK_RCO);
          end if;    

        -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

        when HOLD =>
          null;

        -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

      end case; -- MODE

  end process COUNT;
    
  -------------------------------------------------------------------------

  RIPPLE_CARRY : process (RC_CLOCK,RC_ENT)

  begin

   case RC_CLOCK is
     when 'U' | 'X' | 'Z' =>  -- propagate unknown states
       RC <= RC_CLOCK;
     when others => 
       if (RC_ENT = '1' or RC_ENT = 'H') then  -- ENT must be high
         RC <= RC_CLOCK;
       else
         RC <= '0';  -- disabled
       end if;  
    end case;

  end process RIPPLE_CARRY;

  -------------------------------------------------------------------------

end BEHAVIORAL;
    



