-- Filename : SN54xx140_timing.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
library STD_PACK;
  use STD_PACK.SIMFLAG.all;     -- Global simulation options
  use STD_PACK.TTL_TIMING.all;  -- TTL derating coefficients

package SN54xx140_TIMING is

  -------------- type declarations & constants -----------

  type Speed_Type is (MIN,TYP,MAX);

  type Prop_Times is record
    TPLH : Time;
    TPHL : Time;
  end record;

  type Delay_Type is array (Tech_Type,Speed_Type) of Prop_Times;

  ----------- data book times ----------------
  -- NOTE:  The default load capacitance for each SN54xx140 output is 50 pf.

  constant DATASHEET_TIMES : Delay_Type :=
  (
   -- STD family
    ( others => (others => 0 ns)),
  
   -- S family
    (
      --  tplh    tphl
       (  1.4 ns,  1.4 ns), -- MIN
       (  4.0 ns,  4.0 ns), -- TYP
       (  6.5 ns,  6.5 ns)  -- MAX
     ),
  
   -- LS family
    ( others => (others => 0 ns))

  );
  
  ----------------------function declarations--------------------
  
  function CALC_CURVE ( X :  Real; COEFFS :Four_Coeffs ) return Real;
  
  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times;

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOAD  : Real) return Prop_Times;

end SN54xx140_TIMING;

-- ======================================================================

package body SN54xx140_TIMING is

------------------------------------------------------------

  function CALC_CURVE ( X : Real; COEFFS : Four_Coeffs) return Real is
    variable O : Real;

  begin
    O := COEFFS(4)*(X)**3 + COEFFS(3)*(X)**2 + COEFFS(2)*(X) + COEFFS(1);
    return O;
  end CALC_CURVE;

------------------------------------------------------------

  function GET_TIMING (GEN_FAMILY      : Tech_Type;
                       GEN_TIME_MODE   : Time_Modes;
                       GEN_VCC         : Real;
                       GEN_TA          : Real;
                       GEN_DFACTOR     : Real) return Prop_Times is

    variable KVLH,KVHL       : Real;        -- voltage derating factors
    variable KTLH,KTHL       : Real;        -- temperature derating factors
    variable DERATINGLH      : Real;        -- composite derating factors
    variable DERATINGHL      : Real;
    variable MTIME           : Prop_Times;

  begin

    -- Check for SIMFLAG options out-of-range

    assert not ( GEN_TA < -55.0 or GEN_TA > 125.0 ) report
        "Temperature out of range" severity error;
    assert not ( GEN_VCC < 4.5 or GEN_VCC > 5.5 ) report
        "Voltage out of range" severity error;
    assert not ( GEN_DFACTOR < 0.0 ) report
        "Negative DERATE_FACTOR not allowed" severity error;

    assert (GEN_FAMILY = S_FAMILY) report 
        "Only the S family is available" severity error;

    -- Calculate derating factors

    KTLH := CALC_CURVE(GEN_TA,    TLH_COEFFICIENT(GEN_FAMILY));
    KTHL := CALC_CURVE(GEN_TA,    THL_COEFFICIENT(GEN_FAMILY));
    KVLH := CALC_CURVE(GEN_VCC,   VLH_COEFFICIENT(GEN_FAMILY));
    KVHL := CALC_CURVE(GEN_VCC,   VHL_COEFFICIENT(GEN_FAMILY));

    -- select delay values

    case GEN_TIME_MODE is
      when MINIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MIN);
      when TYPICAL        => MTIME := DATASHEET_TIMES(GEN_FAMILY,TYP);
      when MAXIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MAX);
    end case;

    -- composite derating factors

    DERATINGLH := KVLH * KTLH * (GEN_DFACTOR /100.0);
    DERATINGHL := KVHL * KTHL * (GEN_DFACTOR /100.0);

    -- apply derating factors to propagation delays

    MTIME.TPLH := MTIME.TPLH * DERATINGLH;
    MTIME.TPHL := MTIME.TPHL * DERATINGHL;

    return(MTIME);

  end GET_TIMING;

------------------------------------------------------------

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOAD  : Real) return Prop_Times is

    -- The times specified in the TTL Data Book assume a 50 pf capacitive load.
    -- Derating factors below are used to normalize the times in accordance with
    -- the capacitive load assumed in the data book.

    variable KCLH_50pf : Real;    -- load capacitance derating factors
    variable KCHL_50pf : Real;

    variable MTIME     : Prop_Times;

  begin

    -- normalize around 50.0 pf

    KCLH_50pf := CALC_CURVE (GEN_CLOAD, CLH_COEFFICIENT(GEN_FAMILY))
               / CALC_CURVE (50.0, CLH_COEFFICIENT(GEN_FAMILY));

    KCHL_50pf := CALC_CURVE (GEN_CLOAD, CHL_COEFFICIENT(GEN_FAMILY))
               / CALC_CURVE (50.0, CHL_COEFFICIENT(GEN_FAMILY));

    MTIME.TPLH := CHIP_TIMES.TPLH * KCLH_50pf;
    MTIME.TPHL := CHIP_TIMES.TPHL * KCHL_50pf;

    return(MTIME);

  end BACK_ANNOTATE;

------------------------------------------------------------

end SN54xx140_TIMING;
