-- Filename : SN54xx138_timing.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
library IEEE;                        -- Logic system
  use IEEE.STD_LOGIC_1164.all;       -- Defines logic types, operators, functions

library STD_PACK;
  use STD_PACK.SIMFLAG.all;       -- Global simulation options
  use STD_PACK.TTL_TIMING.all;    -- TTL derating coefficients

package SN54xx138_TIMING is

  type Speed_Type is (MIN,TYP,MAX);

  type All_Outputs is (PIN_Y0, PIN_Y1, PIN_Y2, PIN_Y3, 
                       PIN_Y4, PIN_Y5, PIN_Y6, PIN_Y7);

  type Outputs_Vector is array (All_Outputs) of UX01Z;

  type Cload_Out is array (All_Outputs) of Real;
  type Delay_Paths is array (All_Outputs) of Time;

  type Prop_Times is record
    TPLH_BS2 : Delay_Paths;  -- binary select, 2 levels of delay
    TPHL_BS2 : Delay_Paths;
    TPLH_BS3 : Delay_Paths;  -- binary select, 3 levels of delay
    TPHL_BS3 : Delay_Paths;

    TPLH_EN2 : Delay_Paths;  -- enable, 2 levels of delay
    TPHL_EN2 : Delay_Paths;
    TPLH_EN3 : Delay_Paths;  -- enable, 3 levels of delay
    TPHL_EN3 : Delay_Paths;
  end record;

  type Delay_Type is array (Tech_Type,Speed_Type) of Prop_Times;

  ----------- data book times ----------------

  constant DATASHEET_TIMES : Delay_Type := 
  (
    -- STD_family 
    -- ***  data is not available ***
    (others => ( others => (others => 0.0 ns))),

    -- S family
    ( -- minimum times
      ( TPLH_BS2 => (others => 1.6 ns),
        TPHL_BS2 => (others => 2.5 ns),
        TPLH_BS3 => (others => 2.6 ns),
        TPHL_BS3 => (others => 2.8 ns),
        TPLH_EN2 => (others => 1.8 ns),
        TPHL_EN2 => (others => 2.5 ns),
        TPLH_EN3 => (others => 2.5 ns),
        TPHL_EN3 => (others => 2.5 ns)
      ),
      -- typical times
      ( TPLH_BS2 => (others => 4.5 ns),
        TPHL_BS2 => (others => 7.0 ns),
        TPLH_BS3 => (others => 7.5 ns),
        TPHL_BS3 => (others => 8.0 ns),
        TPLH_EN2 => (others => 5.0 ns),
        TPHL_EN2 => (others => 7.0 ns),
        TPLH_EN3 => (others => 7.0 ns),
        TPHL_EN3 => (others => 7.0 ns)
      ),
      -- maximum times
      ( TPLH_BS2 => (others =>  7.0 ns),
        TPHL_BS2 => (others => 10.5 ns),
        TPLH_BS3 => (others => 12.0 ns),
        TPHL_BS3 => (others => 12.0 ns),
        TPLH_EN2 => (others =>  8.0 ns),
        TPHL_EN2 => (others => 11.0 ns),
        TPLH_EN3 => (others => 11.0 ns),
        TPHL_EN3 => (others => 11.0 ns)
      )
   ),

    -- LS family
    ( -- minimum times
      ( TPLH_BS2 => (others => 3.9 ns),
        TPHL_BS2 => (others => 6.3 ns),
        TPLH_BS3 => (others => 7.4 ns),
        TPHL_BS3 => (others => 7.0 ns),
        TPLH_EN2 => (others => 4.2 ns),
        TPHL_EN2 => (others => 7.0 ns),
        TPLH_EN3 => (others => 4.9 ns),
        TPHL_EN3 => (others => 4.6 ns)
      ),
      -- typical times
      ( TPLH_BS2 => (others => 11.0 ns),
        TPHL_BS2 => (others => 18.0 ns),
        TPLH_BS3 => (others => 21.0 ns),
        TPHL_BS3 => (others => 20.0 ns),
        TPLH_EN2 => (others => 12.0 ns),
        TPHL_EN2 => (others => 20.0 ns),
        TPLH_EN3 => (others => 14.0 ns),
        TPHL_EN3 => (others => 13.0 ns)
      ),
      -- maximum times
      ( TPLH_BS2 => (others => 20.0 ns),
        TPHL_BS2 => (others => 41.0 ns),
        TPLH_BS3 => (others => 27.0 ns),
        TPHL_BS3 => (others => 39.0 ns),
        TPLH_EN2 => (others => 18.0 ns),
        TPHL_EN2 => (others => 32.0 ns),
        TPLH_EN3 => (others => 26.0 ns),
        TPHL_EN3 => (others => 38.0 ns)
      )
   )
  );

  ----------------------function declarations--------------------
  
  function CALC_CURVE ( X :  Real; COEFFS :Four_Coeffs ) return Real;
  
  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times;

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times;

end SN54xx138_TIMING;

-- ======================================================================

package body SN54xx138_TIMING is

------------------------------------------------------------

  function CALC_CURVE ( X : Real; COEFFS : Four_Coeffs) return Real is
    variable O : Real;

  begin
    O := COEFFS(4)*(X)**3 + COEFFS(3)*(X)**2 + COEFFS(2)*(X) + COEFFS(1);
    return O;
  end CALC_CURVE;

------------------------------------------------------------

  function GET_TIMING (GEN_FAMILY      : Tech_Type;
                       GEN_TIME_MODE   : Time_Modes;
                       GEN_VCC         : Real;
                       GEN_TA          : Real;
                       GEN_DFACTOR     : Real) return Prop_Times is

    variable KVLH,KVHL       : Real;        -- voltage derating factors
    variable KTLH,KTHL       : Real;        -- temperature derating factors
    variable DERATINGLH      : Real;        -- composite derating factors
    variable DERATINGHL      : Real;
    variable MTIME           : Prop_Times;  

  begin

    -- Check for SIMFLAG options out-of-range

    assert not ( GEN_TA < -55.0 or GEN_TA > 125.0 ) report
        "Temperature out of range" severity error;
    assert not ( GEN_VCC < 4.5 or GEN_VCC > 5.5 ) report
        "Voltage out of range" severity error;
    assert not ( GEN_DFACTOR < 0.0 ) report
        "Negative DERATE_FACTOR not allowed" severity error;

    -- check for illegal family selection

    assert not (GEN_FAMILY = STD_FAMILY) report
      "STD family not available for this part" severity error;

    -- Calculate derating factors

    KTLH := CALC_CURVE(GEN_TA,    TLH_COEFFICIENT(GEN_FAMILY));
    KTHL := CALC_CURVE(GEN_TA,    THL_COEFFICIENT(GEN_FAMILY));
    KVLH := CALC_CURVE(GEN_VCC,   VLH_COEFFICIENT(GEN_FAMILY));
    KVHL := CALC_CURVE(GEN_VCC,   VHL_COEFFICIENT(GEN_FAMILY));

    -- select delay values

    case GEN_TIME_MODE is
      when MINIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MIN);
      when TYPICAL        => MTIME := DATASHEET_TIMES(GEN_FAMILY,TYP);
      when MAXIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MAX);
    end case;

    -- composite derating factors

    DERATINGLH := KVLH * KTLH * (GEN_DFACTOR /100.0);
    DERATINGHL := KVHL * KTHL * (GEN_DFACTOR /100.0);

    -- apply derating factors to propagation delays

    for INDEX in All_Outputs loop
      MTIME.TPLH_BS2(INDEX) := MTIME.TPLH_BS2(INDEX) * DERATINGLH;  
      MTIME.TPHL_BS2(INDEX) := MTIME.TPHL_BS2(INDEX) * DERATINGHL;  
      MTIME.TPLH_BS3(INDEX) := MTIME.TPLH_BS3(INDEX) * DERATINGLH;  
      MTIME.TPHL_BS3(INDEX) := MTIME.TPHL_BS3(INDEX) * DERATINGHL;  

      MTIME.TPLH_EN2(INDEX) := MTIME.TPLH_EN2(INDEX) * DERATINGLH;  
      MTIME.TPHL_EN2(INDEX) := MTIME.TPHL_EN2(INDEX) * DERATINGHL;  
      MTIME.TPLH_EN3(INDEX) := MTIME.TPLH_EN3(INDEX) * DERATINGLH;  
      MTIME.TPHL_EN3(INDEX) := MTIME.TPHL_EN3(INDEX) * DERATINGHL;  
    end loop;

    return(MTIME);

  end GET_TIMING;

------------------------------------------------------------

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times is

    variable KCLH     : Real;    -- load capacitance derating factors
    variable KCHL     : Real;

    variable MTIME    : Prop_Times;

  begin

    for INDEX in All_Outputs loop
      KCLH := CALC_CURVE (GEN_CLOADS(INDEX), CLH_COEFFICIENT(GEN_FAMILY));
      KCHL := CALC_CURVE (GEN_CLOADS(INDEX), CHL_COEFFICIENT(GEN_FAMILY));

      MTIME.TPLH_BS2(INDEX) := CHIP_TIMES.TPLH_BS2(INDEX) * KCLH;  
      MTIME.TPHL_BS2(INDEX) := CHIP_TIMES.TPHL_BS2(INDEX) * KCHL;  
      MTIME.TPLH_BS3(INDEX) := CHIP_TIMES.TPLH_BS3(INDEX) * KCLH;  
      MTIME.TPHL_BS3(INDEX) := CHIP_TIMES.TPHL_BS3(INDEX) * KCHL;  
      MTIME.TPLH_EN2(INDEX) := CHIP_TIMES.TPLH_EN2(INDEX) * KCLH;  
      MTIME.TPHL_EN2(INDEX) := CHIP_TIMES.TPHL_EN2(INDEX) * KCHL;  
      MTIME.TPLH_EN3(INDEX) := CHIP_TIMES.TPLH_EN3(INDEX) * KCLH;  
      MTIME.TPHL_EN3(INDEX) := CHIP_TIMES.TPHL_EN3(INDEX) * KCHL;  
    end loop;

    return(MTIME);

  end BACK_ANNOTATE;

------------------------------------------------------------

end SN54xx138_TIMING;
