-- Filename : SN54xx138.vhd
--
-- Version 2.1
--
--
--                                   NOTICE
--
--      LICENSEE acknowledges that it has read and executed the
--      Institute for Technology Development, VHDL Standard Component
--      Library Software Standard License Agreement.  Licensee may not
--      sublicense, assign or transfer the Software, the Documentation or its
--      rights under the Agreement.
--      
--      The Agreement authorizes Licensee to use at any one time only
--      one copy of the Software (either the original Software or the
--      Archival Copy but never both.)  Licensee may make one (1) copy
--      of the Software (the "Archival Copy") and one (1) copy of the
--      Documentation for archival purposes.  Licensee may not otherwise copy,
--      modify, transfer or disclose in any manner the Software or
--      Documentation.  Licensee shall store the Archival Copy in a secure
--      location to which access shall be restricted.  Licensee shall maintain
--      a written record of the data on which the Archival Copy is created and
--      its location.
--      
--      Licensee expressly recognizes that the Software is the proprietary
--      property of ITD.  Licensee warrants that it will not take any action
--      which would result in the impairment or loss of proprietary rights of
--      ITD in the Software.  Licensee warrants that the Software will
--      not be incorporated into any Licensee product.  Licensee warrants 
--      that it will not decompile, disassemble or reverse engineer the 
--      Software or attempt to do so for other than internal applications.
--      Licensee warrants that it will never divulge to any person without the
--      prior written consent of ITD the Software source code or Software
--      object code.
--      
--      The Software is provided "AS IS" WITHOUT WARRANTY of any kind, either
--      express or implied, including but not limited to implied warranties of
--      merchantability and fitness for a particular purpose.
--      
--      RIGHT TO ACCESS OF THIS SOFTWARE IS DENIED IF THIS NOTICE IS REMOVED.
--      
-- 
--
--        ------------------------------------------------------------------

library IEEE;                        -- Logic system
  use IEEE.STD_LOGIC_1164.all;    -- Defines logic types, operators, functions

library STD_PACK;                   -- Standard library for all models
  use STD_PACK.TIME_FUNC.all;       -- Timing functions
  use STD_PACK.MISC_FUNC.all;       -- Miscellaneous functions
  use STD_PACK.TTL_TIMING.all;       -- TTL derating coefficients

library SN54xx138;
  use SN54xx138.SN54xx138_TIMING.all;    -- SN54xx138 timing module

entity SN54xx138 is
  generic
  (
    GEN_PROP_TIMES : Prop_Times;
    GEN_FAMILY     : Tech_Type;

    TWD_A   : Time := 0.0 ns;    -- wire delays for each input pin
    TWD_B   : Time := 0.0 ns;    -- wire delays for each input pin
    TWD_C   : Time := 0.0 ns;    -- wire delays for each input pin
    TWD_G1  : Time := 0.0 ns;
    TWD_G2A : Time := 0.0 ns;
    TWD_G2B : Time := 0.0 ns;

    CL_Y0 : Real  := 15.0;       -- fanout capacitance seen by output (in pf)
    CL_Y1 : Real  := 15.0;
    CL_Y2 : Real  := 15.0;
    CL_Y3 : Real  := 15.0;
    CL_Y4 : Real  := 15.0;
    CL_Y5 : Real  := 15.0;
    CL_Y6 : Real  := 15.0;
    CL_Y7 : Real  := 15.0;

    REF : String := "SN54xx138"
  );

  port
  (
    Y0     : out Std_Ulogic;
    Y1     : out Std_Ulogic;
    Y2     : out Std_Ulogic;
    Y3     : out Std_Ulogic;
    Y4     : out Std_Ulogic;
    Y5     : out Std_Ulogic;
    Y6     : out Std_Ulogic;
    Y7     : out Std_Ulogic;

    A      : in Std_Ulogic;
    B      : in Std_Ulogic;
    C      : in Std_Ulogic;
    G1     : in Std_Ulogic;
    G2A    : in Std_Ulogic;
    G2B    : in Std_Ulogic
  );

end SN54xx138;

-- ======================================================================
-- Note: 
--       If enable and select inputs change in close sequence, timing
--       complications may arise if an input event preempts the effects 
--       of a previous input event.  In order to assign the correct 
--       propagation delay to an input signal, one must know what the 
--       future output will be after a particular propagation delay.  
--       For the SN54xx138, one may predict the effect, if any, of an
--       input event on each output.
--
--       This SN54xx138 model is based on the logic diagram on page 2-426 
--       of the 1988 TTL Data Book.  An internal signal exists for every
--       unique path from an input to an output, and is assigned the 
--       corresponding input pin value after the appropriate propagation 
--       delay.
--
--       This approach provides a straightforward way to assign
--       propagation delays, depending upon the number of logic levels and
--       the effect an input has on an output.  Although the logic diagram 
--       does not necessarily represent the actual implementation of the 
--       circuit, it does allow the model to accurately implement the 
--       timing behavior of the part.
---------------------------------------------------------------------------

architecture behavioral of SN54xx138 is


  constant CL_OUT : Cload_Out := (
    PIN_Y0 => CL_Y0,
    PIN_Y1 => CL_Y1,
    PIN_Y2 => CL_Y2,
    PIN_Y3 => CL_Y3,
    PIN_Y4 => CL_Y4,
    PIN_Y5 => CL_Y5,
    PIN_Y6 => CL_Y6,
    PIN_Y7 => CL_Y7
  );

  constant MODEL_TIMES : Prop_Times := 
    BACK_ANNOTATE(GEN_PROP_TIMES,GEN_FAMILY,CL_OUT);

  -- Local signal declarations

  signal GIN1,GIN2A,GIN2B  : UX01           := 'U';
  signal AIN,BIN,CIN       : UX01           := 'U';
  signal GIN1_Y            : Outputs_Vector := (others => 'U');
  signal GIN2A_Y           : Outputs_Vector := (others => 'U');
  signal GIN2B_Y           : Outputs_Vector := (others => 'U');
  signal AIN_Y             : Outputs_Vector := (others => 'U');
  signal BIN_Y             : Outputs_Vector := (others => 'U');
  signal CIN_Y             : Outputs_Vector := (others => 'U');
  signal YOUT              : Outputs_Vector := (others => 'U');

begin

  -- assign pin values to local signals after wire delay

  AIN   <= transport CONVERT_TO_UX01(A)   after TWD_A;
  BIN   <= transport CONVERT_TO_UX01(B)   after TWD_B;
  CIN   <= transport CONVERT_TO_UX01(C)   after TWD_C;
  GIN1  <= transport CONVERT_TO_UX01(G1)  after TWD_G1;
  GIN2A <= transport CONVERT_TO_UX01(G2A) after TWD_G2A;
  GIN2B <= transport CONVERT_TO_UX01(G2B) after TWD_G2B;

  -- Assign local values to output pins

  Y0 <= YOUT(PIN_Y0);
  Y1 <= YOUT(PIN_Y1);
  Y2 <= YOUT(PIN_Y2);
  Y3 <= YOUT(PIN_Y3);
  Y4 <= YOUT(PIN_Y4);
  Y5 <= YOUT(PIN_Y5);
  Y6 <= YOUT(PIN_Y6);
  Y7 <= YOUT(PIN_Y7);

  -- Check for invalid states on control inputs

  assert not (A'event and BIT_UNKNOWN(A))
    report "{" & REF & "} " & STD_ULOGIC_TO_CHAR(A) & 
    " state on A input" severity warning;

  assert not (B'event and BIT_UNKNOWN(B))
    report "{" & REF & "} " & STD_ULOGIC_TO_CHAR(B) & 
    " state on B input" severity warning;

  assert not (C'event and BIT_UNKNOWN(C))
    report "{" & REF & "} " & STD_ULOGIC_TO_CHAR(C) & 
    " state on C input" severity warning;

  assert not (G1'event and BIT_UNKNOWN(G1))
    report "{" & REF & "} " & STD_ULOGIC_TO_CHAR(G1) & 
    " state on G1 input" severity warning;

  assert not (G2A'event and BIT_UNKNOWN(G2A))
    report "{" & REF & "} " & STD_ULOGIC_TO_CHAR(G2A) & 
    " state on G2A input" severity warning;

  assert not (G2B'event and BIT_UNKNOWN(G2B))
    report "{" & REF & "} " & STD_ULOGIC_TO_CHAR(G2B) & 
    " state on G2B input" severity warning;

------------------------------------------------------------------------------

GIN_Y : process (GIN1, GIN2A, GIN2B)

begin

  if (GIN1'EVENT) then
    for OUTPIN in All_Outputs loop
      GIN1_Y(OUTPIN) <= transport GIN1 after TP_DELAY(not GIN1,
                        MODEL_TIMES.TPLH_EN3(OUTPIN),
                        MODEL_TIMES.TPHL_EN3(OUTPIN)); 
    end loop;
  end if;

  if (GIN2A'EVENT) then
    for OUTPIN in All_Outputs loop
      GIN2A_Y(OUTPIN) <= transport GIN2A after TP_DELAY(GIN2A,
                         MODEL_TIMES.TPLH_BS2(OUTPIN),
                         MODEL_TIMES.TPHL_BS2(OUTPIN));                   
    end loop;
  end if;

  if (GIN2B'EVENT) then
    for OUTPIN in All_Outputs loop
      GIN2B_Y(OUTPIN) <= transport GIN2B after TP_DELAY(GIN2B,
                         MODEL_TIMES.TPLH_EN2(OUTPIN),
                         MODEL_TIMES.TPHL_EN2(OUTPIN));
    end loop;
  end if;
end process GIN_Y;

------------------------------------------------------------------------------

IN_Y : process (AIN, BIN, CIN)

begin

  if (AIN'EVENT) then
    AIN_Y(PIN_Y0) <= transport AIN after TP_DELAY(AIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y0),
                     MODEL_TIMES.TPHL_BS2(PIN_Y0));
        
    AIN_Y(PIN_Y1) <= transport not AIN after TP_DELAY(not AIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y1),
                     MODEL_TIMES.TPHL_BS3(PIN_Y1));

    AIN_Y(PIN_Y2) <= transport AIN after TP_DELAY(AIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y2),
                     MODEL_TIMES.TPHL_BS2(PIN_Y2));

    AIN_Y(PIN_Y3) <= transport not AIN after TP_DELAY(not AIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y3),
                     MODEL_TIMES.TPHL_BS3(PIN_Y3));

    AIN_Y(PIN_Y4) <= transport AIN after TP_DELAY(AIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y4),
                     MODEL_TIMES.TPHL_BS2(PIN_Y4));

    AIN_Y(PIN_Y5) <= transport not AIN after TP_DELAY(not AIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y5),
                     MODEL_TIMES.TPHL_BS3(PIN_Y5));

    AIN_Y(PIN_Y6) <= transport AIN after TP_DELAY(AIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y6),
                     MODEL_TIMES.TPHL_BS2(PIN_Y6));

    AIN_Y(PIN_Y7) <= transport not AIN after TP_DELAY(not AIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y7),
                     MODEL_TIMES.TPHL_BS3(PIN_Y7));
  end if;

  if (BIN'EVENT) then
    BIN_Y(PIN_Y0) <= transport BIN after TP_DELAY(BIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y0),
                     MODEL_TIMES.TPHL_BS2(PIN_Y0));

    BIN_Y(PIN_Y1) <= transport BIN after TP_DELAY(BIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y1),
                     MODEL_TIMES.TPHL_BS2(PIN_Y1));

    BIN_Y(PIN_Y2) <= transport not BIN after TP_DELAY(not BIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y2),
                     MODEL_TIMES.TPHL_BS3(PIN_Y2));
    
    BIN_Y(PIN_Y3) <= transport not BIN after TP_DELAY(not BIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y3),
                     MODEL_TIMES.TPHL_BS3(PIN_Y3));

    BIN_Y(PIN_Y4) <= transport BIN after TP_DELAY(BIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y4),
                     MODEL_TIMES.TPHL_BS2(PIN_Y4));

    BIN_Y(PIN_Y5) <= transport BIN after TP_DELAY(BIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y5),
                     MODEL_TIMES.TPHL_BS2(PIN_Y5));

    BIN_Y(PIN_Y6) <= transport not BIN after TP_DELAY(not BIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y6),
                     MODEL_TIMES.TPHL_BS3(PIN_Y6));

    BIN_Y(PIN_Y7) <= transport not BIN after TP_DELAY(not BIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y7),
                     MODEL_TIMES.TPHL_BS3(PIN_Y7));
  end if;

  if (CIN'event) then
    CIN_Y(PIN_Y0) <= transport CIN after TP_DELAY(CIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y0),
                     MODEL_TIMES.TPHL_BS2(PIN_Y0));

    CIN_Y(PIN_Y1) <= transport CIN after TP_DELAY(CIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y1),
                     MODEL_TIMES.TPHL_BS2(PIN_Y1));

    CIN_Y(PIN_Y2) <= transport CIN after TP_DELAY(CIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y2),
                     MODEL_TIMES.TPHL_BS2(PIN_Y2));

    CIN_Y(PIN_Y3) <= transport CIN after TP_DELAY(CIN,
                     MODEL_TIMES.TPLH_BS2(PIN_Y3),
                     MODEL_TIMES.TPHL_BS2(PIN_Y3));

    CIN_Y(PIN_Y4) <= transport not CIN after TP_DELAY(not CIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y4),
                     MODEL_TIMES.TPHL_BS3(PIN_Y4));

    CIN_Y(PIN_Y5) <= transport not CIN after TP_DELAY(not CIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y5),
                     MODEL_TIMES.TPHL_BS3(PIN_Y5));

    CIN_Y(PIN_Y6) <= transport not CIN after TP_DELAY(not CIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y6),
                     MODEL_TIMES.TPHL_BS3(PIN_Y6));

    CIN_Y(PIN_Y7) <= transport not CIN after TP_DELAY(not CIN,
                     MODEL_TIMES.TPLH_BS3(PIN_Y7),
                     MODEL_TIMES.TPHL_BS3(PIN_Y7));
  end if;

end process IN_Y;

------------------------------------------------------------------------------

ASSERTYOUT : process (GIN1_Y, GIN2A_Y, GIN2B_Y, AIN_Y, BIN_Y, CIN_Y)

  variable ENABLE : Std_Ulogic;

begin

    for OUTPIN in All_Outputs loop
      ENABLE := GIN1_Y(OUTPIN) and not GIN2A_Y(OUTPIN) and not GIN2B_Y(OUTPIN);
      YOUT(OUTPIN) <= not (ENABLE and not AIN_Y(OUTPIN) and not
                      BIN_Y(OUTPIN) and not CIN_Y(OUTPIN));
    end loop;

end process ASSERTYOUT;

------------------------------------------------------------------------------

end BEHAVIORAL;

