-- Filename : SN54xx112A_timing.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
library STD_PACK;
  use STD_PACK.SIMFLAG.all;
  use STD_PACK.TTL_TIMING.all;

package SN54xx112A_TIMING is

  type Mode_Type is (ERROR,CLR_MODE,PR_MODE,PASS,SET,RESET,TOGGLE);
  type Speed_Type is (MIN,TYP,MAX);

  type Prop_Times is record 
    TPLH_POC_Q  : Time;
    TPLH_POC_QB : Time;
    TPHL_POC_Q  : Time;
    TPHL_POC_QB : Time;
    TPLH_CLK_Q  : Time;
    TPLH_CLK_QB : Time;
    TPHL_CLK_Q  : Time;
    TPHL_CLK_QB : Time;
  end record;

  type Delay_Type is array (Tech_Type,Speed_Type) of Prop_Times;

  type Cload_Out is record  -- capacitive loadings on output pins
    PIN_Q  : Real ;
    PIN_QB : Real ;
  end record;

  type Constraint_Type is array (Tech_Type) of Time;

  ----------- data book times ----------------

  constant DATASHEET_TIMES : Delay_Type :=
  (
   -- STD family
    (others => (others => 0.0 ns)),

   -- S family
    (  --  minimum times
       ( TPLH_POC_Q  =>  1.4 ns,
         TPLH_POC_QB =>  1.4 ns,
         TPHL_POC_Q  =>  1.8 ns,
         TPHL_POC_QB =>  1.8 ns,
         TPLH_CLK_Q  =>  1.4 ns,
         TPLH_CLK_QB =>  1.4 ns,
         TPHL_CLK_Q  =>  1.8 ns,
         TPHL_CLK_QB =>  1.8 ns
       ),
      --  typical times
       ( TPLH_POC_Q  =>  4.0 ns,
         TPLH_POC_QB =>  4.0 ns,
         TPHL_POC_Q  =>  5.0 ns,
         TPHL_POC_QB =>  5.0 ns,
         TPLH_CLK_Q  =>  4.0 ns,
         TPLH_CLK_QB =>  4.0 ns,
         TPHL_CLK_Q  =>  5.0 ns,
         TPHL_CLK_QB =>  5.0 ns
       ),
      --  maximum times
       ( TPLH_POC_Q  =>  7.0 ns,
         TPLH_POC_QB =>  7.0 ns,
         TPHL_POC_Q  =>  7.0 ns,
         TPHL_POC_QB =>  7.0 ns,
         TPLH_CLK_Q  =>  7.0 ns,
         TPLH_CLK_QB =>  7.0 ns,
         TPHL_CLK_Q  =>  7.0 ns,
         TPHL_CLK_QB =>  7.0 ns
       )
    ),
   -- LS family
    (  --  minimum times
       ( TPLH_POC_Q  =>  5.3 ns,
         TPLH_POC_QB =>  5.3 ns,
         TPHL_POC_Q  =>  5.3 ns,
         TPHL_POC_QB =>  5.3 ns,
         TPLH_CLK_Q  =>  5.3 ns,
         TPLH_CLK_QB =>  5.3 ns,
         TPHL_CLK_Q  =>  5.3 ns,
         TPHL_CLK_QB =>  5.3 ns
       ),
      --  typical times
       ( TPLH_POC_Q  =>  15.0 ns,
         TPLH_POC_QB =>  15.0 ns,
         TPHL_POC_Q  =>  15.0 ns,
         TPHL_POC_QB =>  15.0 ns,
         TPLH_CLK_Q  =>  15.0 ns,
         TPLH_CLK_QB =>  15.0 ns,
         TPHL_CLK_Q  =>  15.0 ns,
         TPHL_CLK_QB =>  15.0 ns
       ),
      --  maximum times
       ( TPLH_POC_Q  =>  20.0 ns,
         TPLH_POC_QB =>  20.0 ns,
         TPHL_POC_Q  =>  20.0 ns,
         TPHL_POC_QB =>  20.0 ns,
         TPLH_CLK_Q  =>  20.0 ns,
         TPLH_CLK_QB =>  20.0 ns,
         TPHL_CLK_Q  =>  20.0 ns,
         TPHL_CLK_QB =>  20.0 ns
       )
    )
  );

-- note:  TW_CLK_LO for LS family is not spec'd, but was derived from
--        30 MHz maximum clock frequency (33.33 ns period)

--                                         STD      S        LS
  constant TW_CLK_HI : Constraint_Type := (0.0 ns, 6.0 ns, 20.0 ns);
  constant TW_CLK_LO : Constraint_Type := (0.0 ns, 6.5 ns, 13.3 ns);
  constant TW_POC_LO : Constraint_Type := (0.0 ns, 8.0 ns, 25.0 ns);
  constant TSU       : Constraint_Type := (0.0 ns, 7.0 ns, 20.0 ns);
  constant TH        : Constraint_Type := (0.0 ns, 0.0 ns, 0.0 ns);

  -------------- function declarations -----------------------

  function CALC_CURVE ( X :  Real; COEFFS :Four_Coeffs ) return Real;

  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times;

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times;

end SN54xx112A_TIMING;

-- ======================================================================

package body SN54xx112A_TIMING is

-- ------------------------------------------------------------
  function CALC_CURVE ( X : Real; COEFFS : Four_Coeffs ) return Real is

    variable O : Real;

  begin
    O := COEFFS(4)*(X)**3 + COEFFS(3)*(X)**2 + COEFFS(2)*(X) + COEFFS(1);
    return O;
  end CALC_CURVE;

-- --------------------GET_TIMING----------------------------------------
  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times is

    variable KVLH,KVHL       : Real;   -- voltage derating factors
    variable KTLH,KTHL       : Real;   -- temperature derating factors

    variable DERATINGLH      : Real;   -- composite derating factors
    variable DERATINGHL      : Real;

    variable MTIME           : Prop_Times;

  begin

    -- Check for SIMFLAG options out-of-range 

    assert not (GEN_TA < -55.0 or GEN_TA > 125.0) report 
      "Temperature out of range" severity error;
    assert not (GEN_VCC < 4.5 or GEN_VCC > 5.5 ) report 
      "Voltage out of range" severity error;
    assert not (GEN_DFACTOR < 0.0 ) report 
       "Negative GEN_DFACTOR not allowed" severity error;

    -- check for illegal family selection

    assert (GEN_FAMILY = S_FAMILY or GEN_FAMILY = LS_FAMILY) report 
        "Only the S and LS families are available" severity error;


    -- calculate derating factors

    KTLH := CALC_CURVE (GEN_TA,   TLH_COEFFICIENT(GEN_FAMILY));
    KTHL := CALC_CURVE (GEN_TA,   THL_COEFFICIENT(GEN_FAMILY));
    KVLH := CALC_CURVE (GEN_VCC,  VLH_COEFFICIENT(GEN_FAMILY));
    KVHL := CALC_CURVE (GEN_VCC,  VHL_COEFFICIENT(GEN_FAMILY));

    -- select delay values

    case GEN_TIME_MODE is
      when MINIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MIN);
      when TYPICAL        => MTIME := DATASHEET_TIMES(GEN_FAMILY,TYP);
      when MAXIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MAX);
    end case;

    -- composite derating factors

    DERATINGLH := KVLH * KTLH * (GEN_DFACTOR / 100.0);
    DERATINGHL := KVHL * KTHL * (GEN_DFACTOR / 100.0);

    -- apply derating factors to propagation delays

    MTIME.TPLH_POC_Q    := MTIME.TPLH_POC_Q  * DERATINGLH;
    MTIME.TPLH_POC_QB   := MTIME.TPLH_POC_QB * DERATINGLH;
    MTIME.TPHL_POC_Q    := MTIME.TPHL_POC_Q  * DERATINGHL;
    MTIME.TPHL_POC_QB   := MTIME.TPHL_POC_QB * DERATINGHL;
    MTIME.TPLH_CLK_Q    := MTIME.TPLH_CLK_Q  * DERATINGLH;
    MTIME.TPLH_CLK_QB   := MTIME.TPLH_CLK_QB * DERATINGLH;
    MTIME.TPHL_CLK_Q    := MTIME.TPHL_CLK_Q  * DERATINGHL;
    MTIME.TPHL_CLK_QB   := MTIME.TPHL_CLK_QB * DERATINGHL;

    return(MTIME);

end GET_TIMING;

-- ----------------------------------------------------------
  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times is

    variable KCLH  : Cload_Out;   -- load capacitance derating factors
    variable KCHL  : Cload_Out;

    variable MTIME : Prop_Times;

  begin

    KCLH.PIN_Q  := CALC_CURVE (GEN_CLOADS.PIN_Q,CLH_COEFFICIENT(GEN_FAMILY));
    KCLH.PIN_QB := CALC_CURVE (GEN_CLOADS.PIN_QB,CLH_COEFFICIENT(GEN_FAMILY));

    KCHL.PIN_Q  := CALC_CURVE (GEN_CLOADS.PIN_Q,CHL_COEFFICIENT(GEN_FAMILY));
    KCHL.PIN_QB := CALC_CURVE (GEN_CLOADS.PIN_QB,CHL_COEFFICIENT(GEN_FAMILY));

    MTIME.TPLH_POC_Q    := CHIP_TIMES.TPLH_POC_Q  * KCLH.PIN_Q;
    MTIME.TPLH_POC_QB   := CHIP_TIMES.TPLH_POC_QB * KCLH.PIN_QB;
    MTIME.TPHL_POC_Q    := CHIP_TIMES.TPHL_POC_Q  * KCHL.PIN_Q;
    MTIME.TPHL_POC_QB   := CHIP_TIMES.TPHL_POC_QB * KCHL.PIN_QB;
    MTIME.TPLH_CLK_Q    := CHIP_TIMES.TPLH_CLK_Q  * KCLH.PIN_Q;
    MTIME.TPLH_CLK_QB   := CHIP_TIMES.TPLH_CLK_QB * KCLH.PIN_QB;
    MTIME.TPHL_CLK_Q    := CHIP_TIMES.TPHL_CLK_Q  * KCHL.PIN_Q;
    MTIME.TPHL_CLK_QB   := CHIP_TIMES.TPHL_CLK_QB * KCHL.PIN_QB;

  return(MTIME);

end BACK_ANNOTATE;

------------------------------------------------------------

end SN54xx112A_TIMING;
