-- Filename : wgenerator_SN54xx109.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
  use STD.TEXTIO.all;

library WAVES_STANDARD;
  use WAVES_STANDARD.WAVES_SYSTEM.all;
  use WAVES_STANDARD.WAVES_INTERFACE.all;

library SN54xx109;
  use SN54xx109.WAVES_DEVICE.all;
  use SN54xx109.WAVES_OBJECTS.all;
  use SN54xx109.WAVES_FRAMES.all;
  use SN54xx109.WAVES_UTILITIES.all;

package WGENERATOR_SN54xx109 is
  procedure SN54xx109_TEST
  (
    signal CONNECT   : inout Waves_Port_List
  );

  procedure SN54xx109_TEST
  (
    signal CONNECT   : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time
  );

end WGENERATOR_SN54xx109;

-- ======================================================================

package body WGENERATOR_SN54xx109 is

  constant I_PINS : Pinset := NEW_PINSET
  (
    (
      TP_CLK1, TP_J1, TP_K1, TP_PR1, TP_CLR1, 
      TP_CLK2, TP_J2, TP_K2, TP_PR2, TP_CLR2
    )
  );

  constant O_PINS : Pinset := ALL_PINS and not I_PINS;

  constant EMPTY_EVENT : Integer := -1;

  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs) are contained within
  --  the procedure, and are analyzed with the VHDL code.  The test vectors
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After 95 ns (95% of period):  Expected outputs set to appropriate values
  --
  --   After 100 ns (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
  procedure SN54xx109_TEST(
    signal CONNECT   : inout Waves_Port_List) is

    constant PERIOD          : Time       := 100 ns;
    constant UNCERTAINTY_PCT : Real       := 0.95;
    constant TDELAY          : Event_Time := ETIME (UNCERTAINTY_PCT * PERIOD);

    constant FSA : Frame_Set_Array := 
      NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS);

    constant DT_BASIS : Delay_Time_Basis :=(FALSE, 0 ns, 0 ns,
                                            TIMED_DELAY, 0, 0, 0);

    variable TD1 : Time_Data   := NEW_TIME_DATA ( FSA );
    variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1);

    constant PULSE      : Time       := 1 ns;
    constant PULSEDELAY : Event_Time := ETIME (UNCERTAINTY_PCT * PULSE);
        
    constant FSA_pulse : FRAME_SET_ARRAY := 
      NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(PULSEDELAY), O_PINS);
                              
    variable TD_PULSE : TIME_DATA := NEW_TIME_DATA ( FSA_PULSE);
    variable TP  : WAVE_TIMING := ((PULSE, DT_BASIS), TD_PULSE);

    --        Order of signals:
    --
    --        (Q1, QB1, Q2, QB2,
    --         CLK1, J1, K1, PR1, CLR1,
    --         CLK2, J2, K2, PR2, CLR2)
    --

    variable NORMAL_OPERATION : Wtime_Slice_List (1 to 15) := (
      (  "UUUU"
     & "UUUUUUUUUU", T1),

      (  "01UU"
     & "---10UUUUU", T1),-- clear

      (  "11UU"
     & "---00UUUUU", T1),-- both asserted

      (  "10UU"
     & "---01UUUUU", T1),-- preset

      (  "10UU"
     & "00001UUUUU", T1),

      (  "10UU"
     & "00011UUUUU", T1),

      (  "01UU"
     & "10011UUUUU", T1),-- reset

      (  "01UU"
     & "01011UUUUU", T1),

      (  "10UU"
     & "11011UUUUU", T1),-- toggle

      (  "10UU"
     & "00111UUUUU", T1),

      (  "10UU"
     & "10111UUUUU", T1),-- pass

      (  "10UU"
     & "01011UUUUU", T1),

      (  "01UU"
     & "11011UUUUU", T1),-- toggle

      (  "01UU"
     & "01111UUUUU", T1),

      (  "10UU"
     & "11111UUUUU", T1)-- set
      );



    variable PULSE_WIDTH_VIOLATIONS : Wtime_Slice_List (1 to 11) := (
      (  "10UU"
     & "01111UUUUU", T1),

      (  "XXUU"
     & "P1111UUUUU", T1),-- CLK pulse width violation (tw_clk_hi) 

      (  "10UU"
     & "11101UUUUU", T1),-- preset

      (  "XXUU"
     & "N1111UUUUU", T1),-- CLK pulse width violation (tw_clk_lo)

      (  "01UU"
     & "11110UUUUU", T1),-- clear

      (  "01UU"
     & "11111UUUUU", T1),

      (  "XXUU"
     & "111N1UUUUU", T1),-- PR pulse width violation (tw_poc_lo)

      (  "10UU"
     & "11101UUUUU", T1),-- preset

      (  "10UU"
     & "11111UUUUU", T1),

      (  "XXUU"
     & "1111NUUUUU", T1),-- CLR pulse width violation (tw_poc_lo)

      (  "XXUU"
     & "11111UUUUU", T1)
      );



    variable SETUP_VIOLATIONS : Wtime_Slice_List (1 to 6) := (
      (  "10UU"
     & "01101UUUUU", T1),-- preset

      (  "10UU"
     & "00111UUUUU", TP),-- change J

      (  "XXUU"
     & "10111UUUUU", T1),-- J setup violation

      (  "10UU"
     & "01101UUUUU", T1),-- preset

      (  "10UU"
     & "01011UUUUU", TP),-- change K

      (  "XXUU"
     & "11011UUUUU", T1)-- K setup violation
      );



    variable HOLD_VIOLATIONS : Wtime_Slice_List (1 to 8) := (
      (  "10UU"
     & "01101UUUUU", T1),-- preset

      (  "10UU"
     & "01111UUUUU", T1),

      (  "10UU"
     & "11111UUUUU", TP),-- clock

      (  "XXUU"
     & "10111UUUUU", T1),-- J hold violation

      (  "10UU"
     & "01101UUUUU", T1),

      (  "10UU"
     & "01111UUUUU", T1),

      (  "10UU"
     & "11111UUUUU", TP),-- clock

      (  "XXUU"
     & "11011UUUUU", T1)-- K hold violation
      );



    variable UNKNOWNS_ON_INPUTS : Wtime_Slice_List (1 to 35) := (
      (  "10UU"
     & "01101UUUUU", T1),-- preset

      (  "1XUU"
     & "0110XUUUUU", T1),-- unknown on CLR

      (  "01UU"
     & "01110UUUUU", T1),-- clear

      (  "X1UU"
     & "011X0UUUUU", T1),-- unknown on PR

      (  "01UU"
     & "01110UUUUU", T1),-- clear

      (  "01UU"
     & "01111UUUUU", T1),

      (  "01UU"
     & "0111XUUUUU", T1),-- unknown on CLR (no effect)

      (  "10UU"
     & "01101UUUUU", T1),-- preset

      (  "10UU"
     & "01111UUUUU", T1),

      (  "10UU"
     & "011X1UUUUU", T1),-- unknown on PR (no effect)

      (  "10UU"
     & "01101UUUUU", T1),-- preset

      (  "10UU"
     & "01111UUUUU", T1),

      (  "XXUU"
     & "0111XUUUUU", T1),-- unknown on CLR

      (  "01UU"
     & "01110UUUUU", T1),-- clear

      (  "01UU"
     & "01111UUUUU", T1),

      (  "XXUU"
     & "011X1UUUUU", T1),-- unknown on PR

      (  "01UU"
     & "01110UUUUU", T1),-- clear

      (  "01UU"
     & "01111UUUUU", T1),

      (  "XXUU"
     & "X1111UUUUU", T1),-- unknown on CLK

      (  "10UU"
     & "01101UUUUU", T1),-- preset

      (  "10UU"
     & "01111UUUUU", T1),

      (  "10UU"
     & "X1111UUUUU", T1),-- unknown on CLK (no effect)

      (  "01UU"
     & "01110UUUUU", T1),-- clear

      (  "01UU"
     & "0X111UUUUU", T1),-- unknown on J

      (  "XXUU"
     & "1X111UUUUU", T1),-- clock

      (  "10UU"
     & "01101UUUUU", T1),-- preset

      (  "10UU"
     & "0X111UUUUU", T1),-- unknown on J

      (  "10UU"
     & "1X111UUUUU", T1),-- clock (no effect)

      (  "10UU"
     & "00101UUUUU", T1),-- preset

      (  "10UU"
     & "00X11UUUUU", T1),-- unknown on K

      (  "XXUU"
     & "10X11UUUUU", T1),-- clock

      (  "01UU"
     & "00110UUUUU", T1),-- clear

      (  "01UU"
     & "00X11UUUUU", T1),-- unknown on K

      (  "01UU"
     & "10X11UUUUU", T1),-- clock (no effect)

      (  "01UU"
     & "11111UUUUU", T1)
      );



  begin
    assert FALSE
      report "NORMAL_OPERATION vectors"
        severity note;
      for I in NORMAL_OPERATION' range loop
        APPLY(CONNECT,NORMAL_OPERATION(I).CODES,NORMAL_OPERATION(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "PULSE_WIDTH_VIOLATIONS vectors"
        severity note;
      for I in PULSE_WIDTH_VIOLATIONS' range loop
        APPLY(CONNECT,PULSE_WIDTH_VIOLATIONS(I).CODES,PULSE_WIDTH_VIOLATIONS(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "SETUP_VIOLATIONS vectors"
        severity note;
      for I in SETUP_VIOLATIONS' range loop
        APPLY(CONNECT,SETUP_VIOLATIONS(I).CODES,SETUP_VIOLATIONS(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "HOLD_VIOLATIONS vectors"
        severity note;
      for I in HOLD_VIOLATIONS' range loop
        APPLY(CONNECT,HOLD_VIOLATIONS(I).CODES,HOLD_VIOLATIONS(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "UNKNOWNS_ON_INPUTS vectors"
        severity note;
      for I in UNKNOWNS_ON_INPUTS' range loop
        APPLY(CONNECT,UNKNOWNS_ON_INPUTS(I).CODES,UNKNOWNS_ON_INPUTS(I).WTIME,ALL_PINS);
      end loop;

    end;

  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs, along with the period
  --  of each slice) are read in from an external file.  The test vectors 
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After <START_TOLERANCE>:  Expected outputs set to appropriate values
  --
  --   After <FS_TIME - END_TOLERANCE>: Get next set of inputs and outputs
  --
  --   After <FS_TIME> (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
  procedure SN54xx109_Test
  (
    signal CONNECT   : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time
  ) is

    variable TOLERANCE : Time:= START_TOLERANCE + END_TOLERANCE;
    constant FSA : Frame_Set_Array :=
      NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY (OUTPUT_FRAMES, O_PINS);

    constant DT_BASIS : Delay_Time_Basis :=
    (
      FALSE, 0 ns, 0 ns, TIMED_DELAY, 0, 0, 0
    );

    variable TD : Time_Data := NEW_TIME_DATA (FSA);
    constant FSA_DONT_CARE : Frame_Set_Array :=
      NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY (DONT_CARE_FRAMES, O_PINS);

    variable TD_DONT_CARE : Time_Data := NEW_TIME_DATA (FSA_DONT_CARE);
    file INPUT_FILE : Text is in INPUT_FILE_NAME;

    -- the file slice must be allocated
    variable SN54xx109_FILE_SLICE : File_Slice := NEW_FILE_SLICE;

  begin
   loop
      READ_FILE_SLICE(INPUT_FILE, SN54xx109_FILE_SLICE);
      exit when SN54xx109_FILE_SLICE.END_OF_FILE;
      if SN54xx109_FILE_SLICE.FS_TIME > TOLERANCE then
        APPLY(CONNECT,SN54xx109_FILE_SLICE.CODES.all,DELAY(START_TOLERANCE),
           TD_DONT_CARE);
        APPLY(CONNECT, SN54xx109_FILE_SLICE.CODES.all,
          DELAY(SN54xx109_FILE_SLICE.FS_TIME - TOLERANCE), TD);
        APPLY(CONNECT, SN54xx109_FILE_SLICE.CODES.all,
          DELAY(END_TOLERANCE), TD_DONT_CARE);
      else
        APPLY(CONNECT, SN54xx109_FILE_SLICE.CODES.all,
          DELAY(SN54xx109_FILE_SLICE.FS_TIME), TD);
      end if;
    end loop;
  end;

end WGENERATOR_SN54xx109;




