-- Filename : SN54xx109_timing.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
library STD_PACK;
  use STD_PACK.SIMFLAG.all;    -- Global simulation options
  use STD_PACK.TTL_TIMING.all; -- TTL derating coefficients

package SN54xx109_TIMING is 

  type Mode_Type is (ERROR,CLR_MODE,PR_MODE,PASS,SET,RESET,TOGGLE);
  type Speed_Type is (MIN,TYP,MAX);

  type Prop_Times is record
    TPLH_PRE_Q  : Time;
    TPHL_PRE_QB : Time;
    TPLH_CLR_QB : Time;
    TPHL_CLR_Q  : Time;
    TPLH_CLK_Q  : Time;
    TPLH_CLK_QB : Time;
    TPHL_CLK_Q  : Time;
    TPHL_CLK_QB : Time;
  end record;

  type Delay_Type is array (Tech_Type,Speed_Type) of Prop_Times;

  type Cload_Out is record  -- capacitive loadings on output pins
    PIN_Q  : Real ;
    PIN_QB : Real ;
  end record;

  type Constraint_Type is array (Tech_Type) of Time;

----------- data book times ----------------

  constant DATASHEET_TIMES : Delay_Type :=
  (
   -- STD family
    (  --  minimum times
       ( TPLH_PRE_Q  => 3.5 ns,
         TPHL_PRE_QB => 8.1 ns,
         TPLH_CLR_QB => 3.5 ns,
         TPHL_CLR_Q  => 6.0 ns,
         TPLH_CLK_Q  => 3.5 ns,
         TPLH_CLK_QB => 3.5 ns,
         TPHL_CLK_Q  => 6.3 ns,
         TPHL_CLK_QB => 6.3 ns
       ),
      --  typical times
       ( TPLH_PRE_Q  => 10.0 ns,
         TPHL_PRE_QB => 23.0 ns,
         TPLH_CLR_QB => 10.0 ns,
         TPHL_CLR_Q  => 17.0 ns,
         TPLH_CLK_Q  => 10.0 ns,
         TPLH_CLK_QB => 10.0 ns,
         TPHL_CLK_Q  => 18.0 ns,
         TPHL_CLK_QB => 18.0 ns
       ),
      --  maximum times
       ( TPLH_PRE_Q  => 15.0 ns,
         TPHL_PRE_QB => 35.0 ns,
         TPLH_CLR_QB => 15.0 ns,
         TPHL_CLR_Q  => 25.0 ns,
         TPLH_CLK_Q  => 16.0 ns,
         TPLH_CLK_QB => 16.0 ns,
         TPHL_CLK_Q  => 28.0 ns,
         TPHL_CLK_QB => 28.0 ns
       )
    ),

   -- S family
    (others => (others => 0.0 ns)),

   -- LS family
    (others => (others => 0.0 ns))
  );

--                                         STD      S        LS
  constant TW_CLK    : Constraint_Type := (20.0 ns, 0.0 ns, 0.0 ns);
  constant TW_POC_LO : Constraint_Type := (20.0 ns, 0.0 ns, 0.0 ns);
  constant TSU       : Constraint_Type := (10.0 ns, 0.0 ns, 0.0 ns);
  constant TH        : Constraint_Type := ( 6.0 ns, 0.0 ns, 0.0 ns);

  ----------------------function declarations--------------------

  function CALC_CURVE ( X :  Real; COEFFS :Four_Coeffs ) return Real;

  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times;

  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times;

end SN54xx109_TIMING;

-- ======================================================================

package body SN54xx109_TIMING is

-- ------------------------------------------------------------
  function CALC_CURVE ( X : Real; COEFFS : Four_Coeffs ) return Real is

    variable O : Real;

  begin
    O := COEFFS(4)*(X)**3 + COEFFS(3)*(X)**2 + COEFFS(2)*(X) + COEFFS(1);
    return O;
  end CALC_CURVE;

-- ------------------------------------------------------------
  function GET_TIMING (GEN_FAMILY     : Tech_Type;
                       GEN_TIME_MODE  : Time_Modes;
                       GEN_VCC        : Real;
                       GEN_TA         : Real;
                       GEN_DFACTOR    : Real) return Prop_Times is

    variable KVLH,KVHL       : Real;   -- voltage derating factors
    variable KTLH,KTHL       : Real;   -- temperature derating factors

    variable DERATINGLH      : Real;   -- composite derating factors
    variable DERATINGHL      : Real;

    variable MTIME           : Prop_Times;
  
  begin

    -- Check for SIMFLAG options out-of-range 

    assert not (GEN_TA < -55.0 or GEN_TA > 125.0) report 
      "Temperature out of range" severity error;
    assert not (GEN_VCC < 4.5 or GEN_VCC > 5.5 ) report 
      "Voltage out of range" severity error;
    assert not (GEN_DFACTOR < 0.0 ) report 
       "Negative GEN_DFACTOR not allowed" severity error;

    -- check for illegal family selection

    assert (GEN_FAMILY = STD_FAMILY) report 
        "Only the STD family is available" severity error;

    -- calculate derating factors

    KTLH := CALC_CURVE (GEN_TA,   TLH_COEFFICIENT(GEN_FAMILY));
    KTHL := CALC_CURVE (GEN_TA,   THL_COEFFICIENT(GEN_FAMILY));
    KVLH := CALC_CURVE (GEN_VCC,  VLH_COEFFICIENT(GEN_FAMILY));
    KVHL := CALC_CURVE (GEN_VCC,  VHL_COEFFICIENT(GEN_FAMILY));

    -- select delay values

    case GEN_TIME_MODE is
      when MINIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MIN);
      when TYPICAL        => MTIME := DATASHEET_TIMES(GEN_FAMILY,TYP);
      when MAXIMUM        => MTIME := DATASHEET_TIMES(GEN_FAMILY,MAX);
    end case;

    -- composite derating factors

    DERATINGLH := KVLH * KTLH * (GEN_DFACTOR / 100.0);
    DERATINGHL := KVHL * KTHL * (GEN_DFACTOR / 100.0);

    -- apply derating factors to propagation delays

    MTIME.TPLH_PRE_Q    := MTIME.TPLH_PRE_Q  * DERATINGLH;
    MTIME.TPHL_PRE_QB   := MTIME.TPHL_PRE_QB * DERATINGHL;
    MTIME.TPLH_CLR_QB   := MTIME.TPLH_CLR_QB * DERATINGLH;
    MTIME.TPHL_CLR_Q    := MTIME.TPHL_CLR_Q  * DERATINGHL;
    MTIME.TPLH_CLK_Q    := MTIME.TPLH_CLK_Q  * DERATINGLH;
    MTIME.TPLH_CLK_QB   := MTIME.TPLH_CLK_QB * DERATINGLH;
    MTIME.TPHL_CLK_Q    := MTIME.TPHL_CLK_Q  * DERATINGHL;
    MTIME.TPHL_CLK_QB   := MTIME.TPHL_CLK_QB * DERATINGHL;

    return(MTIME);

  end GET_TIMING;

-- ----------------------------------------------------------
  function BACK_ANNOTATE (CHIP_TIMES : Prop_Times;
                          GEN_FAMILY : Tech_Type;
                          GEN_CLOADS : Cload_Out) return Prop_Times is

    variable KCLH  : Cload_Out;   -- load capacitance derating factors
    variable KCHL  : Cload_Out;

    variable MTIME : Prop_Times;

  begin

    KCLH.PIN_Q  := CALC_CURVE (GEN_CLOADS.PIN_Q,CLH_COEFFICIENT(GEN_FAMILY));
    KCLH.PIN_QB := CALC_CURVE (GEN_CLOADS.PIN_QB,CLH_COEFFICIENT(GEN_FAMILY));

    KCHL.PIN_Q  := CALC_CURVE (GEN_CLOADS.PIN_Q,CHL_COEFFICIENT(GEN_FAMILY));
    KCHL.PIN_QB := CALC_CURVE (GEN_CLOADS.PIN_QB,CHL_COEFFICIENT(GEN_FAMILY));

    MTIME.TPLH_PRE_Q    := CHIP_TIMES.TPLH_PRE_Q  * KCLH.PIN_Q;
    MTIME.TPHL_PRE_QB   := CHIP_TIMES.TPHL_PRE_QB * KCHL.PIN_QB;
    MTIME.TPLH_CLR_QB   := CHIP_TIMES.TPLH_CLR_QB * KCLH.PIN_QB;
    MTIME.TPHL_CLR_Q    := CHIP_TIMES.TPHL_CLR_Q  * KCHL.PIN_Q;
    MTIME.TPLH_CLK_Q    := CHIP_TIMES.TPLH_CLK_Q  * KCLH.PIN_Q;
    MTIME.TPLH_CLK_QB   := CHIP_TIMES.TPLH_CLK_QB * KCLH.PIN_QB;
    MTIME.TPHL_CLK_Q    := CHIP_TIMES.TPHL_CLK_Q  * KCHL.PIN_Q;
    MTIME.TPHL_CLK_QB   := CHIP_TIMES.TPHL_CLK_QB * KCHL.PIN_QB;

  return(MTIME);

end BACK_ANNOTATE;

------------------------------------------------------------

end SN54xx109_TIMING;
