-- Filename : SN54xx109.vhd
--
-- Version 2.1
--
--
--                                   NOTICE
--
--      LICENSEE acknowledges that it has read and executed the
--      Institute for Technology Development, VHDL Standard Component
--      Library Software Standard License Agreement.  Licensee may not
--      sublicense, assign or transfer the Software, the Documentation or its
--      rights under the Agreement.
--      
--      The Agreement authorizes Licensee to use at any one time only
--      one copy of the Software (either the original Software or the
--      Archival Copy but never both.)  Licensee may make one (1) copy
--      of the Software (the "Archival Copy") and one (1) copy of the
--      Documentation for archival purposes.  Licensee may not otherwise copy,
--      modify, transfer or disclose in any manner the Software or
--      Documentation.  Licensee shall store the Archival Copy in a secure
--      location to which access shall be restricted.  Licensee shall maintain
--      a written record of the data on which the Archival Copy is created and
--      its location.
--      
--      Licensee expressly recognizes that the Software is the proprietary
--      property of ITD.  Licensee warrants that it will not take any action
--      which would result in the impairment or loss of proprietary rights of
--      ITD in the Software.  Licensee warrants that the Software will
--      not be incorporated into any Licensee product.  Licensee warrants 
--      that it will not decompile, disassemble or reverse engineer the 
--      Software or attempt to do so for other than internal applications.
--      Licensee warrants that it will never divulge to any person without the
--      prior written consent of ITD the Software source code or Software
--      object code.
--      
--      The Software is provided "AS IS" WITHOUT WARRANTY of any kind, either
--      express or implied, including but not limited to implied warranties of
--      merchantability and fitness for a particular purpose.
--      
--      RIGHT TO ACCESS OF THIS SOFTWARE IS DENIED IF THIS NOTICE IS REMOVED.
--      
--
--        ------------------------------------------------------------------

  use STD.TEXTIO.ALL;

library IEEE;                        -- Logic system
  use IEEE.STD_LOGIC_1164.all;    -- Defines logic types, operators, functions

library STD_PACK;                   -- Standard library for all models
  use STD_PACK.MISC_FUNC.all;       -- Miscellaneous functions package
  use STD_PACK.TIME_FUNC.all;       -- Timing functions package
  use STD_PACK.TTL_TIMING.all;      -- TTL derating coefficients

library SN54xx109;
  use SN54xx109.SN54xx109_TIMING.ALL;    -- timing module

entity SN54xx109 is

  generic 
  (
    GEN_PROP_TIMES  : Prop_Times ;
    GEN_XGEN        : Boolean ;
    GEN_TIMING_MESG : Boolean ;
    GEN_FAMILY      : Tech_Type ;

    GEN_TW_CLK     : Time ;
    GEN_TW_POC_LO  : Time ;
    GEN_TSU        : Time ;
    GEN_TH         : Time ;

    TWD_CLK : Time ;
    TWD_J   : Time ;
    TWD_K   : Time ;
    TWD_PR  : Time ;
    TWD_CLR : Time ;

    CL_Q    : Real ;
    CL_QB   : Real ;

    REF     : String
  );

  port
  (
    Q      : out Std_Ulogic;
    QB     : out Std_Ulogic;
    CLK    : in  Std_Ulogic;
    J      : in  Std_Ulogic;
    K      : in  Std_Ulogic;
    PR     : in  Std_Ulogic;
    CLR    : in  Std_Ulogic
  );

end SN54xx109;

-- ======================================================================

architecture BEHAVIORAL of SN54xx109 is

  constant CL_OUT : Cload_Out := (
    PIN_Q  => CL_Q,
    PIN_QB => CL_QB
  );

  constant MODEL_TIMES : Prop_Times := 
    BACK_ANNOTATE(GEN_PROP_TIMES,GEN_FAMILY,CL_OUT) ;
 
  -- Local signal declarations

  signal CLOCK,JIN,KIN,PRESET,CLEAR  : UX01  := 'U';
  signal QOUT, QBOUT                 : UX01Z := 'U';

  signal VIOLATION   : Resolved_Boolean := FALSE;
  signal CLOCK_LAST  : Time := 0 ns;
  signal PRESET_LAST : Time := 0 ns;
  signal CLEAR_LAST  : Time := 0 ns;

begin

  -- assign pin values to local signals after wire delay

  CLOCK  <= transport CONVERT_TO_UX01(CLK) after TWD_CLK;
  KIN    <= transport CONVERT_TO_UX01(K)   after TWD_K;
  JIN    <= transport CONVERT_TO_UX01(J)   after TWD_J;
  PRESET <= transport CONVERT_TO_UX01(PR)  after TWD_PR;
  CLEAR  <= transport CONVERT_TO_UX01(CLR) after TWD_CLR;

  -- Assign local values to output pins

  Q  <=  QOUT;
  QB <=  QBOUT;

  -- Check for invalid inputs on control input ports

  assert not (CLK'event and
    ((CLK = 'U') or (CLK = 'X') or (CLK = 'Z') or (CLK = 'W') or (CLK = '-'))) 
    report "{" & REF & "} " & 
    STD_ULOGIC_TO_CHAR(CLK) & " state on CLK input" severity warning;

  assert not (PR'event and 
    ((PR = 'U') or (PR = 'X') or (PR = 'Z') or (PR = 'W') or (PR = '-'))) 
    report "{" & REF & "} " & 
    STD_ULOGIC_TO_CHAR(PR) & " state on PR input" severity warning;

  assert not (CLR'event and 
    ((CLR = 'U') or (CLR = 'X') or (CLR = 'Z') or (CLR = 'W') or (CLR = '-')))
    report "{" & REF & "} " & 
    STD_ULOGIC_TO_CHAR(CLR) & " state on CLR input" severity warning;

  assert not ((CLR = '0' or CLR = 'L') and (PR = '0' or PR = 'L')) 
    report "{" & REF & "} CLR and PR both asserted" severity warning;

---------------------------------------------------------------------------------  

  LAST_CLOCK : process (CLOCK)
  begin
    CLOCK_LAST  <= NOW;
  end process LAST_CLOCK;

---------------------------------------------------------------------------------  

  LAST_PRESET : process(PRESET)
  begin
    PRESET_LAST <= NOW;
  end process LAST_PRESET;

---------------------------------------------------------------------------------

  LAST_CLEAR : process (CLEAR)
  begin
    CLEAR_LAST  <= NOW;
  end process LAST_CLEAR;

---------------------------------------------------------------------------------

  CLK_PULSE_CHECK : process

    variable CLK_VIOLATE : Boolean;
    variable TIMESTRING  : Line;

  begin

    wait on CLOCK until ((PRESET = '1') and (CLEAR = '1')
      and (RISING_EDGE(CLOCK) or FALLING_EDGE(CLOCK)));

    if CLOCK = '0' then
      CLK_VIOLATE := (NOW - CLOCK_LAST) < GEN_TW_CLK;
      write (TIMESTRING,NOW);
      assert not (GEN_TIMING_MESG and CLK_VIOLATE)
        report "{" & REF & "} pulse check warning: CLK - HIGH at time"
        & TIMESTRING.all severity warning;
      deallocate (TIMESTRING);

    else
      CLK_VIOLATE :=  (NOW - CLOCK_LAST) < GEN_TW_CLK;
      write (TIMESTRING,NOW);
      assert not (GEN_TIMING_MESG and CLK_VIOLATE)
        report "{" & REF & "} pulse check warning:  CLK - LOW at time "
        & TIMESTRING.all severity warning;
      deallocate (TIMESTRING);
    end if;

    VIOLATION <= CLK_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process CLK_PULSE_CHECK;

-------------------------------------------------------------------------

  CLR_PULSE_CHECK : process

    variable CLR_VIOLATE : Boolean;
    variable TIMESTRING  : Line;

  begin

    wait until (RISING_EDGE(CLEAR));

    CLR_VIOLATE := (NOW - CLEAR_LAST) < GEN_TW_POC_LO;
    write (TIMESTRING,NOW);
    assert not (GEN_TIMING_MESG and CLR_VIOLATE)
      report "{" & REF & "} pulse check warning:  CLR - LOW at time "
      & TIMESTRING.all severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= CLR_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process CLR_PULSE_CHECK;

-------------------------------------------------------------------------

  PR_PULSE_CHECK : process

    variable PR_VIOLATE : Boolean;
    variable TIMESTRING : Line;

  begin

    wait until (RISING_EDGE(PRESET));

    PR_VIOLATE := (NOW - PRESET_LAST) < GEN_TW_POC_LO;
    write (TIMESTRING,NOW);
    assert not (GEN_TIMING_MESG and PR_VIOLATE)
      report "{" & REF & "} pulse check warning:  PR - LOW at time "
      & TIMESTRING.ALL severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= PR_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process PR_PULSE_CHECK;

-------------------------------------------------------------------------

  JK_SETUP_CHECK : process

    variable SETUPJIN_VIOLATE : Boolean;
    variable SETUPKIN_VIOLATE : Boolean;
    variable TIMESTRING       : Line;

  begin

    wait on CLOCK until ((PRESET = '1') and (CLEAR = '1') and RISING_EDGE(CLOCK));

    SETUPJIN_VIOLATE := not JIN'stable(GEN_TSU);
    write (TIMESTRING,NOW);
    assert not (GEN_TIMING_MESG and SETUPJIN_VIOLATE)
      report "{" & REF & "} SETUP CHECK WARNING:  J at time "
      & TIMESTRING.ALL severity warning;
    deallocate (TIMESTRING);

    SETUPKIN_VIOLATE := not KIN'stable(GEN_TSU);
    write (TIMESTRING,NOW);
    assert not (GEN_TIMING_MESG and SETUPKIN_VIOLATE)
      report "{" & REF & "} SETUP CHECK WARNING:  K at time "
      & TIMESTRING.ALL severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= (SETUPJIN_VIOLATE or SETUPKIN_VIOLATE) and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process JK_SETUP_CHECK;

-------------------------------------------------------------------------

  J_HOLD_CHECK : process

    variable HOLD_JIN_VIOLATE : Boolean;
    variable TIMESTRING       : Line;

  begin

    wait on JIN until ((PRESET = '1') and (CLEAR = '1')
      and (CLOCK'Last_Value = '0') and (CLOCK = '1'));

    HOLD_JIN_VIOLATE := not CLOCK'stable (GEN_TH);
    write (TIMESTRING,NOW);
    assert not(GEN_TIMING_MESG and HOLD_JIN_VIOLATE)
      report "{" & REF & "} HOLD CHECK WARNING:  J at time "
      & TIMESTRING.ALL severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= HOLD_JIN_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process J_HOLD_CHECK;

-------------------------------------------------------------------------

  K_HOLD_CHECK : process

    variable HOLD_KIN_VIOLATE : Boolean;
    variable TIMESTRING       : Line;

  begin

    wait on KIN until ((PRESET = '1') and (CLEAR = '1')
      and (CLOCK'Last_Value = '0') and (CLOCK = '1'));

    HOLD_KIN_VIOLATE := not CLOCK'stable(GEN_TH);
    write (TIMESTRING,NOW);
    assert not(GEN_TIMING_MESG and HOLD_KIN_VIOLATE)
      report "{" & REF & "} HOLD CHECK WARNING:  K at time "
      & TIMESTRING.ALL severity warning;
    deallocate (TIMESTRING);

    VIOLATION <= HOLD_KIN_VIOLATE and GEN_XGEN;
    wait for 0 ns;
    VIOLATION <= FALSE;

  end process K_HOLD_CHECK;

-------------------------------------------------------------------------

  JK : process(CLEAR, PRESET, CLOCK, VIOLATION) 

    variable FFMODE : Mode_Type := PASS;

  begin

  -- Select mode

      if (VIOLATION) then
        FFMODE := ERROR;
      elsif (CLEAR = '0') then
        FFMODE := CLR_MODE;
      elsif (PRESET = '0') then
        FFMODE := PR_MODE;
      elsif ((BIT_UNKNOWN(CLEAR) and QOUT /= '0') 
             or (BIT_UNKNOWN(PRESET) and QOUT /= '1')) then
        FFMODE := ERROR;
      elsif (JIN = '0' and KIN = '1' and CLEAR = '1' and PRESET = '1') then
        FFMODE := PASS;
      elsif (JIN = '1' and KIN = '1' and CLEAR = '1' and PRESET = '1') then
        FFMODE := SET;
      elsif (JIN = '0' and KIN = '0' and CLEAR = '1' and PRESET = '1') then
        FFMODE := RESET;
      elsif (JIN = '1' and KIN = '0' and CLEAR = '1' and PRESET = '1') then
        FFMODE := TOGGLE;              
      elsif (BIT_UNKNOWN(JIN) and (RISING_EDGE(CLOCK) or BIT_UNKNOWN(CLOCK))) then
         if not (KIN = '1' and QOUT = '1') then
          FFMODE := ERROR;
        end if;
      elsif (BIT_UNKNOWN(KIN) and (RISING_EDGE(CLOCK) or BIT_UNKNOWN(CLOCK))) then
        if not (JIN = '0' and QOUT = '0') then
          FFMODE := ERROR;
        end if;
      else
        FFMODE := PASS;
      end if;

  -- operation depends on mode

      case FFMODE is
        when ERROR =>
          if(CLEAR = 'U' or PRESET = 'U' or CLOCK = 'U' or JIN = 'U' 
             or KIN = 'U') then
            QOUT <= 'U';
            QBOUT <= 'U';         
          else
            QOUT <= 'X';
            QBOUT <= 'X';         
          end if;
  
        when CLR_MODE => 
          QBOUT <= transport '1' after MODEL_TIMES.TPLH_CLR_QB;
          if (PRESET = '1') then
            QOUT  <= transport '0' after MODEL_TIMES.TPHL_CLR_Q;
          elsif (PRESET = '0') then
            QOUT  <= transport '1' after MODEL_TIMES.TPLH_PRE_Q;
          elsif(PRESET = 'U' or CLOCK = 'U' or JIN = 'U' or KIN = 'U') then
            QOUT <= 'U';
          else
            QOUT  <= 'X';
          end if;
        when PR_MODE => 
          QOUT  <= transport '1' after MODEL_TIMES.TPLH_PRE_Q ;

          if (CLEAR = '1') then
            QBOUT <= transport '0' after MODEL_TIMES.TPHL_PRE_QB;

          elsif (CLEAR = '0') then
            QBOUT <= transport '1' after MODEL_TIMES.TPLH_CLR_QB;

          elsif(CLEAR = 'U' or CLOCK = 'U' or JIN = 'U' or KIN = 'U') then
            QBOUT <= 'U';
          else
            QBOUT <= 'X';
          end if;
        when PASS => null;   -- do nothing
        when SET =>if (RISING_EDGE(CLOCK)) then
          QOUT  <= transport '1' after MODEL_TIMES.TPLH_CLK_Q;
          QBOUT <= transport '0' after MODEL_TIMES.TPHL_CLK_QB;

          elsif(CLOCK = 'U' or (CLOCK = 'X' and (JIN = 'U' or KIN = 'U'))) then
            if (QOUT /= '1') then
              QOUT <= 'U';
            end if;
            if (QBOUT /= '0') then
              QBOUT <= 'U';
            end if;
            elsif (CLOCK = 'X') then
            if (QOUT /= '1') then
              QOUT <= 'X';
            end if;
            if (QBOUT /= '0') then
              QBOUT <= 'X';
            end if;
          end if;
        when RESET =>
          if (RISING_EDGE(CLOCK)) then
            QOUT  <= transport '0' after MODEL_TIMES.TPHL_CLK_Q;
            QBOUT <= transport '1' after MODEL_TIMES.TPLH_CLK_QB;

          elsif(CLOCK = 'U' or(CLOCK = 'X' and (JIN = 'U' or KIN = 'U'))) then
            if (QOUT /= '0') then
              QOUT <= 'U';
            end if;
            if (QBOUT /= '1') then
              QBOUT <= 'U';
            end if;
            elsif (CLOCK = 'X') then
              if (QOUT /= '0') then
                QOUT <= 'X';
              end if;
              if (QBOUT /= '1') then
                QBOUT <= 'X';
              end if;
            end if;
        when TOGGLE =>
          if (RISING_EDGE(CLOCK)) then
            if (QOUT = '1') then
              QOUT  <= transport '0' after MODEL_TIMES.TPHL_CLK_Q;
              QBOUT <= transport '1' after MODEL_TIMES.TPLH_CLK_QB;

            elsif (QOUT = '0') then
              QOUT  <= transport '1' after MODEL_TIMES.TPLH_CLK_Q; 
              QBOUT <= transport '0' after MODEL_TIMES.TPHL_CLK_QB;

            elsif (QOUT = 'U') then
              QOUT  <= 'U';
              QBOUT <= 'U';
            else
              QOUT  <= 'X';
              QBOUT <= 'X';
            end if;
            elsif (CLOCK = 'U') then
              QOUT  <= 'U';
              QBOUT <= 'U';
            elsif (CLOCK = 'X') then
              QOUT  <= 'X';
              QBOUT <= 'X';
            end if;
      end case;

      if(CLEAR = 'U' or PRESET = 'U' or CLOCK = 'U' 
            or J = 'U' or K = 'U') then
        QOUT <= 'U';
        QBOUT <= 'U';
      else
        QOUT <= 'X';
        QBOUT <= 'X';
      end if;
    
  end process JK;

-------------------------------------------------------------------------

end BEHAVIORAL;
    
