-- Filename : wgenerator_SN54xx04.vhd
--
-- Version 2.1
----------------------------------------------------------------------
--
use STD.TEXTIO.all;

library WAVES_STANDARD;
  use WAVES_STANDARD.WAVES_SYSTEM.all;
  use WAVES_STANDARD.WAVES_INTERFACE.all;

library SN54xx04;
  use SN54xx04.WAVES_DEVICE.all;
  use SN54xx04.WAVES_OBJECTS.all;
  use SN54xx04.WAVES_FRAMES.all;
  use SN54xx04.WAVES_UTILITIES.all;

package WGENERATOR_SN54xx04 is
  procedure SN54xx04_TEST
  (
    signal CONNECT            : inout Waves_Port_List
  );

  procedure SN54xx04_TEST
  (
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time 
  );

end WGENERATOR_SN54xx04;

-- ======================================================================

package body WGENERATOR_SN54xx04 is

  constant I_PINS : Pinset := NEW_PINSET
  ( 
    (TP_A1, TP_A2, TP_A3, TP_A4, TP_A5, TP_A6)
  );

  constant O_PINS : Pinset := ALL_PINS and not I_PINS;
  constant EMPTY_EVENT : Integer := -1;

  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs) are contained within
  --  the procedure, and are analyzed with the VHDL code.  The test vectors
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After 95 ns (95% of period):  Expected outputs set to appropriate values
  --
  --   After 100 ns (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
  procedure SN54xx04_TEST
  (
    signal CONNECT   : inout Waves_Port_List
  ) is

    constant PERIOD          : Time       := 100 ns;
    constant UNCERTAINTY_PCT : Real       := 0.95;
    constant TDELAY          : Event_Time := ETIME(UNCERTAINTY_PCT * PERIOD);

    constant FSA : Frame_Set_Array := NEW_FRAME_SET_ARRAY(INPUT_FRAMES, I_PINS)
      +NEW_FRAME_SET_ARRAY(STROBED_OUTPUT_FRAMES(TDELAY), O_PINS);

    constant DT_BASIS : Delay_Time_Basis :=(FALSE,0 ns,0 ns,Timed_Delay,0,0,0);

    variable TD1 : Time_Data   := NEW_TIME_DATA ( FSA );
    variable T1  : Wave_Timing := ( (PERIOD, DT_BASIS), TD1 );

    --        Order of signals:
    --
    --        (Y1, Y2, Y3, Y4, Y5, Y6,
    --         A1, A2, A3, A4, A5, A6)
    --

    variable ALL_STATES : Wtime_Slice_List (1 to 11) := (
      (  "UUUUUU"
     & "UUUUUU", T1),

      (  "111111"
     & "000000", T1), -- reset to normal

      (  "UX01XX"
     & "U-HLWZ", T1), -- 9 states => A1-A6 inputs

      (  "XUX01X"
     & "XU-HLW", T1),

      (  "1XUX01"
     & "0XU-HL", T1),

      (  "01XUX0"
     & "10XU-H", T1),

      (  "X01XUX"
     & "Z10XU-", T1),

      (  "XX01XU"
     & "WZ10XU", T1),

      (  "1XX01X"
     & "LWZ10X", T1),

      (  "01XX01"
     & "HLWZ10", T1),

      (  "X01XX0"
     & "-HLWZ1", T1)
      );



    variable TIMING_PARAMS : Wtime_Slice_List (1 to 4) := (
      (  "111111"
     & "000000", T1), -- reset to normal 

      (  "000000"
     & "111111", T1), -- timing parameters

      (  "111111"
     & "000000", T1), -- tplh

      (  "000000"
     & "111111", T1) -- tphl
      );



    variable TRUTH_TABLE : Wtime_Slice_List (1 to 3) := (
      (  "111111"
     & "000000", T1), -- reset to normal

      (  "000000"
     & "111111", T1), -- truth table

      (  "111111"
     & "000000", T1)
      );



  begin
    assert FALSE
      report "ALL_STATES vectors"
        severity note;
      for I in ALL_STATES' range loop
        APPLY(CONNECT,ALL_STATES(I).CODES,ALL_STATES(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "TIMING_PARAMS vectors"
        severity note;
      for I in TIMING_PARAMS' range loop
        APPLY(CONNECT,TIMING_PARAMS(I).CODES,TIMING_PARAMS(I).WTIME,ALL_PINS);
      end loop;

    assert FALSE
      report "TRUTH_TABLE vectors"
        severity note;
      for I in TRUTH_TABLE' range loop
        APPLY(CONNECT,TRUTH_TABLE(I).CODES,TRUTH_TABLE(I).WTIME,ALL_PINS);
      end loop;

    end;


  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs, along with the period
  --  of each slice) are read in from an external file.  The test vectors 
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After <START_TOLERANCE>:  Expected outputs set to appropriate values
  --
  --   After <FS_TIME - END_TOLERANCE>: Get next set of inputs and outputs
  --
  --   After <FS_TIME> (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
  procedure SN54xx04_Test
  (
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time
  ) is

    variable TOLERANCE : Time := START_TOLERANCE + END_TOLERANCE;

    constant FSA : Frame_Set_Array :=
      NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY (OUTPUT_FRAMES, O_PINS);

    constant DT_BASIS : Delay_Time_Basis :=(FALSE,0 ns,0 ns,Timed_Delay,0,0,0);

    variable TD : Time_Data := NEW_TIME_DATA (FSA);

    constant FSA_DONT_CARE : Frame_Set_Array :=
      NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
      NEW_FRAME_SET_ARRAY (DONT_CARE_FRAMES, O_PINS);

    variable TD_DONT_CARE : Time_Data := NEW_TIME_DATA (FSA_DONT_CARE);

    file INPUT_FILE : Text is in INPUT_FILE_NAME;

    -- the file slice must be allocated
    variable SN54xx04_FILE_SLICE : File_Slice := NEW_FILE_SLICE;

    begin
      loop
        READ_FILE_SLICE(INPUT_FILE, SN54xx04_FILE_SLICE);
        exit when SN54xx04_FILE_SLICE.END_OF_FILE;
        if SN54xx04_FILE_SLICE.FS_TIME > TOLERANCE then
          APPLY(CONNECT, SN54xx04_FILE_SLICE.CODES.all,DELAY(START_TOLERANCE),
            TD_DONT_CARE);
          APPLY(CONNECT, SN54xx04_FILE_SLICE.CODES.all,
            DELAY(SN54xx04_FILE_SLICE.FS_TIME - TOLERANCE), TD);
          APPLY(CONNECT,SN54xx04_FILE_SLICE.CODES.all,DELAY(END_TOLERANCE),
            TD_DONT_CARE);
        else
          APPLY(CONNECT, SN54xx04_FILE_SLICE.CODES.all,
            DELAY(SN54xx04_FILE_SLICE.FS_TIME), TD);
        end if;
      end loop;
    end;
end WGENERATOR_SN54xx04;


