--  Filename:        shifter_TB.vhd
--
-- Version 2.1
-----------------------------------------------------------------------------
--
library IEEE;                     -- Logic system
  use IEEE.STD_LOGIC_1164.all;    -- Defines logic types, operators, functions


library WAVES_STANDARD;
  use WAVES_STANDARD.WAVES_INTERFACE.all;

library AM29C01_PACKAGE;
  use Am29C01_PACKAGE.TYPES.all;

library SHIFTER;
  use SHIFTER.WAVES_DEVICE.all;
  use SHIFTER.WAVES_OBJECTS.all;
  use SHIFTER.WAVES_UTILITIES.all;
  use SHIFTER.WGENERATOR_shifter.all;

entity shifter_TB is
  begin
  end shifter_TB;

-- ======================================================================


architecture TEST_BENCH of shifter_TB is

  component DUT 
    port
    (
      DATA      : in  Fourbit;
      Y         : out Fourbit;
      SHIFTSEL  : in  Std_Ulogic;
      DIRECTION : in  Std_Ulogic;
      SINLSB    : in  Std_Ulogic;
      sinmsb    : in  Std_Ulogic;
      soutlsb   : out Std_Ulogic;
      soutmsb   : out Std_Ulogic );
  end component;

  for all : DUT use entity SHIFTER.SHIFTER (BEHAVIORAL);

  subtype Test_Pin_Range is integer range
      Test_Pins'pos(TP_DATA3) + 1 to Test_Pins'pos(TP_soutmsb) + 1;

  -- Input and Output traffic of DUT
  signal TRAFFIC : Std_Ulogic_Vector(Test_Pin_Range);

  signal DATA : Fourbit;
  signal Y    : Fourbit;

  -- Test program options
  constant IN_FILE_NAME        : String := "shifter_in.dat";
  constant START_TOLERANCE     : Time   := 2 ns;
  constant END_TOLERANCE       : Time   := 2 ns;
  constant DEFAULT_PERIOD      : Time   := 10 ns;
  constant DEFAULT_UNCERTAINTY : Real   := 0.95;

  -- Comparator options
  signal COMPARE           : Boolean           := TRUE;
  constant HOW_TO_REPORT   : Comparison_Format := DO_ASSERTION;
  constant ASSERTION_LEVEL : Severity_Level    := WARNING;

  -- Signal monitor options
  constant HOW_TO_PRINT  : Output_Format := WAVES;
  constant OUT_FILE_NAME : String        := "shifter_out.dat";

  -- signal SMATCH : WAVES_MATCH_LIST(Test_Pin_Range) := 
  -- (others.M_VALUE => TRUE);
  signal SMATCH : WAVES_MATCH_LIST(Test_Pin_Range);

  -- Connection to WAVES dataset
  signal CONNECT : Waves_Port_List (Test_Pin_Range);

  subtype Compare_Pin_Range1 is Integer range
      Test_Pins'pos(TP_Y3) + 1 to Test_Pins'pos(TP_Y0) + 1;
        
  subtype Compare_Pin_Range2 is Integer range
      Test_Pins'pos(TP_SOUTLSB) + 1 to Test_Pins'pos(TP_SOUTMSB) + 1;

------------------------------------------------------------------------------

begin

  -- Call the waveform generator (i.e., the WAVES dataset)
  process 
  begin
    SHIFTER_TEST(CONNECT, IN_FILE_NAME, DEFAULT_PERIOD, DEFAULT_UNCERTAINTY);
    -- SHIFTER_TEST(CONNECT);
    wait;
  end process;

----------------------------------------------------------------------------
  -- Instantiate the Device under test

  DATA(3) <= TRAFFIC(Test_Pins'pos(TP_DATA3) + 1);
  DATA(2) <= TRAFFIC(Test_Pins'pos(TP_DATA2) + 1);
  DATA(1) <= TRAFFIC(Test_Pins'pos(TP_DATA1) + 1);
  DATA(0) <= TRAFFIC(Test_Pins'pos(TP_DATA0) + 1);

  TRAFFIC(Test_Pins'pos(TP_Y3) + 1) <= Y(3);
  TRAFFIC(Test_Pins'pos(TP_Y2) + 1) <= Y(2);
  TRAFFIC(Test_Pins'pos(TP_Y1) + 1) <= Y(1);
  TRAFFIC(Test_Pins'pos(TP_Y0) + 1) <= Y(0);

  UNDER_TEST : DUT
    port map 
    (
      DATA,
      Y, 
      TRAFFIC(Test_Pins'pos(TP_SHIFTSEL) + 1),
      TRAFFIC(Test_Pins'pos(TP_DIRECTION) + 1),
      TRAFFIC(Test_Pins'pos(TP_SINLSB) + 1),
      TRAFFIC(Test_Pins'pos(TP_SINMSB) + 1),
      TRAFFIC(Test_Pins'pos(TP_SOUTLSB) + 1),
      TRAFFIC(Test_Pins'pos(TP_SOUTMSB) + 1)
    );

----------------------------------------------------------------------------
  -- Collect all signals (inputs and output of DUT) into signal TRAFFIC

  COLLECT_INPUTS1:
    for I in Test_Pins'pos(TP_DATA3) + 1 to Test_Pins'pos(TP_DATA0) + 1 generate
        TRAFFIC(I) <= LOGIC_VALUE_TO_STD_ULOGIC
        (WAVES_PORT_TO_LOGIC_VALUE(CONNECT(I)));
  end generate COLLECT_INPUTS1;

  COLLECT_INPUTS2:
    for I in Test_Pins'pos(TP_SHIFTSEL) + 1 to Test_Pins'pos(TP_SINMSB) + 1 generate
        TRAFFIC(I) <= LOGIC_VALUE_TO_STD_ULOGIC
        (WAVES_PORT_TO_LOGIC_VALUE(CONNECT(I)));
  end generate COLLECT_INPUTS2;

----------------------------------------------------------------------------

  SIGNAL_TRAFFIC : SIGNAL_MONITOR(OUT_FILE_NAME, TRAFFIC, HOW_TO_PRINT);

----------------------------------------------------------------------------

  GEN_COMPARATOR1:
    for I in Compare_Pin_Range1 generate
      COMPARATOR : CHECK_RESPONSE(WHICH_ONE => I,
                                  RESPONSE => TRAFFIC(I),
                                  PREDICTION => CONNECT(I),
                                  COMPARE => COMPARE,
                                  HOW_TO_REPORT => HOW_TO_REPORT,
                                  ASSERTION_LEVEL => ASSERTION_LEVEL,
                                  MATCH => SMATCH(I).M_VALUE );

  end generate GEN_COMPARATOR1;

  GEN_COMPARATOR2:
    for I in Compare_Pin_Range2 generate
      COMPARATOR : CHECK_RESPONSE(WHICH_ONE => I,
                                  RESPONSE => TRAFFIC(I),
                                  PREDICTION => CONNECT(I),
                                  COMPARE => COMPARE,
                                  HOW_TO_REPORT => HOW_TO_REPORT,
                                  ASSERTION_LEVEL => ASSERTION_LEVEL,
                                  MATCH => SMATCH(I).M_VALUE );

  end generate GEN_COMPARATOR2;

end TEST_BENCH;

