--  Filename : wgenerator_qreg.vhd
--
-- Version 2.1
-----------------------------------------------------------------------------
--
use STD.TEXTIO.all;

library WAVES_STANDARD;
  use WAVES_STANDARD.WAVES_SYSTEM.all;
  use WAVES_STANDARD.WAVES_INTERFACE.all;

library QREG;
  use QREG.WAVES_DEVICE.all;
  use QREG.WAVES_FRAMES.all;
  use QREG.WAVES_OBJECTS.all;
  use QREG.WAVES_UTILITIES.all;

package WGENERATOR_qreg is
  procedure QREG_TEST(signal CONNECT   : inout Waves_Port_List);

  procedure QREG_TEST
  (
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time
  );

    procedure QREG_Test
    (
      signal CONNECT   : inout Waves_Port_List;
      constant INPUT_FILE_NAME  : String;
      constant PERIOD  : Time;
      constant UNCERTAINTY : Real
    );

end WGENERATOR_QREG;

-- ======================================================================

package body WGENERATOR_QREG is

  constant I_PINS : Pinset := NEW_PINSET(( TP_DATA_IN3, TP_DATA_IN2,
                                           TP_DATA_IN1, TP_DATA_IN0,
                                           TP_CP, TP_QEN));

  constant O_PINS : Pinset := ALL_PINS and not I_PINS;

  constant EMPTY_EVENT : Integer := -1;

---------------------------------------------------------------------------
--  The following procedure is used to provide test vectors for the model.
--  The test vectors (inputs and expected outputs) are contained within
--  the procedure, and are analyzed with the VHDL code.  The test vectors
--  are applied as follows:
--
--   Beginning of period:  Inputs set to appropriate values
--                         Expected outputs set to "don't care"
--
--   After 95 ns (95% of period):  Expected outputs set to appropriate values
--
--   After 100 ns (end of period): Get next set of inputs and outputs
---------------------------------------------------------------------------
  procedure QREG_TEST(
    signal CONNECT   : inout Waves_Port_List) is

    constant PERIOD          : Time       := 100 ns;
    constant UNCERTAINTY_PCT : Real       := 0.95;
    constant TDELAY          : Event_Time := ETIME (UNCERTAINTY_PCT * PERIOD);

    constant FSA : Frame_Set_Array := 
        NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, Timed_Delay, 0, 0, 0);

    variable TD1 : Time_Data := NEW_TIME_DATA ( FSA );
    variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1);

--        Order of signals:
--
--        (DATA_IN3, DATA_IN2, DATA_IN1, DATA_IN0,
--         DATA_OUT3, DATA_OUT2, DATA_OUT1, DATA_OUT0,
--         CP, QEN)
--
    variable WAVEFORM : WTime_Slice_List (1 to 2) := (
        (  "UUUU"
         & "UUUU"
         & "UU", T1),

        (  "XXXX"
         & "XXXX"
         & "XX", T1)
    );

    begin
      for I in WAVEFORM'range loop
        APPLY(CONNECT, WAVEFORM(I).CODES, WAVEFORM(I).WTIME, ALL_PINS);
      end loop;
    end;

---------------------------------------------------------------------------
--  The following procedure is used to provide test vectors for the model.
--  The test vectors (inputs and expected outputs, along with the period
--  of each slice) are read in from an external file.  The test vectors 
--  are applied as follows:
--
--   Beginning of period:  Inputs set to appropriate values
--                         Expected outputs set to "don't care"
--
--   After <START_TOLERANCE>:  Expected outputs set to appropriate values
--
--   After <FS_TIME - END_TOLERANCE>: Get next set of inputs and outputs
--
--   After <FS_TIME> (end of period): Get next set of inputs and outputs
---------------------------------------------------------------------------
  procedure QREG_TEST(
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time ) is


    variable TOLERANCE : Time := START_TOLERANCE + END_TOLERANCE;

    constant FSA : Frame_Set_Array :=
        NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY (OUTPUT_FRAMES, O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, Timed_Delay,
                                             0, 0, 0);

    variable TD : Time_Data := NEW_TIME_DATA (FSA);

    constant FSA_DONT_CARE : Frame_Set_Array :=
        NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY (DONT_CARE_FRAMES, O_PINS);

    variable TD_DONT_CARE : Time_Data := NEW_TIME_DATA (FSA_DONT_CARE);

    file INPUT_FILE : Text is in INPUT_FILE_NAME;

    -- the file slice must be allocated
    variable QREG_FILE_SLICE : File_Slice := NEW_FILE_SLICE;

  begin
    loop
      READ_FILE_SLICE(INPUT_FILE, QREG_FILE_SLICE);
      exit when QREG_FILE_SLICE.END_OF_FILE;
      if QREG_FILE_SLICE.FS_TIME > TOLERANCE then
        APPLY(CONNECT, QREG_FILE_SLICE.CODES.all,
              DELAY(START_TOLERANCE), TD_DONT_CARE);
        APPLY(CONNECT, QREG_FILE_SLICE.CODES.all,
              DELAY(QREG_FILE_SLICE.FS_TIME - TOLERANCE), TD);
        APPLY(CONNECT, QREG_FILE_SLICE.CODES.all,
              DELAY(END_TOLERANCE), TD_DONT_CARE);
      else
        APPLY(CONNECT, QREG_FILE_SLICE.CODES.all,
              DELAY(QREG_FILE_SLICE.FS_TIME), TD);
      end if;
    end loop;

  end;

---------------------------------------------------------------------------
--  The following procedure is used to provide test vectors for the model.
--  The functional test vectors (inputs and corresponding expected outputs, 
--  of each slice) are read in from an external file (see "A User's Guide
--  to WAVES", December 10, 1990, WAVES Analysis and Standardization Group,
--  for information on WAVES level 1 external file format).  It is assumed 
--  that the inputs values of each slice cause the output values of the same
--  slice to occur within the period given, which is a parameter to this
--  procedure.  The test vectors are applied as follows:
--
--   Beginning of period:  Inputs set to appropriate values
--                         Expected outputs set to "don't care"
--
--   After <UNCERTAINTY> * <PERIOD>:  Expected outputs set to appropriate 
--                                    values (and checked in the comparator)
--
--   After <PERIOD> (end of period): Get next set of inputs and outputs
---------------------------------------------------------------------------
  procedure QREG_Test(
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant PERIOD           : Time;
    constant UNCERTAINTY      : Real ) is


    constant TDELAY : Event_Time := ETIME (UNCERTAINTY * PERIOD);

    constant FSA : Frame_Set_Array := 
        NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, Timed_Delay,
                                             0, 0, 0);

    variable TD1 : Time_Data   := NEW_TIME_DATA ( FSA );
    variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1);

    file INPUT_FILE : Text is in INPUT_FILE_NAME;

    -- the file slice must be allocated
    variable QREG_FILE_SLICE : File_Slice := NEW_FILE_SLICE;
    variable WAVEFORM        : WTime_Slice;

  begin
    loop
      READ_FILE_SLICE(INPUT_FILE, QREG_FILE_SLICE);
      exit when QREG_FILE_SLICE.END_OF_FILE;
      WAVEFORM := (QREG_FILE_SLICE.CODES.all, T1);
      APPLY(CONNECT, WAVEFORM.CODES, WAVEFORM.WTIME, ALL_PINS);
    end loop;

  end;

end WGENERATOR_QREG;
