-- Filename: functions.vhd
--
-- Version 2.1
--
--                                   NOTICE
--
--      LICENSEE acknowledges that it has read and executed the
--      Institute for Technology Development, VHDL Standard Component
--      Library Software Standard License Agreement.  Licensee may not
--      sublicense, assign or transfer the Software, the Documentation or its
--      rights under the Agreement.
--      
--      The Agreement authorizes Licensee to use at any one time only
--      one copy of the Software (either the original Software or the
--      Archival Copy but never both.)  Licensee may make one (1) copy
--      of the Software (the "Archival Copy") and one (1) copy of the
--      Documentation for archival purposes.  Licensee may not otherwise copy,
--      modify, transfer or disclose in any manner the Software or
--      Documentation.  Licensee shall store the Archival Copy in a secure
--      location to which access shall be restricted.  Licensee shall maintain
--      a written record of the data on which the Archival Copy is created and
--      its location.
--      
--      Licensee expressly recognizes that the Software is the proprietary
--      property of ITD.  Licensee warrants that it will not take any action
--      which would result in the impairment or loss of proprietary rights of
--      ITD in the Software.  Licensee warrants that the Software will
--      not be incorporated into any Licensee product.  Licensee warrants 
--      that it will not decompile, disassemble or reverse engineer the 
--      Software or attempt to do so for other than internal applications.
--      Licensee warrants that it will never divulge to any person without the
--      prior written consent of ITD the Software source code or Software
--      object code.
--      
--      The Software is provided "AS IS" WITHOUT WARRANTY of any kind, either
--      express or implied, including but not limited to implied warranties of
--      merchantability and fitness for a particular purpose.
--      
--      RIGHT TO ACCESS OF THIS SOFTWARE IS DENIED IF THIS NOTICE IS REMOVED.
------------------------------------------------------------------------------

library IEEE;                     -- Logic system
  use IEEE.STD_LOGIC_1164.all;    -- Defines logic types, operators, functions

library AM29C01_PACKAGE;          -- local package
  use AM29C01_PACKAGE.TYPES.all;  -- local type declarations

package FUNCTIONS is

  function INT_TO_BITVECTOR  (INT, LENGTH:Natural) return Std_Ulogic_Vector;
  function MODIFY_RAM    (B : Std_Ulogic_Vector; LOCATION : Integer)
                          return Boolean;
  function FILTER_VECTOR (INPUT : Std_Ulogic_Vector) return 
                          Std_Ulogic_Vector; 
  function UNINIT_ON_BUS (BUS_VECTOR : Std_Ulogic_Vector) return Boolean;

end FUNCTIONS;

-- ======================================================================

package body FUNCTIONS is

  type Std_Ulogic_Array is array (Std_Ulogic) of Std_Ulogic;

---------------------------------------------------------------------------
-- This function takes as input an integer and a length and returns
-- a Std_Ulogic_Vector of the input length with the value of the
-- integer passed in.
-- Example: int_to_bitvector(6,4) would return a Std_Ulogic_Vector (of length 4)
-- containing the value 0110.
---------------------------------------------------------------------------
function INT_TO_BITVECTOR  (INT, LENGTH : Natural) return Std_Ulogic_Vector is
  variable DIGIT  : Natural := 2 ** (LENGTH-1);
  variable LOCAL  : Natural;
  variable RESULT : Std_Ulogic_Vector (0 to LENGTH-1);

begin
  LOCAL := INT mod (2 ** LENGTH);
  for I in RESULT'range loop
    if LOCAL >= DIGIT then
      RESULT(I) := '1';
      LOCAL := LOCAL - DIGIT;
    else
      RESULT(I) := '0';
    end if;
    DIGIT := DIGIT/2;
  end loop;

  return RESULT;

end INT_TO_BITVECTOR;

------------------------------------------------------------------------------
-- MODIFY_RAM 
-- This function takes as input a std_ulogic vector and an integer and returns TRUE
-- if the the std_ulogic vector could be = to the integer address. For example, if
-- B = "010X" the function returns TRUE if location equals 4 or 5 .
--
---------------------------------------------------------------------------
function MODIFY_RAM (B : Std_Ulogic_Vector; LOCATION : Integer) 
                     return Boolean is 

  variable TEMP  : Fourbit;
  variable COUNT : Integer := 0;

begin
  TEMP := Int_To_Bitvector(LOCATION, 4);
    for i in B'range loop
      case B(i) is         
        when '0' => 
          if TEMP(i) = '0' then 
            COUNT := COUNT +1;
          end if;   
        when '1' => 
          if TEMP(i) = '1' then 
            COUNT := COUNT + 1;
          end if;   
        when others => COUNT := COUNT + 1;
      end case;
    end loop;
    
  if COUNT = 4 then 
    return TRUE;
  end if;
  return FALSE;    
end MODIFY_RAM;                

------------------------------------------------------------------------------
-- FILTER_VECTOR
-- converts a Std_Ulogic_Vector to X01 type
------------------------------------------------------------------------------
function FILTER_VECTOR (INPUT : Std_Ulogic_Vector) 
                        return Std_Ulogic_Vector is 

  variable TEMP : Std_Ulogic_Vector(INPUT'range);
  constant FILTER_TABLE : Std_Ulogic_Array := 
     ('X', 'X', '0', '1', 'X', 'X', '1', '0', 'X');
     
begin
  for i in INPUT'range loop
    TEMP(i) := FILTER_TABLE(INPUT(i));
  end loop;
  return TEMP;
end FILTER_VECTOR;    
             
------------------------------------------------------------------------------
-- This function takes as input a std_ulogic_vector and returns
-- TRUE if any of the bits are 'U', else it returns FALSE.
---------------------------------------------------------------------------

FUNCTION UNINIT_ON_BUS(BUS_VECTOR : Std_Ulogic_Vector) 
                       return Boolean is
begin
  for i in BUS_VECTOR'range loop
    case BUS_VECTOR(i) is
      when 'U' => return TRUE;
      when others => NULL;
    end case;   
  end loop;
  return FALSE;
end UNINIT_ON_BUS;


end FUNCTIONS;

