--  Filename : wgenerator_decoder.vhd
--
-- Version 2.1
--
-----------------------------------------------------------------------------
--
use STD.TEXTIO.all;

library WAVES_STANDARD;
  use WAVES_STANDARD.WAVES_SYSTEM.all;
  use WAVES_STANDARD.WAVES_INTERFACE.all;

library DECODER;
  use DECODER.WAVES_DEVICE.all;
  use DECODER.WAVES_FRAMES.all;
  use DECODER.WAVES_OBJECTS.all;
  use DECODER.WAVES_UTILITIES.all;

package WGENERATOR_decoder is
  procedure decoder_TEST
  (
    signal CONNECT   : inout Waves_Port_List
  );

  procedure decoder_TEST
  (
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time 
  );

  procedure decoder_Test
  (
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant PERIOD           : Time;
    constant UNCERTAINTY      : Real
  );

end WGENERATOR_decoder;

-- ======================================================================

package body WGENERATOR_decoder is

  constant I_PINS : Pinset := NEW_PINSET(( TP_SRC2, TP_SRC1, TP_SRC0,
                                           TP_DEST2, TP_DEST1, TP_DEST0,
                                           TP_FUNT2, TP_FUNT1, TP_FUNT0));

  constant O_PINS : Pinset := ALL_PINS and not I_PINS;

  constant EMPTY_EVENT : Integer := -1;

---------------------------------------------------------------------------
--  The following procedure is used to provide test vectors for the model.
--  The test vectors (inputs and expected outputs) are contained within
--  the procedure, and are analyzed with the VHDL code.  The test vectors
--  are applied as follows:
--
--   Beginning of period:  Inputs set to appropriate values
--                         Expected outputs set to "don't care"
--
--   After 95 ns (95% of period):  Expected outputs set to appropriate values
--
--   After 100 ns (end of period): Get next set of inputs and outputs
---------------------------------------------------------------------------
  procedure decoder_TEST
  (
    signal CONNECT   : inout Waves_Port_List) is

    constant PERIOD          : Time       := 100 ns;
    constant UNCERTAINTY_PCT : Real       := 0.95;
    constant TDELAY          : Event_Time := ETIME (UNCERTAINTY_PCT * PERIOD);

    constant FSA : Frame_Set_Array := 
        NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (false, 0 ns, 0 ns, Timed_Delay, 0, 0, 0);

    variable TD1 : Time_Data   := NEW_TIME_DATA ( FSA );
    variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1);

--        Order of signals:
--
--        (SRC2, SRC1, SRC0, DEST2, DEST1, DEST0, FUNT2, FUNT1, FUNT0,
--         RMUX_SEL1, RMUX_SEL0, SMUX_SEL1, SMUX_SEL0,
--         YMUX_SEL, QMUX_SEL, Q_LOAD, QSHIFT_DIR, QSHIFT_SEL, RAM_EN,
--         RAMSHIFT_DIR, RAMSHIFT_SEL, ALU_FUNT2, ALU_FUNT1, ALU_FUNT0)
--

    variable WAVEFORM : WTime_Slice_List (1 to 2) := (
        (  "UUUUUUUUU"
         & "UUUU"
         & "UUUUUUUU"
         & "UUU", T1),

        (  "XXXXXXXXX"
         & "XXXX"
         & "XXXXXXXX"
         & "XXX", T1)
    );

    begin
      for I in WAVEFORM'range loop
        APPLY(CONNECT, WAVEFORM(I).CODES, WAVEFORM(I).WTIME, ALL_PINS);
      end loop;
    end;

---------------------------------------------------------------------------
--  The following procedure is used to provide test vectors for the model.
--  The test vectors (inputs and expected outputs, along with the period
--  of each slice) are read in from an external file.  The test vectors 
--  are applied as follows:
--
--   Beginning of period:  Inputs set to appropriate values
--                         Expected outputs set to "don't care"
--
--   After <START_TOLERANCE>:  Expected outputs set to appropriate values
--
--   After <FS_TIME - END_TOLERANCE>: Get next set of inputs and outputs
--
--   After <FS_TIME> (end of period): Get next set of inputs and outputs
---------------------------------------------------------------------------
  procedure decoder_Test
  (
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time ) is

    variable TOLERANCE : Time := START_TOLERANCE + END_TOLERANCE;

    constant FSA : Frame_Set_Array :=
        NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY (OUTPUT_FRAMES, O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, Timed_Delay, 0, 0, 0);

    variable TD : Time_Data := NEW_TIME_DATA (FSA);

    constant FSA_DONT_CARE : Frame_Set_Array :=
        NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY (DONT_CARE_FRAMES, O_PINS);

    variable TD_DONT_CARE : Time_Data := NEW_TIME_DATA (FSA_DONT_CARE);

    file INPUT_FILE : text is in INPUT_FILE_NAME;

    -- the file slice must be allocated
    variable DECODER_FILE_SLICE : File_Slice := NEW_FILE_SLICE;

  begin
    loop
      READ_FILE_SLICE(INPUT_FILE, DECODER_FILE_SLICE);
      exit when DECODER_FILE_SLICE.END_OF_FILE;
      if DECODER_FILE_SLICE.FS_TIME > TOLERANCE then
        APPLY(CONNECT, DECODER_FILE_SLICE.CODES.all,
              DELAY(START_TOLERANCE), TD_DONT_CARE);
        APPLY(CONNECT, DECODER_FILE_SLICE.CODES.all,
              DELAY(DECODER_FILE_SLICE.FS_TIME - TOLERANCE), TD);
        APPLY(CONNECT, DECODER_FILE_SLICE.CODES.all,
              DELAY(END_TOLERANCE), TD_DONT_CARE);
      else
        APPLY(CONNECT, DECODER_FILE_SLICE.CODES.all,
              DELAY(DECODER_FILE_SLICE.FS_TIME), TD);
      end if;
    end loop;
  end;

---------------------------------------------------------------------------
--  The following procedure is used to provide test vectors for the model.
--  The functional test vectors (inputs and corresponding expected outputs, 
--  of each slice) are read in from an external file (see "A User's Guide
--  to WAVES", December 10, 1990, WAVES Analysis and Standardization Group,
--  for information on WAVES level 1 external file format).  It is assumed 
--  that the inputs values of each slice cause the output values of the same
--  slice to occur within the period given, which is a parameter to this
--  procedure.  The test vectors are applied as follows:
--
--   Beginning of period:  Inputs set to appropriate values
--                         Expected outputs set to "don't care"
--
--   After <UNCERTAINTY> * <PERIOD>:  Expected outputs set to appropriate 
--                                    values (and checked in the comparator)
--
--   After <PERIOD> (end of period): Get next set of inputs and outputs
---------------------------------------------------------------------------
  procedure decoder_Test
  (
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant PERIOD           : Time;
    constant UNCERTAINTY      : Real ) is

    constant TDELAY : Event_Time := ETIME (UNCERTAINTY * PERIOD);

    constant FSA : Frame_Set_Array := 
        NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, Timed_Delay, 0, 0, 0);

    variable TD1 : Time_Data   := NEW_TIME_DATA ( FSA );
    variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1);

    file INPUT_FILE : text is in INPUT_FILE_NAME;

    -- the file slice must be allocated
    variable DECODER_FILE_SLICE : File_Slice := NEW_FILE_SLICE;
    variable WAVEFORM           : WTime_Slice;

  begin
    loop
      READ_FILE_SLICE(INPUT_FILE, DECODER_FILE_SLICE);
      exit when DECODER_FILE_SLICE.END_OF_FILE;
      WAVEFORM := (DECODER_FILE_SLICE.CODES.all, T1);
      APPLY(CONNECT, WAVEFORM.CODES, WAVEFORM.WTIME, ALL_PINS);
    end loop;
  end;

end WGENERATOR_decoder;
