-- Filename:              decoder.vhd
--
-- Version 2.1
--
--                                   NOTICE
--
--      LICENSEE acknowledges that it has read and executed the
--      Institute for Technology Development, VHDL Standard Component
--      Library Software Standard License Agreement.  Licensee may not
--      sublicense, assign or transfer the Software, the Documentation or its
--      rights under the Agreement.
--      
--      The Agreement authorizes Licensee to use at any one time only
--      one copy of the Software (either the original Software or the
--      Archival Copy but never both.)  Licensee may make one (1) copy
--      of the Software (the "Archival Copy") and one (1) copy of the
--      Documentation for archival purposes.  Licensee may not otherwise copy,
--      modify, transfer or disclose in any manner the Software or
--      Documentation.  Licensee shall store the Archival Copy in a secure
--      location to which access shall be restricted.  Licensee shall maintain
--      a written record of the data on which the Archival Copy is created and
--      its location.
--      
--      Licensee expressly recognizes that the Software is the proprietary
--      property of ITD.  Licensee warrants that it will not take any action
--      which would result in the impairment or loss of proprietary rights of
--      ITD in the Software.  Licensee warrants that the Software will
--      not be incorporated into any Licensee product.  Licensee warrants 
--      that it will not decompile, disassemble or reverse engineer the 
--      Software or attempt to do so for other than internal applications.
--      Licensee warrants that it will never divulge to any person without the
--      prior written consent of ITD the Software source code or Software
--      object code.
--      
--      The Software is provided "AS IS" WITHOUT WARRANTY of any kind, either
--      express or implied, including but not limited to implied warranties of
--      merchantability and fitness for a particular purpose.
--      
--      RIGHT TO ACCESS OF THIS SOFTWARE IS DENIED IF THIS NOTICE IS REMOVED.
--
-----------------------------------------------------------------------------
-- DESCRIPTION:             
--
-- This file contains the entity declaration and behavioral model for an 
-- instruction decoder intended for incorporation into the Am2901. The
-- decoder section consist of a source and destination decoder. The 
-- function decoder is in the file alu.vhd
--

library IEEE;                     -- Logic system
  use IEEE.STD_LOGIC_1164.all;

library AM29C01_PACKAGE;
  use AM29C01_PACKAGE.TYPES.all;
  use AM29C01_PACKAGE.FUNCTIONS.all;

entity DECODER is
  port 
  (
    SRC          : in Threebit;      -- decoded for mux select
    DEST         : in Threebit;      -- decoded for destination code
    FUNT         : in Threebit;      -- decoded for alu function code
    RMUX_SEL     : out Twobit;       -- select inputs for the rmux
    SMUX_SEL     : out Twobit;       -- select inputs for the smux
    YMUX_SEL     : out Std_Ulogic;   -- select inputs for the ymux
    QMUX_SEL     : out Std_Ulogic;   -- select inputs for the qmux
    Q_LOAD       : out Std_Ulogic;   -- q register enable
    QSHIFT_DIR   : out Std_Ulogic;   -- direction input for q shifter
    QSHIFT_SEL   : out Std_Ulogic;   -- shift/dont shift input for q shifter
    RAM_EN       : out Std_Ulogic;   -- active low RAM enable
    RAMSHIFT_DIR : out Std_Ulogic;   -- direction input for ram shifter
    RAMSHIFT_SEL : out Std_Ulogic;   -- shift/dont shift input for ram shifter
    ALU_FUNT     : out Threebit
  );   
end DECODER;

-- ======================================================================

architecture BEHAVIORAL of DECODER is
begin 
   
  process(SRC)
  begin
    case SRC is
      when "000" => RMUX_SEL <= "00";     -- A select
                    SMUX_SEL <= "00";     -- Q select
      when "001" => RMUX_SEL <= "00";     -- A select 
                    SMUX_SEL <= "01";     -- B select
      when "010" => RMUX_SEL <= "10";     -- 0 select
                    SMUX_SEL <= "00";     -- Q select
      when "011" => RMUX_SEL <= "10";     -- 0 select
                    SMUX_SEL <= "01";     -- B select
      when "100" => RMUX_SEL <= "10";     -- 0 select
                    SMUX_SEL <= "10";     -- A select
      when "101" => RMUX_SEL <= "01";     -- D select
                    SMUX_SEL <= "10";     -- A select
      when "110" => RMUX_SEL <= "01";     -- D select
                    SMUX_SEL <= "00";     -- Q select
      when "111" => RMUX_SEL <= "01";     -- D select
                    SMUX_SEL <= "11";     -- 0 select
        
      -- set mux select inputs to UNKNOWNS  
      when others =>
        if (UNINIT_ON_BUS(SRC)) then   -- UNINITIALIZED select
          RMUX_SEL <= "UU";    
          SMUX_SEL <= "UU"; 
        else    -- UNKNOWN select
          RMUX_SEL <= "XX";   
          SMUX_SEL <= "XX";
        end if;
    end case; 
  end process;
  
  process(DEST)
  begin 
    case DEST is
      when "000" => RAMSHIFT_SEL <= '0';   -- dont shift 
                    RAM_EN       <= '1';   -- ram hold
                    QSHIFT_SEL   <= '0';   -- dont shift
                    Q_LOAD       <= '1';   -- load
                    QMUX_SEL     <= '0';   -- F -> Q
                    YMUX_SEL     <= '0';   -- F -> Y
                           
      when "001" => RAMSHIFT_SEL <= '0';   -- dont shift
                    RAM_EN       <= '1';   -- ram hold
                    QSHIFT_SEL   <= '0';   -- dont shift
                    Q_LOAD       <= '0';   -- hold Q
                    YMUX_SEL     <= '0';   -- F -> Y
                        
      when "010" => RAMSHIFT_SEL <= '0';   -- dont shift
                    RAM_EN       <= '0';   -- F -> B
                    QSHIFT_SEL   <= '0';   -- dont shift
                    YMUX_SEL     <= '1';   -- A -> Y
                    Q_LOAD       <= '0';   -- hold Q
                               
      when "011" => RAMSHIFT_SEL <= '0';   -- dont shift
                    RAM_EN       <= '0';   -- F -> B
                    QSHIFT_SEL   <= '0';   -- dont shift
                    YMUX_SEL     <= '0';   -- F -> Y
                    Q_LOAD       <= '0';   -- hold Q
                        
      when "100" => RAMSHIFT_SEL <= '1';   -- shift
                    RAMSHIFT_DIR <= '0';   -- shift down
                    RAM_EN       <= '0';   -- F/2 -> B
                    QMUX_SEL     <= '1';   -- Q -> shifter
                    QSHIFT_SEL   <= '1';   -- shift
                    QSHIFT_DIR   <= '0';   -- shift down
                    Q_LOAD       <= '1';   -- Q/2 -> Q
                    YMUX_SEL     <= '0';   -- F -> Y
                        
      when "101" => RAMSHIFT_SEL <= '1';   -- shift
                    RAMSHIFT_DIR <= '0';   -- shift down
                    RAM_EN       <= '0';   -- F/2 -> B 
                    QSHIFT_SEL   <= '0';   -- dont shift
                    Q_LOAD       <= '0';   -- hold
                    YMUX_SEL     <= '0';   -- F -> Y
                        
      when "110" => RAMSHIFT_SEL <= '1';   -- shift 
                    RAMSHIFT_DIR <= '1';   -- shift up 
                    RAM_EN       <= '0';   -- 2F -> B
                    QMUX_SEL     <= '1';   -- Q -> shifter
                    QSHIFT_SEL   <= '1';   -- shift
                    QSHIFT_DIR   <= '1';   -- shift up
                    Q_LOAD       <= '1';   -- 2Q -> Q
                    YMUX_SEL     <= '0';   -- F -> Y
                        
      when "111" => RAMSHIFT_SEL <= '1';   -- shift
                    RAMSHIFT_DIR <= '1';   -- shift up
                    RAM_EN       <= '0';   -- 2F -> B
                    Q_LOAD       <= '0';   -- hold
                    QSHIFT_SEL   <= '0';   -- dont shift
                    YMUX_SEL     <= '0';   -- F -> Y
                    
      when others =>
        if (UNINIT_ON_BUS(DEST)) then
          RAMSHIFT_SEL <= 'U';
          RAMSHIFT_DIR <= 'U';
          RAM_EN       <= 'U';
          QMUX_SEL     <= 'U';
          QSHIFT_SEL   <= 'U';
          QSHIFT_DIR   <= 'U';
          Q_LOAD       <= 'U';
          YMUX_SEL     <= 'U';
        else   
          RAMSHIFT_SEL <= 'X';
          RAMSHIFT_DIR <= 'X';
          RAM_EN       <= 'X';
          QMUX_SEL     <= 'X';
          QSHIFT_SEL   <= 'X';
          QSHIFT_DIR   <= 'X';
          Q_LOAD       <= 'X';
          YMUX_SEL     <= 'X';
        end if;    
    end case; 
  end process;
  
  process(FUNT)
  begin
    ALU_FUNT <= FUNT;
  end process;   

end BEHAVIORAL;
