--  Filename: wgenerator_alu.vhd
--
-- Version 2.1
--
-----------------------------------------------------------------------------
--
use STD.TEXTIO.all;

library WAVES_STANDARD;
  use WAVES_STANDARD.WAVES_SYSTEM.all;
  use WAVES_STANDARD.WAVES_INTERFACE.all;

library ALU;
  use ALU.WAVES_DEVICE.all;
  use ALU.WAVES_FRAMES.all;
  use ALU.WAVES_OBJECTS.all;
  use ALU.WAVES_UTILITIES.all;

package WGENERATOR_alu is
  procedure alu_TEST
  (
    signal CONNECT   : inout Waves_Port_List);

    procedure alu_TEST
    (
      signal CONNECT           : inout Waves_Port_List;
      constant INPUT_FILE_NAME : String;
      constant START_TOLERANCE : Time;
      constant END_TOLERANCE   : Time 
    );

    procedure alu_Test
    (
      signal CONNECT           : inout Waves_Port_List;
      constant INPUT_FILE_NAME : String;
      constant PERIOD          : Time;
      constant UNCERTAINTY     : Real 
    );

end WGENERATOR_alu;

-- ======================================================================

package body WGENERATOR_alu is

  constant O_PINS : Pinset := NEW_PINSET(( TP_FBUS3, TP_FBUS2, TP_FBUS1,
                                           TP_FBUS0, TP_G, TP_P, TP_CO, 
                                           TP_F3, TP_OVR, TP_F));

  constant I_PINS : Pinset := ALL_PINS and not O_PINS;

  constant EMPTY_EVENT : Integer := -1;

---------------------------------------------------------------------------
--  The following procedure is used to provide test vectors for the model.
--  The test vectors (inputs and expected outputs) are contained within
--  the procedure, and are analyzed with the VHDL code.  The test vectors
--  are applied as follows:
--
--   Beginning of period:  Inputs set to appropriate values
--                         Expected outputs set to "don't care"
--
--   After 95 ns (95% of period):  Expected outputs set to appropriate values
--
--   After 100 ns (end of period): Get next set of inputs and outputs
---------------------------------------------------------------------------
  procedure alu_TEST(
    signal CONNECT : inout Waves_Port_List) is

    constant PERIOD          : Time := 100 ns;
    constant UNCERTAINTY_PCT : Real := 0.95;
    constant TDELAY          : Event_Time := ETIME (UNCERTAINTY_PCT * PERIOD);

    constant FSA : Frame_Set_Array := 
        NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, Timed_Delay,
                                             0, 0, 0);

    variable TD1 : Time_Data   := NEW_TIME_DATA ( FSA );
    variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1);

--        Order of signals:
--
--        (RBUS3, RBUS2, RBUS1, RBUS0,
--         SBUS3, SBUS2, SBUS1, SBUS0, 
--         FBUS3, FBUS2, FBUS1, FBUS0, 
--         FUNT2, FUNT1, FUNT0, CIN,
--         G, P, CO, F3, OVR,F) 
--

-- NOTE:  The test vectors contained in this VHDL source code file are
--        not really useful, but are intended to provide "hooks" for 
--        including functional vectors if the user desires.

    variable WAVEFORM : Wtime_Slice_List (1 to 2) := (
        (  "UUUU"
         & "UUUU"
         & "UUUU"
         & "UUUU"
         & "UUUUUU", T1),

        (  "XXXX"
         & "XXXX"
         & "XXXX"
         & "XXXX"
         & "XXXXXX", T1)
    );

    begin
      for I in WAVEFORM'range loop
          APPLY(CONNECT, WAVEFORM(I).CODES, WAVEFORM(I).WTIME, ALL_PINS);
      end loop;
    end;

---------------------------------------------------------------------------
--  The following procedure is used to provide test vectors for the model.
--  The test vectors (inputs and expected outputs, along with the period
--  of each slice) are read in from an external file (see "A User's Guide
--  to WAVES", December 10, 1990, WAVES Analysis and Standardization Group,
--  for information on WAVES level 1 external file format).  The test vectors 
--  are applied as follows:
--
--   Beginning of period:  Inputs set to appropriate values
--                         Expected outputs set to "don't care"
--
--   After <START_TOLERANCE>:  Expected outputs set to appropriate values
--
--   After <FS_TIME - END_TOLERANCE>: Get next set of inputs and outputs
--
--   After <FS_TIME> (end of period): Get next set of inputs and outputs
---------------------------------------------------------------------------
  procedure alu_Test(
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant START_TOLERANCE  : Time;
    constant END_TOLERANCE    : Time ) is

    variable TOLERANCE        : Time := START_TOLERANCE + END_TOLERANCE;

    constant FSA : Frame_Set_Array :=
        NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY (OUTPUT_FRAMES, O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, Timed_Delay,
                                             0, 0, 0);

    variable TD : Time_Data := NEW_TIME_DATA (FSA);

    constant FSA_DONT_CARE : Frame_Set_Array :=
        NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY (DONT_CARE_FRAMES, O_PINS);

    variable TD_DONT_CARE : Time_Data := NEW_TIME_DATA (FSA_DONT_CARE);

    file INPUT_FILE : Text is in INPUT_FILE_NAME;

    -- the file slice must be allocated
    variable ALU_FILE_SLICE : File_Slice := NEW_FILE_SLICE;

  begin
    loop
      READ_FILE_SLICE(INPUT_FILE, ALU_FILE_SLICE);
      exit when ALU_FILE_SLICE.END_OF_FILE;
      if ALU_FILE_SLICE.FS_TIME > TOLERANCE then
        APPLY(CONNECT, ALU_FILE_SLICE.CODES.all, DELAY(START_TOLERANCE),
              TD_DONT_CARE);
        APPLY(CONNECT, ALU_FILE_SLICE.CODES.all,
              DELAY(ALU_FILE_SLICE.FS_TIME - TOLERANCE), TD);
        APPLY(CONNECT, ALU_FILE_SLICE.CODES.all, DELAY(END_TOLERANCE),
              TD_DONT_CARE);
      else
        APPLY(CONNECT, ALU_FILE_SLICE.CODES.all, DELAY(ALU_FILE_SLICE.FS_TIME),
              TD);
      end if;
    end loop;
  end;

---------------------------------------------------------------------------
--  The following procedure is used to provide test vectors for the model.
--  The functional test vectors (inputs and corresponding expected outputs, 
--  of each slice) are read in from an external file (see "A User's Guide
--  to WAVES", December 10, 1990, WAVES Analysis and Standardization Group,
--  for information on WAVES level 1 external file format).  It is assumed 
--  that the inputs values of each slice cause the output values of the same
--  slice to occur within the period given, which is a parameter to this
--  procedure.  The test vectors are applied as follows:
--
--   Beginning of period:  Inputs set to appropriate values
--                         Expected outputs set to "don't care"
--
--   After <UNCERTAINTY> * <PERIOD>:  Expected outputs set to appropriate 
--                                    values (and checked in the comparator)
--
--   After <PERIOD> (end of period): Get next set of inputs and outputs
---------------------------------------------------------------------------
  procedure alu_Test(
    signal CONNECT            : inout Waves_Port_List;
    constant INPUT_FILE_NAME  : String;
    constant PERIOD           : Time;
    constant UNCERTAINTY      : real ) is

    constant TDELAY : Event_Time := ETIME (UNCERTAINTY * PERIOD);

    constant FSA : Frame_Set_Array := 
        NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
        NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS);

    constant DT_BASIS : Delay_Time_Basis := (FALSE, 0 ns, 0 ns, Timed_Delay,
                                             0, 0, 0);

    variable TD1 : Time_Data   := NEW_TIME_DATA ( FSA );
    variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1);

    file INPUT_FILE : Text is in INPUT_FILE_NAME;

    -- the file slice must be allocated
    variable ALU_FILE_SLICE : File_Slice := NEW_FILE_SLICE;
    variable WAVEFORM       : Wtime_Slice;

  begin
    loop
      READ_FILE_SLICE(INPUT_FILE, ALU_FILE_SLICE);
      exit when ALU_FILE_SLICE.END_OF_FILE;
      WAVEFORM := (ALU_FILE_SLICE.CODES.all, T1);
      APPLY(CONNECT, WAVEFORM.CODES, WAVEFORM.WTIME, ALL_PINS);
    end loop;
  end;

end WGENERATOR_alu;
