-- Filename: alu.vhd
--
-- Version 2.1
--
--                                   NOTICE
--
--      LICENSEE acknowledges that it has read and executed the
--      Institute for Technology Development, VHDL Standard Component
--      Library Software Standard License Agreement.  Licensee may not
--      sublicense, assign or transfer the Software, the Documentation or its
--      rights under the Agreement.
--      
--      The Agreement authorizes Licensee to use at any one time only
--      one copy of the Software (either the original Software or the
--      Archival Copy but never both.)  Licensee may make one (1) copy
--      of the Software (the "Archival Copy") and one (1) copy of the
--      Documentation for archival purposes.  Licensee may not otherwise copy,
--      modify, transfer or disclose in any manner the Software or
--      Documentation.  Licensee shall store the Archival Copy in a secure
--      location to which access shall be restricted.  Licensee shall maintain
--      a written record of the data on which the Archival Copy is created and
--      its location.
--      
--      Licensee expressly recognizes that the Software is the proprietary
--      property of ITD.  Licensee warrants that it will not take any action
--      which would result in the impairment or loss of proprietary rights of
--      ITD in the Software.  Licensee warrants that the Software will
--      not be incorporated into any Licensee product.  Licensee warrants 
--      that it will not decompile, disassemble or reverse engineer the 
--      Software or attempt to do so for other than internal applications.
--      Licensee warrants that it will never divulge to any person without the
--      prior written consent of ITD the Software source code or Software
--      object code.
--      
--      The Software is provided "AS IS" WITHOUT WARRANTY of any kind, either
--      express or implied, including but not limited to implied warranties of
--      merchantability and fitness for a particular purpose.
--      
--      RIGHT TO ACCESS OF THIS SOFTWARE IS DENIED IF THIS NOTICE IS REMOVED.
------------------------------------------------------------------------------
-- DESCRIPTION:             
--
-- This file contains the entity declaration and behavioral model for a 4-bit alu.
-- It was written to be used as the alu for the Am2901 structural model.
--


library IEEE;                        -- Logic system
  use IEEE.STD_LOGIC_1164.all;    -- Defines logic types, operators, functions

library AM29C01_PACKAGE;
  use Am29C01_PACKAGE.TYPES.all;
  use Am29C01_PACKAGE.FUNCTIONS.all;

entity ALU is
       
  port
  (
    RBUS : in Fourbit;          -- R source operand
    SBUS : in Fourbit;          -- S source operand
    FBUS : out Fourbit;         -- data out of alu
    FUNT : in Threebit;         -- function code
    CIN  : in Std_Ulogic;       -- carry in
    G    : out Std_Ulogic;      -- generate output
    P    : out Std_Ulogic;      -- propogate output
    Co   : out Std_Ulogic;      -- carry out
    F3   : out Std_Ulogic;      -- sign bit
    OVR  : out Std_Ulogic;      -- overflow 
    F    : out Std_Ulogic       -- Fbus = 0
  );

end ALU;

-- ======================================================================

architecture BEHAVIORAL of ALU is

begin

  process(RBUS, SBUS, FUNT, CIN)

    variable PTERM  : Fourbit;  -- array storage for Pi 
    variable GTERM  : Fourbit;  -- array storage for Gi
    variable CTERM  : Fourbit;  -- array storage for Ci

    variable RTEMP  : Fourbit;  -- temporary for R
    variable STEMP  : Fourbit;  -- temporary for S
    variable GAND   : Fourbit;     
    variable RESULT : Fourbit;  -- temporary for FBUS
   
    variable P3, P32, P321, P3210 : Std_Ulogic;

  begin
            
    -- case statement to calculate the Pi and Gi terms 
    case FUNT is
      when "000" | "011" | "100" | "111" =>
        RTEMP := RBUS;
        STEMP := SBUS;
      when "001" | "101" | "110" =>
        RTEMP := not RBUS;
        STEMP := SBUS;
      when "010" =>
        RTEMP := RBUS;
        STEMP := not SBUS;
      when others => null;
    end case;

    PTERM := RTEMP or STEMP;
    GTERM := RTEMP and STEMP;

    P3    := PTERM(3);
    P32   := P3 and PTERM(2);
    P321  := P32 and PTERM(1);
    P3210 := P321 and PTERM(0);
                  
    -- Generate the carry terms
    CTERM(0) := GTERM(0) or (PTERM(0) and CIN);
    CTERM(1) := GTERM(1) or (PTERM(1) and CTERM(0));
    CTERM(2) := GTERM(2) or (PTERM(2) and CTERM(1));
    CTERM(3) := GTERM(3) or (PTERM(3) and CTERM(2));

    case FUNT is 
      when "000" | "001" | "010"  => 
        RESULT(0) := RTEMP(0) xor STEMP(0) xor CIN;
        RESULT(1) := RTEMP(1) xor STEMP(1) xor CTERM(0);
        RESULT(2) := RTEMP(2) xor STEMP(2) xor CTERM(1);
        RESULT(3) := RTEMP(3) xor STEMP(3) xor CTERM(2);
        P   <= not P3210;
        G   <= not (GTERM(3) or (P3 and GTERM(2)) or (P32 and GTERM(1))
               or (P321 and GTERM(0)));
        CO  <= CTERM(3);
        OVR <= CTERM(3) xor CTERM(2);

      when "011" =>   
        RESULT := RTEMP or STEMP;
        P   <= '0';
        G   <= P3210;
        CO  <= not P3210 or CIN;
        OVR <= not P3210 or CIN;

      when "100" | "101" =>
        RESULT := RTEMP and STEMP;
        P   <= '0';
        G   <= not (GTERM(3) or GTERM(2) or GTERM(1) or GTERM(0));
        CO  <= GTERM(3) or GTERM(2) or GTERM(1) or GTERM(0) or CIN;
        OVR <= GTERM(3) or GTERM(2) or GTERM(1) or GTERM(0) or CIN;

      when "110" | "111" => 
        RESULT := not RTEMP xor STEMP;
        
        P <= GTERM(3) or GTERM(2) or GTERM(1) or GTERM(0);
        G <= GTERM(3) or (P3 and GTERM(2)) or (P32 and GTERM(1)) 
             or (P321 AND GTERM(0));
        CO <= not (GTERM(3) or (P3 and GTERM(2)) or (P32 and GTERM(1))
              or (P321 and (GTERM(0) or not CIN)));
              
        GTERM := not GTERM;
        OVR <= (PTERM(2) or (GTERM(2) and PTERM(1)) 
               or (GTERM(2) and GTERM(1) and PTERM(0)) 
               or (GTERM(2) and GTERM(1) and GTERM(0) and CIN))
               xor (PTERM(3) or (GTERM(3) and PTERM(2)) 
               or (GTERM(3) and GTERM(2) and PTERM(1))
               or (GTERM(3) and GTERM(2) and GTERM(1) and PTERM(0)) 
               or (GTERM(3) and GTERM(2) and GTERM(1) and GTERM(0) and CIN));
      
      -- set all alu outputs to unknowns
      when others => 
        if (UNINIT_ON_BUS(RBUS) or UNINIT_ON_BUS(SBUS)) then
          RESULT := "UUUU";
          P   <= 'U';
          G   <= 'U';
          CO  <= 'U';
          OVR <= 'U';
        else
          RESULT := "XXXX";
          P    <= 'X';
          G    <= 'X';
          CO   <= 'X';
          OVR  <= 'X';
        end if;
                
    end case;
          
    -- make F signal assignments
    if (RESULT = "0000") then
      F <= '1';
    elsif (UNINIT_ON_BUS(RESULT)) then
      F <= 'U';
    else
      F <= 'Z';
    end if;

    FBUS <= RESULT;
    F3   <= RESULT(3);

  end process;

end BEHAVIORAL;

 
