--FILENAME: wgenerator_Am29C01.vhd
--
--                                   DISCLAIMER
--
--        This code is the sole property of the Institute for Technology
--        Development (ITD), Jackson, Mississippi, and is distributed for
--        the purpose of providing examples of VHDL models written to
--        modeling standards.  This code may not be used for commercial
--        purposes, and may not be redistributed without permission from
--        the Institute for Technology Development.  ITD assumes no 
--        responsibility for errors, omissions, uses made, or decisions
--        based on its use.  No warranties, expressed or implied, are given.
--
--        This is a modified IEEE WAVES package.  It was written specifically
--        to apply test vectors to the model being tested.
--
--        ------------------------------------------------------------------

use STD.TEXTIO.all ;

library WAVES_STANDARD ;
use WAVES_STANDARD.WAVES_SYSTEM.all ;
use WAVES_STANDARD.WAVES_INTERFACE.all ;

library WORK ;
use WORK.WAVES_DEVICE.all ;
use WORK.WAVES_OBJECTS.all ;
use WORK.WAVES_FRAMES.all ;
use WORK.WAVES_UTILITIES.all ;

package WGENERATOR_Am29C01 is
    procedure Am29C01_TEST(
      signal CONNECT   : inout Waves_Port_List) ;

    procedure Am29C01_TEST(
      signal CONNECT   : inout Waves_Port_List ;
      constant INPUT_FILE_NAME  : string ;
      constant START_TOLERANCE  : time ;
      constant END_TOLERANCE    : time ) ;

    procedure Am29C01_Test(
      signal CONNECT   : inout Waves_Port_List ;
      constant INPUT_FILE_NAME  : string ;
      constant PERIOD  : time ;
      constant UNCERTAINTY : real ) ;

end WGENERATOR_Am29C01 ;
-- ======================================================================

package body WGENERATOR_Am29C01 is

    constant O_PINS : Pinset := NEW_PINSET
      (( 
         TP_Y0, TP_Y1, TP_Y2, TP_Y3, TP_Co, TP_G, TP_P, TP_OVR, TP_F, TP_F3
      )) ;

    constant I_PINS : Pinset := ALL_PINS and not O_PINS ;

    constant EMPTY_EVENT : integer := -1 ;

  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs) are contained within
  --  the procedure, and are analyzed with the VHDL code.  The test vectors
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After 95 ns (95% of period):  Expected outputs set to appropriate values
  --
  --   After 100 ns (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
    procedure Am29C01_TEST(
      signal CONNECT   : inout Waves_Port_List) is

      constant PERIOD : Time := 100 ns ;
      constant UNCERTAINTY_PCT : real := 0.95 ;
      constant TDELAY : Event_Time := ETIME (UNCERTAINTY_PCT * PERIOD);

      constant FSA : Frame_Set_Array := 
          NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
          NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS) ;

      constant DT_BASIS : Delay_Time_Basis := (false, 0 ns, 0 ns, Timed_Delay, 0, 0, 0) ;

      variable TD1 : Time_Data := NEW_TIME_DATA ( FSA ) ;
      variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1) ;

--        Order of signals:
--
--        (Y0,   Y1,   Y2,   Y3,
--         Co,   G,    P,    OVR,   F, F3
--         A3,   Ram3, Q0,   Ram0,
--         A0,   A1,   A2,   A3,
--         B0,   B1,   B2,   B3,
--         D0,   D1,   D2,   D3,
--         Cn,
--         I0,   I1,   I2,
--         I6,   I7,   I8,
--         I3,   I4,   I5,
--         OE,   CP)
--

    variable WAVEFORM : Wtime_Slice_List (1 to 2) := (
        (  "UUUU"
         & "UUUUUU"
         & "UUUU"
         & "UUUU"
         & "UUUU"
         & "UUUU"
         & "U"
         & "UUU"
         & "UUU"
         & "UUU"
         & "UU", T1),

        (  "XXXX"
         & "XXXXXX"
         & "XXXX"
         & "XXXX"
         & "XXXX"
         & "XXXX"
         & "X"
         & "XXX"
         & "XXX"
         & "XXX"
         & "XX", T1)
    ) ;

    begin
        for I in WAVEFORM'range loop
            APPLY(CONNECT, WAVEFORM(I).CODES, WAVEFORM(I).WTIME, ALL_PINS) ;
        end loop ;
    end ;

  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The test vectors (inputs and expected outputs, along with the period
  --  of each slice) are read in from an external file.  The test vectors 
  --  are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After <START_TOLERANCE>:  Expected outputs set to appropriate values
  --
  --   After <FS_TIME - END_TOLERANCE>: Get next set of inputs and outputs
  --
  --   After <FS_TIME> (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
    procedure Am29C01_Test(
      signal CONNECT   : inout Waves_Port_List ;
      constant INPUT_FILE_NAME  : string ;
      constant START_TOLERANCE  : time ;
      constant END_TOLERANCE    : time ) is


      variable TOLERANCE : time := START_TOLERANCE + END_TOLERANCE ;

      constant FSA : Frame_Set_Array :=
          NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
          NEW_FRAME_SET_ARRAY (OUTPUT_FRAMES, O_PINS) ;

      constant DT_BASIS : Delay_Time_Basis := (false, 0 ns, 0 ns, Timed_Delay, 0, 0, 0) ;

      variable TD : Time_Data := NEW_TIME_DATA (FSA) ;

      constant FSA_DONT_CARE : Frame_Set_Array :=
          NEW_FRAME_SET_ARRAY (INPUT_FRAMES, I_PINS) +
          NEW_FRAME_SET_ARRAY (DONT_CARE_FRAMES, O_PINS) ;

      variable TD_DONT_CARE : Time_Data := NEW_TIME_DATA (FSA_DONT_CARE) ;

      file INPUT_FILE : text is in INPUT_FILE_NAME ;

        -- the file slice must be allocated
      variable Am29C01_FILE_SLICE : File_Slice := NEW_FILE_SLICE ;

    begin

      loop
          READ_FILE_SLICE(INPUT_FILE, Am29C01_FILE_SLICE) ;

          exit when Am29C01_FILE_SLICE.END_OF_FILE ;

          if Am29C01_FILE_SLICE.FS_TIME > TOLERANCE then
              APPLY(CONNECT, Am29C01_FILE_SLICE.CODES.all, DELAY(START_TOLERANCE), TD_DONT_CARE) ;
              APPLY(CONNECT, Am29C01_FILE_SLICE.CODES.all, DELAY(Am29C01_FILE_SLICE.FS_TIME - TOLERANCE), TD) ;
              APPLY(CONNECT, Am29C01_FILE_SLICE.CODES.all, DELAY(END_TOLERANCE), TD_DONT_CARE) ;
          else
              APPLY(CONNECT, Am29C01_FILE_SLICE.CODES.all, DELAY(Am29C01_FILE_SLICE.FS_TIME), TD) ;
          end if ;
      end loop ;

    end ;

  ---------------------------------------------------------------------------
  --  The following procedure is used to provide test vectors for the model.
  --  The functional test vectors (inputs and corresponding expected outputs, 
  --  of each slice) are read in from an external file (see "A User's Guide
  --  to WAVES", December 10, 1990, WAVES Analysis and Standardization Group,
  --  for information on WAVES level 1 external file format).  It is assumed 
  --  that the inputs values of each slice cause the output values of the same
  --  slice to occur within the period given, which is a parameter to this
  --  procedure.  The test vectors are applied as follows:
  --
  --   Beginning of period:  Inputs set to appropriate values
  --                         Expected outputs set to "don't care"
  --
  --   After <UNCERTAINTY> * <PERIOD>:  Expected outputs set to appropriate 
  --                                    values (and checked in the comparator)
  --
  --   After <PERIOD> (end of period): Get next set of inputs and outputs
  ---------------------------------------------------------------------------
    procedure Am29C01_Test(
      signal CONNECT   : inout Waves_Port_List ;
      constant INPUT_FILE_NAME  : string ;
      constant PERIOD  : time ;
      constant UNCERTAINTY : real ) is


      constant TDELAY : Event_Time := ETIME (UNCERTAINTY * PERIOD);

      constant FSA : Frame_Set_Array := 
          NEW_FRAME_SET_ARRAY ( INPUT_FRAMES, I_PINS) +
          NEW_FRAME_SET_ARRAY ( STROBED_OUTPUT_FRAMES(TDELAY), O_PINS) ;

      constant DT_BASIS : Delay_Time_Basis := (false, 0 ns, 0 ns, Timed_Delay, 0, 0, 0) ;

      variable TD1 : Time_Data := NEW_TIME_DATA ( FSA ) ;
      variable T1  : Wave_Timing := ((PERIOD, DT_BASIS), TD1) ;

      file INPUT_FILE : text is in INPUT_FILE_NAME ;

        -- the file slice must be allocated
      variable Am29C01_FILE_SLICE : File_Slice := NEW_FILE_SLICE ;
      variable WAVEFORM : Wtime_Slice ;

    begin

      loop
          READ_FILE_SLICE(INPUT_FILE, Am29C01_FILE_SLICE) ;

          exit when Am29C01_FILE_SLICE.END_OF_FILE ;

          WAVEFORM := (Am29C01_FILE_SLICE.CODES.all, T1) ;

          APPLY(CONNECT, WAVEFORM.CODES, WAVEFORM.WTIME, ALL_PINS) ;

      end loop ;

    end ;

end WGENERATOR_Am29C01 ;
