--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  Architecture for the test bench for the timer module 
--  
--  (file timer_test-bench.vhd)
--------------------------------------------------------------------------

ARCHITECTURE bench OF timer_test IS

  COMPONENT timer
    port (a_bus : in std_logic_vector(3 downto 2); 
          d_bus  : inout dlx_half bus;              
          clk    : in std_logic;
          enable : in std_logic;
          rw     : in std_logic;
	  reset  : in std_logic;
          ready  : out std_logic;
	  irq_out: out std_logic_vector(0 to 1) );
  END COMPONENT;

  component clockgen
    PORT (phi1, phi2 : OUT std_logic ;    -- two-phase non-overlapping clocks
      	  reset :      OUT std_logic);
  end component;
  	  
  FOR testtimer : timer
    USE ENTITY WORK.timer(behaviour)
    	GENERIC MAP (tpd => 1 ns);

  FOR testclock : clockgen
    USE ENTITY WORK.clockgen(dataflow)
    	GENERIC MAP (tpw => 8 ns,          -- clock pulse width
      	             tps => 2 ns);         -- clock pulse separation
     					   	
  SIGNAL d : dlx_half;
  signal a : std_logic_vector(3 downto 2);
  signal phi1_sig : std_logic;
  signal phi2_sig : std_logic;
  signal en_sig  : std_logic;
  signal rw_sig  : std_logic;
  signal res_sig : std_logic;
  signal rdy_sig : std_logic;
  signal irq_sig : std_logic_vector(0 to 1);  

  constant tpd_bench : Time := 1 ns;
  
BEGIN

  testclock : clockgen
    PORT MAP (phi1 => phi1_sig, phi2 => phi2_sig, reset => res_sig);
  
  testtimer : timer
    PORT MAP (a_bus => a, d_bus => d, clk => phi1_sig, enable => en_sig,
              rw => rw_sig, reset => res_sig, ready => rdy_sig, irq_out => irq_sig);

  test: process
  
    variable data_w : dlx_half;
    variable L : line;

    procedure ti_write (address : in std_logic_vector(3 downto 2);
		        data : in dlx_half) is
    begin -- write
      a <= address after tpd_bench;
      d <= data after tpd_bench;
      en_sig <= '1' after tpd_bench;
      rw_sig <= '0' after tpd_bench;
      wait until rdy_sig = '1';                 -- wait until data accepted
      d <= (others => 'Z') after tpd_bench;
      a <= (others => 'Z') after tpd_bench;
      en_sig <= '0' after tpd_bench;
      wait until rdy_sig = 'Z';		        -- wait until timer is idle again
    end ti_write;

    procedure ti_read (address : in std_logic_vector(3 downto 2);
		       data : out dlx_half) is
    begin -- read
      a <= address after tpd_bench;
      en_sig <= '1' after tpd_bench;
      rw_sig <= '1' after tpd_bench;
      wait until rdy_sig = '1';	                -- wait until data accepted
      data := d;
      a <= (others => 'Z') after tpd_bench;
      en_sig <= '0' after tpd_bench;
      wait until rdy_sig = 'Z';		        -- wait until ram is idle again  
    end ti_read;

  begin		-- test
    -------------------------------------------------------------------------
    wait until res_sig = '0' and rdy_sig = 'Z';

    write(L, string'(">>> Test read/write to all registers and sync. reset"));
    writeline(output, L);
    
    write(L, string'("load counter_0 with X""000f"":"));
    writeline(output, L);
    ti_write("01", To_StdLogicVector(X"000f"));
    write(L, string'("Read counter_0:"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000");
    ti_read("01", data_w);
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("load counter_1 with X""0005"":"));
    writeline(output, L);
    ti_write("10", To_StdLogicVector(X"0005"));
    write(L, string'("Read counter_1:"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000");
    ti_read("10", data_w);
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("load status_reg with X""feff"":"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"1111_1110_1111_1111"));
    write(L, string'("Read status_reg:"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000");
    ti_read("00", data_w);
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("load status_reg with X""ffff"" (incl. sync. reset):"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"1111_1111_1111_1111"));
    write(L, string'("Read status_reg:"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000");
    ti_read("00", data_w);
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read counter_0:"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000");
    ti_read("01", data_w);
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read counter_1:"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000");
    ti_read("10", data_w);
    write(L, data_w, X, up); 
    writeline(output, L);
-----------------------------------------------------------------------
    write(L, string'(">>> Test two single counters with and without interrupt"));
    writeline(output, L);

    write(L, string'("load counter_0 with X""000f"":"));
    writeline(output, L);
    ti_write("01", To_StdLogicVector(X"000f"));

    write(L, string'("load counter_1 with X""0005"":"));
    writeline(output, L);
    ti_write("10", To_StdLogicVector(X"0005"));

    write(L, string'("enable both counters, mask both interrupts, count every cycle:"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"0011_1100_0000_0000"));    

    wait until irq_sig(1) = '1';
    write(L, string'("irq1 is active"));
    writeline(output, L);    
    wait until irq_sig(0) = '1';
    write(L, string'("irq0 is active"));
    writeline(output, L);

    write(L, string'("Read status_reg"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000");
    ti_read("00", data_w);
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("clear status reg"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"0000_0000_0000_0000"));

    write(L, string'("load counter_0 with X""0009"":"));
    writeline(output, L);
    ti_write("01", To_StdLogicVector(X"0009"));

    write(L, string'("load counter_1 with X""000E"":"));
    writeline(output, L);
    ti_write("10", To_StdLogicVector(X"000E"));
 
    write(L, string'("enable both counters, no interrupt masked, count every cycle:"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"0000_1100_0000_0000"));    

    loop
      write(L, string'("Read counter_1:"));
      writeline(output, L);
      data_w := To_StdLogicVector(X"0000");
      ti_read("10", data_w);
      exit when data_w = To_StdLogicVector(X"0000") or irq_sig /= "00";
    end loop;
    write(L, string'("irq_sig: "));
    write(L, irq_sig, B, up);
    writeline(output, L);    
   
    write(L, string'("clear status reg"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"0000_0000_0000_0000"));
    
  -----------------------------------------------------------------------
    write(L, string'(">>> Test two single counters with interrupt"));
    writeline(output, L);
    write(L, string'("    switch clock divide and count enable while counting"));
    writeline(output, L);
    
    write(L, string'("load counter_0 with X""0013"":"));
    writeline(output, L);
    ti_write("01", To_StdLogicVector(X"0013"));

    write(L, string'("load counter_1 with X""0017"":"));
    writeline(output, L);
    ti_write("10", To_StdLogicVector(X"0017"));

    write(L, string'("enable both counters, mask both interrupts, count after 4 cycles:"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"0011_1100_0000_0100"));        

    wait for 200 ns; -- 10 clock cycles / 2 count cycles

    write(L, string'("enable both counters, mask both interrupts, count after 2 cycles:"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"0011_1100_0000_0010"));        

    wait for 120 ns; -- 6 clock cycles / 2 count cycles

    write(L, string'("disable both counters, mask both interrupts, count after 2 cycles:"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"0011_0000_0000_0010"));        

    wait for 80 ns; -- 4 clock cycles

    write(L, string'("enable both counters, mask both interrupts, count after 1 cycle:"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"0011_1100_0000_0001"));        

    wait for 60 ns; -- 3 clock cycles / 2 count cycles

    write(L, string'("enable both counters, mask both interrupts, count every cycle:"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"0011_1100_0000_0000"));        

    wait until irq_sig(0) = '1';
    write(L, string'("irq0 is active"));
    writeline(output, L);    
    wait until irq_sig(1) = '1';
    write(L, string'("irq1 is active"));
    writeline(output, L);

    write(L, string'("clear status reg"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"0000_0000_0000_0000"));

  -----------------------------------------------------------------------
    write(L, string'(">>> Test one counter (double length) with interrupt"));
    writeline(output, L);
    
    write(L, string'("load counter_0 with X""1234"":"));
    writeline(output, L);
    ti_write("01", To_StdLogicVector(X"1234"));

    write(L, string'("load counter_1 with X""0001"":"));
    writeline(output, L);
    ti_write("10", To_StdLogicVector(X"0001"));

    write(L, string'("enable both counters, mask both interrupts, count every cycle:"));
    writeline(output, L);
    ti_write("00", To_StdLogicVector(B"0011_1110_0000_0000"));

    wait until irq_sig /= "00";
    write(L, string'("irq_sig: "));
    write(L, irq_sig, B, up);
    writeline(output, L);
    
    wait;
  end process test;

END bench;






