--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  Behaviour architecture of the timer module
--  
--  (file timer-behaviour.vhd)
--------------------------------------------------------------------------

architecture behaviour of timer is

  signal counter_0 : dlx_half register;
  signal counter_1 : dlx_half register;
  signal status_reg: dlx_half register;

  signal count_clk : std_logic;
  signal int_ready : std_logic;
  signal int_irq_out : std_logic_vector(0 to 1);
  
  alias intrpt0 : std_logic is status_reg(0);
  alias intrpt1 : std_logic is status_reg(1);
  alias mask0   : std_logic is status_reg(2);
  alias mask1   : std_logic is status_reg(3);
  alias count0  : std_logic is status_reg(4);
  alias count1  : std_logic is status_reg(5);
  alias double  : std_logic is status_reg(6);
  alias sync_res: std_logic is status_reg(7);
  alias divide  : dlx_byte is status_reg(8 to 15);
  
begin
 
  timer_register : process(clk, reset, count_clk, int_irq_out)
  -- sensitive to signals 'clk', 'count_clk' 'irq_out' and  'reset' 
  -- source for signals 'status_reg', 'counter_0', 'counter1', 'd_bus', 'ready'

    variable int_cnt0 : natural;
    variable int_cnt1 : natural;
    
    procedure exec_count is
    begin
      int_cnt0 := sv_to_natural(counter_0);
      int_cnt1 := sv_to_natural(counter_1);	
      if double = '0' then
 	--
	-- two indepemdemt counters
	--
	if count0 = '1' and  int_cnt0 /= 0 then
	  int_cnt0 := int_cnt0 -1;
	end if;
        if count1 = '1' and  int_cnt1 /= 0 then
	  int_cnt1 := int_cnt1 -1;
	end if;	
      elsif double = '1' then
	--
	-- one counter with double length (counter_1 is high word)
	--	
        if count0 = '1' and count1 ='1' then
	  if int_cnt0 /= 0 then
	    int_cnt0 := int_cnt0 - 1;
	  elsif int_cnt0 = 0 and int_cnt1 /= 0 then
	      int_cnt0 := 16#ffff#;
	      int_cnt1 := int_cnt1 - 1;
          end if;
        end if;
      end if;  	
    end exec_count;
    
  begin                               -- process timer_register
    if reset = '1' or (sync_res = '1' and rising_edge(clk)) then
      --
      -- asynchronous reset
      --
      status_reg <= To_StdLogicVector(X"0000");
      counter_0  <= To_StdLogicVector(X"0000");
      counter_1  <= To_StdLogicVector(X"0000");
      d_bus <= (others => 'Z') after tpd;
      int_ready <= 'Z';

    elsif clk = '1' then
      --
      -- check for new actions
      --
      if enable = '1' and rw = '0' and int_ready = 'Z' then
        --
        -- write access to timer
        --
        int_ready <= '0';
	if a_bus = "00" then
    	  status_reg <= d_bus after tpd;
	elsif a_bus = "01" and count0 = '0' then
	  counter_0 <= d_bus after tpd;
	elsif a_bus = "10" and count1 = '0' then
	  counter_1 <= d_bus after tpd;
	end if;
	int_ready <= '1';
      elsif enable = '1' and rw = '1' and int_ready = 'Z' then
	--
	-- read access to timer
	--
        int_ready <= '0';
	case a_bus is
	  when "00" => d_bus <= status_reg(0 to 15) after tpd;
	  when "01" => d_bus <= counter_0 after tpd;
	  when "10" => d_bus <= counter_1 after tpd;
	  when others => d_bus <= (others => 'Z') after tpd;
	end case;
	int_ready <= '1';
      elsif enable ='0' and int_ready = '1' then
  	--
        -- data accepted, reinitialize outputs
        --
	d_bus <= (others => 'Z') after tpd;
	int_ready <= 'Z';
      end if;

      if enable = '0' or rw = '1' or a_bus = "00" then
	--
	-- functions that can't be executed while writing into counters
        -- (set intrptx bits in status_reg / count operation)
        --
	if int_irq_out(0) = '1' then
 	  intrpt0 <= '1';         
 	end if;
	if int_irq_out(1) = '1' then
 	  intrpt1 <= '1';
 	end if;

	if count_clk = '1' then
	  exec_count;
	  counter_0 <= natural_to_sv(int_cnt0, 16);
          counter_1 <= natural_to_sv(int_cnt1, 16);
	end if;
      end if;

    end if;
  end process;

  -------------------------------------------------------------------
  gen_count_clk: process(clk, divide) 
  -- sensitive to signal 'clk' and 'divide'
  -- source for signal 'count_clk'
    variable tmp_count : natural;
  begin
    if reset ='0' then
      if not divide'stable then
        tmp_count := sv_to_natural(status_reg(8 to 15));
      end if;

      if rising_edge(clk) and (count0 = '1' or count1 = '1') then
        if tmp_count = 0 then
	  tmp_count := sv_to_natural(status_reg(8 to 15));
	  count_clk <= '1';
	else
  	  tmp_count := tmp_count - 1;
	end if;
      end if;
      
      if falling_edge(clk) then
	 count_clk <= '0'; 	  
      end if;
    end if;
  end process;

  -------------------------------------------------------------------
  interrupt: process(clk, reset)
  -- sensitive to signal 'clk' and 'reset'
  -- source for signal 'irq_out'
  begin
    if reset = '1' then
      int_irq_out <= "00";
    else
      if clk = '1' then
        --
	-- test interrupt conditions for activating interrupt
	--
        if mask0 = '1' and intrpt0 = '0' and sv_to_natural(counter_0) = 0 then
          if double = '0' then
	    int_irq_out(0) <= '1';
          elsif double = '1' and sv_to_natural(counter_1) = 0 then
	    int_irq_out(0) <= '1';
	  end if;	
        end if;
        if mask1 = '1' and intrpt1 = '0' and sv_to_natural(counter_1) = 0 then
          if double = '0' then
	    int_irq_out(1) <= '1';
          end if;
	end if;
        --
	-- clear interrupts
	--
	if intrpt0 = '1'  or sync_res = '1' then
 	  int_irq_out(0) <= '0';         
 	end if;
	if intrpt1 = '1'  or sync_res = '1' then
 	  int_irq_out(1) <= '0';
 	end if;
      end if;
    end if;
  end process;

  ready <= int_ready after tpd;              -- connect internal signal to output
  irq_out <= int_irq_out after tpd;          -- connect internal signals to output
  
end behaviour;





