--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  Architecture for test bench for the ram module 
--  
--  (file ram_test-bench.vhd)
--------------------------------------------------------------------------

ARCHITECTURE bench OF ram_test IS

  COMPONENT ram
    PORT (a_bus : in dlx_address;
          d_bus : inout dlx_word bus;
	  enable: in std_logic_vector(0 to 3);
          rw    : in std_logic;
   	  ready : out std_logic_vector(0 to 3) );
  END COMPONENT;
  
  FOR testram : ram
    USE ENTITY WORK.ram(structural)
    	GENERIC MAP (tpd => 1 ns, tac => 1 ns, mem_size => 64);
 
  SIGNAL d : dlx_word;
  signal a : dlx_address;
  signal en_sig: std_logic_vector(0 to 3);
  signal rw_sig : std_logic;
  signal rdy_sig : std_logic_vector(0 to 3);

  constant tpd_bench : Time := 1 ns;
  
BEGIN

  testram : ram
    PORT MAP (d_bus => d, a_bus => a, enable => en_sig,
              rw => rw_sig, ready => rdy_sig);
    
  test: process
  
    variable data_w : std_logic_vector(0 to 31);
    variable L : line;

    procedure mem_write (address : in dlx_address;
		         data : in dlx_word;
			 enab : in std_logic_vector(0 to 3) ) is
    begin -- write
      a <= address after tpd_bench;
      d <= data after tpd_bench;
      en_sig <= enab after tpd_bench;
      rw_sig <= '0' after tpd_bench;
      wait until rdy_sig(0) = '1' or rdy_sig(1) = '1' or
                 rdy_sig(2) = '1' or rdy_sig(3) = '1';	-- wait until data accepted
      d <= (others => 'Z') after tpd_bench;
      a <= (others => 'Z') after tpd_bench;
      en_sig <= "0000" after tpd_bench;
      wait until rdy_sig = "ZZZZ";		        -- wait until ram is idle again
    end mem_write;

    procedure mem_read (address : in dlx_address;
		        data : out dlx_word;
			enab : in std_logic_vector(0 to 3) ) is
    begin -- read
      a <= address after tpd_bench;
      en_sig <= enab after tpd_bench;
      rw_sig <= '1' after tpd_bench;
      wait until rdy_sig(0) = '1' or rdy_sig(1) = '1' or
                 rdy_sig(2) = '1' or rdy_sig(3) = '1';	-- wait until data accepted
      data := d;
      a <= (others => 'Z') after tpd_bench;
      en_sig <= "0000" after tpd_bench;
      wait until rdy_sig = "ZZZZ";		        -- wait until ram is idle again  
    end mem_read;

  begin		-- test
    -------------------------------------------------------------------------
    wait until rdy_sig = "ZZZZ";

    write(L, string'("Write word X""00000004"" to 4:"));
    writeline(output, L);
    mem_write(To_StdLogicVector(X"0000_0004"),
              To_StdLogicVector(X"0000_0004"), "1111");
    write(L, string'("Write word X""89ABCDEF"" to 0:"));
    writeline(output, L);
    mem_write(To_StdLogicVector(X"0000_0000"),
              To_StdLogicVector(X"89AB_CDEF"), "1111");

    write(L, string'("Read word from X""00000004"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0004"), data_w, "1111");
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read word from X""000000000"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0000"), data_w, "1111");
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);
    
    -------------------------------------------------------------------------

    write(L, string'("Write halfword X""2222"" to 0:"));
    writeline(output, L);
    mem_write(To_StdLogicVector(X"0000_0000"),
              To_StdLogicVector(X"2222_0000"), "1100");
 
    write(L, string'("Write halfword X""3333"" to 2:"));
    writeline(output, L);
    mem_write(To_StdLogicVector(X"0000_0002"),
              To_StdLogicVector(X"0000_3333"), "0011");
    
    write(L, string'("Read word from X""00000000"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0000"), data_w, "1111");
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read halfword from X""00000000"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0000"), data_w, "1111");
    data_w := sv_zext(data_w(0 to 15), 32);
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read halfword from X""00000002"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0002"), data_w, "1111");
    data_w := sv_zext(data_w(16 to 31), 32);
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read halfword from X""00000001"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0001"), data_w, "1111");
    data_w := sv_zext(data_w(0 to 15), 32);
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read halfword from X""00000003"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0003"), data_w, "1111");
    data_w := sv_zext(data_w(16 to 31),32);
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    -------------------------------------------------------------------------

    write(L, string'("Write word X""ccddeeff"" to 4:"));
    writeline(output, L);
    mem_write(To_StdLogicVector(X"0000_0004"),
              To_StdLogicVector(X"ccddeeff"), "1111");

    write(L, string'("Write byte X""44"" to 4"));
    writeline(output, L);
    mem_write(To_StdLogicVector(X"0000_0004"),
              To_StdLogicVector(X"44_00_00_00"), "1000");
    write(L, string'("Write byte X""55"" to 5"));
    writeline(output, L);
    mem_write(To_StdLogicVector(X"0000_0005"),
              To_StdLogicVector(X"00_55_00_00"), "0100");
    write(L, string'("Write byte X""66"" to 6"));
    writeline(output, L);
    mem_write(To_StdLogicVector(X"0000_0006"),
              To_StdLogicVector(X"00_00_66_00"), "0010");
    write(L, string'("Write byte X""77"" to 7"));
    writeline(output, L);
    mem_write(To_StdLogicVector(X"0000_0007"),
              To_StdLogicVector(X"00_00_00_77"), "0001");

    write(L, string'("Read word from X""00000004"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0004"), data_w, "1111");
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read halfword from X""00000004"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0004"), data_w, "1111");
    data_w := sv_zext(data_w(0 to 15), 32);
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read halfword from X""00000006"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0006"), data_w, "1111");
    data_w := sv_zext(data_w(16 to 31), 32);
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read byte from X""00000004"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0004"), data_w, "1111");
    data_w := sv_zext(data_w(0 to 7), 32);
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);
    
    write(L, string'("Read byte from X""00000005"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0005"), data_w, "1111");
    data_w := sv_zext(data_w(8 to 15), 32);
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read byte from X""00000006"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0006"), data_w, "1111");
    data_w := sv_zext(data_w(16 to 23), 32);
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    write(L, string'("Read byte from X""00000007"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0000_0007"), data_w, "1111");
    data_w := sv_zext(data_w(24 to 31), 32);
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

    -------------------------------------------------------------------------

    -- This should hang
    write(L, string'("Read word from X""00100000"":"));
    writeline(output, L);
    data_w := To_StdLogicVector(X"0000_0000");
    mem_read(To_StdLogicVector(X"0010_0000"), data_w, "1111");
    write(L, string'("  result:"));
    write(L, data_w, X, up); 
    writeline(output, L);

  wait;
  end process test;

END bench;






