--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  *** SYNOPSYS synthesizable code (ver. 3.2.a) ***
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  Package body for DLX instructions
--  
--  (file dlx_instructions-body.vhd)
--------------------------------------------------------------------------

--synopsys synthesis_off
library IEEE_EXTD;
use IEEE_EXTD.stdl1164_vector_arithmetic.sv_to_natural,
    IEEE_EXTD.stdl1164_vector_arithmetic.sv_to_integer;
--synopsys synthesis_on

package body dlx_instructions is
--synopsys synthesis_off

  constant opcode_names : name_array(0 to 63) :=
    ( "RR_ALU ",  "LHU_I  ",  "LBU_I  ",  "UD__03 ",
      "LW_I   ",  "LH_I   ",  "LB_I   ",  "UD__07 ",
      "SW_I   ",  "SH_I   ",  "SB_I   ",  "UD__0B ",
      "J      ",  "JAL    ",  "JR     ",  "JALR   ", 
      "BEQZ   ",  "BNEZ   ",  "RFE    ",  "TRAP   ", 
      "ADD_I  ",  "ADDU_I ",  "SUB_I  ",  "SUBU_I ",
      "AND_I  ",  "OR_I   ",  "XOR_I  ",  "LHI    ",
      "SLL_I  ",  "UD__1D ",  "SRL_I  ",  "SRA_I  ",
      "SEQ_I  ",  "SEQU_I ",  "SNE_I  ",  "SNEU_I ", 
      "SLT_I  ",  "SLTU_I ",  "SLE_I  ",  "SLEU_I ",
      "SGT_I  ",  "SGTU_I ",  "SGE_I  ",  "SGEU_I ",    
      "UD__2C ",  "UD__2D ",  "UD__2E ",  "UD__2F ",
      "LF_I   ",  "LD_I   ",  "UD__32 ",  "UD__2F ",
      "SF_I   ",  "SD_I   ",  "UD__36 ",  "UD__37 ",
      "BFPT   ",  "BFPF   ",  "UD__3A ",  "UD__3B ",
      "FP_FN  ",  "UD__3D ",  "UD__3E ",  "UD__3F ");

  constant rr_func_names : name_array(0 to 63) :=
    ( "NOP    ",  "UD__01 ",  "MOVI2S ",  "MOVS2I ",
      "ADD    ",  "ADDU   ",  "SUB    ",  "SUBU   ",
      "AND    ",  "OR     ",  "XOR    ",  "LH     ",
      "SLL    ",  "UD__0D ",  "SRL    ",  "SRA    ",
      "SEQ    ",  "SEQU   ",  "SNE    ",  "SNEU   ", 
      "SLT    ",  "SLTU   ",  "SLE    ",  "SLEU   ",
      "SGT    ",  "SGTU   ",  "SGE    ",  "SGEU   ",  
      "MOVF   ",  "MOVD   ",  "MOVFP2I",  "MOVI2FP",
      "LW     ",  "LH     ",  "LB     ",  "UD__23 ",
      "UD_24  ",  "LHU    ",  "LBU    ",  "UD__27 ",
      "SW     ",  "SH     ",  "SB     ",  "UD__2B ",
      "LF     ",  "LD     ",  "UD__2E ",  "UD__2F ",
      "SF     ",  "SD     ",  "UD__32 ",  "UD__33 ",
      "UD__34 ",  "UD_35  ",  "UD__36 ",  "UD__37 ",
      "UD__38 ",  "UD_39  ",  "UD__3A ",  "UD__3B ",
      "UD__3C ",  "UD_3D  ",  "UD__3E ",  "UD__3F ");

  constant fp_func_names : name_array(0 to 31) :=
    ( "ADDF   ",  "SUBF   ",  "MULTF  ",  "DIVF   ",
      "ADDD   ",  "SUBD   ",  "MULTD  ",  "DIVD   ",
      "CVTF2D ",  "CVTF2I ",  "CVTD2F ",  "CVTD2I ",
      "CVTI2F ",  "CVTI2D ",  "UD__0E ",  "UD__0F ",
      "MULT   ",  "MULTU  ",  "DIV    ",  "DIVU   ",
      "EQF    ",  "NEF    ",  "LTF    ",  "LEF    ",
      "GTF    ",  "GEF    ",  "EQD    ",  "NED    ",
      "LTD    ",  "LED    ",  "GTD    ",  "GED    ");


  procedure write_reg (L : inout line; reg : reg_index) is
  begin
    write(L, 'R');
    write(L, reg);
  end write_reg;


  procedure write_freg (L : inout line; freg : reg_index) is
  begin
    write(L, 'F');
    write(L, freg);
  end write_freg;


  procedure write_special_reg (L : inout line; reg : reg_index) is
  begin
    case reg is
      when 1 =>
        write(L, string'("ICR"));
      WHEN 2 =>
        write(L, string'("IAR"));
      when 4 =>
        write(L, string'("TBR"));	
      when 8 =>
        write(L, string'("FPSR"));	
      when others =>
        write(L, string'("Undefined special register "));
        write(L, reg);
    end case;
  end write_special_reg;


  procedure write_instr (L : inout line; instr : in dlx_word) is

    alias instr_opcode  : dlx_opcode is instr(0 to 5);
    alias instr_rr_func : dlx_rr_func is instr(26 to 31);
    alias instr_fp_func : dlx_fp_func is instr(21 to 25);
    alias instr_rs1 : dlx_reg_addr is instr(6 to 10);
    alias instr_rs2 : dlx_reg_addr is instr(11 to 15);
    alias instr_Itype_rd : dlx_reg_addr is instr(11 to 15);
    alias instr_Rtype_rd : dlx_reg_addr is instr(16 to 20);
    alias instr_immed16 : dlx_immed16 is instr(16 to 31);
    alias instr_immed26 : dlx_immed26 is instr(6 to 31);

    variable instr_opcode_num  : natural;
    variable instr_rr_func_num : natural;
    variable instr_fp_func_num : natural;
    variable rs1 : reg_index;
    variable rs2 : reg_index;
    variable Itype_rd : reg_index;
    variable Rtype_rd : reg_index;

  begin
    instr_opcode_num := sv_to_natural(instr_opcode);
    instr_rr_func_num := sv_to_natural(instr_rr_func);
    instr_fp_func_num := sv_to_natural(instr_fp_func);
    rs1 := sv_to_natural(instr_rs1);
    rs2 := sv_to_natural(instr_rs2);
    Itype_rd := sv_to_natural(instr_Itype_rd);
    Rtype_rd := sv_to_natural(instr_Rtype_rd);
    --
    if (instr_opcode /= op_rr_alu) and (instr_opcode /= op_fp_fn) then
      write(L, opcode_names(instr_opcode_num));
      write(L, ' ');
    end if;
    
    case instr_opcode is
      --------------------------------------------------------------------
      when op_rr_alu =>
        write(L, rr_func_names(instr_rr_func_num));
        write(L, ' ');
	case instr_rr_func is
          --------------------------------------------------------------------
          when rr_func_nop =>
            null;
          --------------------------------------------------------------------
          when rr_func_sll | rr_func_srl | rr_func_sra |
              rr_func_sequ | rr_func_sneu | rr_func_sltu |
              rr_func_sgtu | rr_func_sleu | rr_func_sgeu |
              rr_func_add | rr_func_addu | rr_func_sub | rr_func_subu |
              rr_func_and | rr_func_or | rr_func_xor |
              rr_func_seq | rr_func_sne | rr_func_slt |
              rr_func_sgt | rr_func_sle | rr_func_sge =>
            write_reg(L, Rtype_rd); write(L, string'(", "));
            write_reg(L, rs1); write(L, string'(", "));
            write_reg(L, rs2);
          --------------------------------------------------------------------
          when rr_func_movi2s =>
            write_special_reg(L, Rtype_rd); write(L, string'(", "));
            write_reg(L, rs1);
          --------------------------------------------------------------------
          when rr_func_movs2i =>
            write_reg(L, Rtype_rd); write(L, string'(", "));
            write_special_reg(L, rs1);
          --------------------------------------------------------------------
          when rr_func_lb | rr_func_lh | rr_func_lw | rr_func_lbu | rr_func_lhu =>
            write_reg(L, Rtype_rd);
 	    write(L, string'(", "));
            write_reg(L, rs2);
 	    write(L, '(');
            write_reg(L, rs1);
 	    write(L, ')');
          --------------------------------------------------------------------
           when rr_func_sb | rr_func_sh | rr_func_sw =>
             write_reg(L, rs2);
             write(L, '(');
 	     write_reg(L, rs1); 
             write(L, string'("), "));
             write_reg(L, Rtype_rd);
           --------------------------------------------------------------------
           when rr_func_lf | rr_func_ld =>
             write_freg(L, Rtype_rd);
 	     write(L, string'(", "));
             write_reg(L, rs2);
 	     write(L, '(');
             write_reg(L, rs1);
 	     write(L, ')');
             write(L, string'(" (instruction not implemented !)"));
            --------------------------------------------------------------------
           when rr_func_sf | rr_func_sd =>
	     write_reg(L, rs2);
             write(L, '(');
 	     write_reg(L, rs1);
             write(L, string'("), "));
             write_freg(L, Rtype_rd);
	     write(L, string'(" (instruction not implemented !)"));
           --------------------------------------------------------------------
          when rr_func_movf | rr_func_movd =>
            write_freg(L, Rtype_rd); write(L, string'(", "));
            write_freg(L, rs1);
	    write(L, string'(" (instruction not implemented !)"));
           --------------------------------------------------------------------
          when rr_func_movfp2i =>
            write_reg(L, Rtype_rd); write(L, string'(", "));
            write_freg(L, rs1);
	    write(L, string'(" (instruction not implemented !)"));
           --------------------------------------------------------------------
          when rr_func_movi2fp =>
            write_freg(L, Rtype_rd); write(L, string'(", "));
            write_reg(L, rs1);
	    write(L, string'(" (instruction not implemented !)"));
           --------------------------------------------------------------------
          when others =>
            write(L, string'(" (undefined instruction!)"));
        end case;

      --------------------------------------------------------------------
      when op_fp_fn =>
        write(L, fp_func_names(instr_fp_func_num));
        write(L, ' ');
	case instr_fp_func is
          when fp_func_addf | fp_func_subf | fp_func_multf | fp_func_divf |
              fp_func_addd | fp_func_subd | fp_func_multd | fp_func_divd |
              fp_func_mult | fp_func_div | fp_func_multu | fp_func_divu =>
            write_freg(L, Rtype_rd); write(L, string'(", "));
            write_freg(L, rs1); write(L, string'(", "));
            write_freg(L, rs2);
          when fp_func_cvtf2d | fp_func_cvtd2f =>
            write_freg(L, Rtype_rd); write(L, string'(", "));
            write_freg(L, rs1);
          when fp_func_cvtf2i | fp_func_cvtd2i =>
            write_reg(L, Rtype_rd); write(L, string'(", "));
            write_freg(L, rs1);
          when fp_func_cvti2f | fp_func_cvti2d =>
            write_freg(L, Rtype_rd); write(L, string'(", "));
            write_reg(L, rs1);
          when fp_func_eqf | fp_func_nef | fp_func_ltf |
              fp_func_gtf | fp_func_lef | fp_func_gef |
              fp_func_eqd | fp_func_ned | fp_func_ltd |
              fp_func_gtd | fp_func_led | fp_func_ged =>
            write_freg(L, rs1); write(L, string'(", "));
            write_freg(L, rs2);
	  --------------------------------------------------------------------
          when others =>
            write(L, string'(" (undefined instruction !)"));
        end case;
	write(L, string'(" (instruction not implemented !)"));
	
      --------------------------------------------------------------------
      when op_j  | op_jal =>
        write(L, sv_to_integer(instr_immed26));

      --------------------------------------------------------------------
      when op_beqz | op_bnez =>
        write_reg(L, rs1); write(L, string'(", "));
        write(L, sv_to_integer(instr_immed16));

      --------------------------------------------------------------------
      when op_bfpt | op_bfpf =>
        write(L, sv_to_integer(instr_immed16));
	write(L, string'(" (instruction not implemented !)"));
	
      --------------------------------------------------------------------
      when op_sll_i | op_srl_i | op_sra_i =>
        write_reg(L, Itype_rd); write(L, string'(", "));
        write_reg(L, rs1); write(L, string'(", "));
        write(L, sv_to_natural(instr_immed16(11 to 15)));

      --------------------------------------------------------------------
      when op_add_i | op_sub_i |
          op_seq_i | op_sne_i | op_slt_i | op_sgt_i | op_sle_i | op_sge_i =>
        write_reg(L, Itype_rd); write(L, string'(", "));
        write_reg(L, rs1); write(L, string'(", "));
        write(L, sv_to_integer(instr_immed16));

      --------------------------------------------------------------------
      when op_addu_i | op_subu_i | op_and_i | op_or_i | op_xor_i |
          op_sequ_i | op_sneu_i | op_sltu_i | op_sgtu_i | op_sleu_i | op_sgeu_i =>
        write_reg(L, Itype_rd); write(L, string'(", "));
        write_reg(L, rs1); write(L, string'(", "));
        write(L, sv_to_natural(instr_immed16));

      --------------------------------------------------------------------
      when op_lhi =>
        write_reg(L, Itype_rd); write(L, string'(", "));
        write(L, sv_to_natural(instr_immed16));

      --------------------------------------------------------------------
      when op_rfe =>
        null;

      --------------------------------------------------------------------
      when op_trap =>
        write(L, sv_to_natural(instr_immed26));

      --------------------------------------------------------------------
      when op_jr | op_jalr =>
        write_reg(L, rs1);

      --------------------------------------------------------------------
      when op_lb_i | op_lh_i | op_lw_i | op_lbu_i | op_lhu_i =>
        write_reg(L, Itype_rd);
 	write(L, string'(", "));
        write(L, sv_to_integer(instr_immed16));
 	write(L, '(');
        write_reg(L, rs1);
 	write(L, ')');
      --------------------------------------------------------------------
      when op_lf_i | op_ld_i =>
        write_freg(L, Itype_rd);
 	write(L, string'(", "));
        write(L, sv_to_integer(instr_immed16));
 	write(L, '(');
        write_reg(L, rs1);
 	write(L, ')');
        write(L, string'(" (instruction not implemented !)"));
      --------------------------------------------------------------------
      when op_sb_i | op_sh_i | op_sw_i =>
        write(L, sv_to_integer(instr_immed16));
        write(L, '(');
 	write_reg(L, rs1); 
        write(L, string'("), "));
        write_reg(L, Itype_rd);

      --------------------------------------------------------------------
      when op_sf_i | op_sd_i =>
	write(L, sv_to_integer(instr_immed16));
        write(L, '(');
 	write_reg(L, rs1);
        write(L, string'("), "));
        write_freg(L, Itype_rd);
	write(L, string'(" (instruction not implemented !)"));
	
      --------------------------------------------------------------------
      when others =>
        write(L, string'(" (undefined instruction !)"));
    end case;
    
  end write_instr;

--synopsys synthesis_on

end dlx_instructions;
