--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  *** SYNOPSYS synthesizable code (ver. 3.2.a) ***
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  Structural architecture of the datapath
--
--  file datapath-structural.vhd
--------------------------------------------------------------------------

architecture structural of datapath is

  component bus_const32
    port (q1 : out dlx_word;
          q2 : out dlx_word;
	  out_en1 : in std_logic;
	  out_en2 : in std_logic;
	  sel     : in std_logic_vector(0 to 1));
  end component;

  component word_mux2 
    port (in0, in1 : in dlx_word;
	  y   : out dlx_word;
	  sel : in std_logic);
  end component;
  
  component word_latch
    port (d : in  dlx_word;
	  q : out dlx_word;
	  latch_en : std_logic);
  end component;

  component word_reg_1e 
    port (d  : in dlx_word;
	  q  : out dlx_word;
	  latch_en : in std_logic;
	  out_en   : in std_logic);
  end component;

  component word_reg_1e1 
    port (d      : in dlx_word;
	  q1, q2 : out dlx_word bus;
	  latch_en : in std_logic;
	  out_en1  : in std_logic);
  end component;

  component mdr 
    port (d      : in dlx_word;
	  q1, q2 : out dlx_word;
	  latch_en   : in std_logic;
	  out_en1    : in std_logic;
	  shift_ctrl : in std_logic_vector(0 to 2);
          mar_ls2_in : in std_logic_vector(0 to 1));
  end component;

  component reg_file 
    port (addr_out1 : in  dlx_reg_addr;
	  q1        : out dlx_word;
	  addr_out2 : in  dlx_reg_addr;
	  q2        : out dlx_word;
	  addr_in   : in  dlx_reg_addr;
	  d         : in  dlx_word;
	  write_en  : in  std_logic);
  end component;

  component icr 
    port (d  : in dlx_word;                -- data in from dest_bus
          latch_en : in std_logic;         -- enable load from dest_bus
          q       : out dlx_word;          -- output to s_bus 
	  out_en  : in std_logic;          -- enable output to s_bus
	  --
          s_en    : in std_logic;          -- set s bit
	  ioc_en  : in std_logic;          -- set ioc bit
	  irra_en : in std_logic;          -- set irra bit
	  iav_en  : in std_logic;          -- set iav bit
	  dav_en  : in std_logic;          -- set dav bit
	  ovad_en : in std_logic;          -- set ovad bit
	  ovar_en : in std_logic;          -- set ovar bit
	  priv_en : in std_logic;          -- set priv bit
	  super   : out std_logic;         -- supervisor bit
	  --
	  intrpt_in : in dlx_nibble;       -- input from intrpt. port
	  intrpt_en : in std_logic;        -- enable load from intrpt. port   
	  intrpt    : out std_logic);      -- at least one masked interrupt active
  end component;

  component ir 
    port (d           : in dlx_word;
	  latch_en    : in std_logic;
	  ir_out      : out dlx_word;
	  immed_o1_en : in std_logic;
	  immed_out1  : out dlx_word;
	  immed_o2_en : in std_logic;
	  immed_out2  : out dlx_word;
	  immed_size  : in std_logic;                 -- '0'-> 16 bit / '1'-> 26 bit
	  immed_sign  : in std_logic);                -- '0'-> unsigned / '1' signed
  end component;

  component alu 
    port (s1       : in dlx_word;
	  s2       : in dlx_word;
          latch_en : in std_logic;
	  result   : out dlx_word;
	  func     : in dlx_nibble;
	  zero     : out std_logic;
	  negative : out std_logic;
	  overflow : out std_logic);
  end component;

  --
  -- internal busses
  --
  signal s1_bus   : dlx_word;
  signal s2_bus   : dlx_word;
  signal dest_bus : dlx_word;

  signal addr_mux_in0 : dlx_word;
  signal addr_mux_in1 : dlx_word;
  signal mdr_in      : dlx_word;

  signal reg_file_out1: dlx_word;
  signal reg_file_out2: dlx_word;
  signal reg_file_in  : dlx_word;
  --
  -- other lines
  --
  signal intrn_alu_overflow : std_logic;

begin
  
  dp_alu : alu                
    port map (s1 => s1_bus, s2 => s2_bus, latch_en => alu_latch_en,
              result => dest_bus, func => alu_func, zero => alu_zero,
 	      negative => alu_negative, overflow => intrn_alu_overflow);

  dp_reg_file : reg_file
    port map (addr_out1 => reg_addr_rs1, q1 => reg_file_out1,
              addr_out2 => reg_addr_rs2, q2 => reg_file_out2,
              addr_in   => reg_addr_rd,  d => reg_file_in,
              write_en => regf_wr_en);

  a_reg : word_reg_1e
    port map (d => reg_file_out1, q => s1_bus,
              latch_en => a_latch_en, out_en => a_out_en);

  b_reg : word_reg_1e
    port map (d => reg_file_out2, q => s2_bus,
              latch_en => b_latch_en, out_en => b_out_en);

  c_reg : word_latch
    port map (d => dest_bus, q => reg_file_in, latch_en => c_latch_en);

  pc_reg : word_reg_1e1
    port map (d => dest_bus, q1 => s2_bus, q2 => addr_mux_in0,
              latch_en => pc_latch_en, out_en1 => pc_out_en);

  instr_reg : ir
    port map (d => data_in, latch_en => ir_latch_en, ir_out => instr_out,
	      immed_o1_en => ir_immed_o1_en, immed_out1 => s1_bus,
	      immed_o2_en => ir_immed_o2_en, immed_out2 => s2_bus,
              immed_size => ir_immed_size, immed_sign => ir_immed_sign);
	      
  icr_reg : icr
    port map (d => dest_bus, q => s1_bus, latch_en => icr_latch_en,
              out_en => icr_out_en,
 	      s_en => icr_s_en, ioc_en => icr_ioc_en, irra_en => icr_irra_en,
	      iav_en => icr_iav_en, dav_en => icr_dav_en, 
              ovad_en => icr_ovad_en, ovar_en => icr_ovar_en,
 	      priv_en => icr_priv_en,
 	      super => icr_super, intrpt_in => icr_intrpt_in,
 	      intrpt_en => icr_intrpt_en, intrpt => icr_intrpt);

  iar_reg : word_reg_1e
    port map (d => dest_bus, q => s1_bus,
              latch_en => iar_latch_en, out_en => iar_out_en);

  tbr_reg : word_reg_1e
    port map (d => dest_bus, q => s1_bus,
              latch_en => tbr_latch_en, out_en => tbr_out_en);

  mar_reg : word_reg_1e1
    port map (d => dest_bus, q1 => s2_bus, q2 => addr_mux_in1,
              latch_en => mar_latch_en, out_en1 => mar_out1_en);

  addr_mux : word_mux2
    port map (in0 => addr_mux_in0, in1 => addr_mux_in1, y => addr_out,
              sel => addr_mux_sel);

  mdr_reg : mdr
    port map (d => mdr_in, q1 => s1_bus, q2 => data_out,
              latch_en => mdr_latch_en, out_en1 => mdr_out1_en,
	      shift_ctrl => mdr_sh_ctrl, mar_ls2_in => addr_mux_in1(30 to 31));

  mdr_mux : word_mux2
    port map (in0 => dest_bus, in1 => data_in, y => mdr_in,
              sel => mdr_mux_sel);

  bus_const: bus_const32
    port map (q1 => s1_bus, out_en1 => const_o1_en,
              q2 => s2_bus, out_en2 => const_o2_en,
	      sel => const_sel);

  alu_overflow <= intrn_alu_overflow;

  mar_adr_ls2 <= addr_mux_in1(30 to 31);
  mar_adr_msb <= addr_mux_in1(0);

end structural;




