--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  *** SYNOPSYS synthesizable code (ver. 3.2.a) ***
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  Structural architecture of the core of the DLX processor
--  (only signed instructions)
--  
--  (file core-structural.vhd)
--------------------------------------------------------------------------

architecture structural of core is

  component controller
  port (phi1, phi2 : in std_logic;                  -- two phase clock in
	reset : in std_logic;                       -- reset input
	halt  : in std_logic;                       -- halt input
	ready : in std_logic;                       -- mem. ready input
	instr_in : in dlx_word;                     -- instruction input
	alu_zero : in std_logic;                    -- alu zero bit
	alu_neg  : in std_logic;                    -- alu negative bit
	alu_ovfl : in std_logic;                    -- alu overflow bit
	intrpt   : in std_logic;                    -- interrupt active
	super    : in std_logic;                    -- supervisor bit 
	adr_msb : in std_logic;                     -- ms address bit
	adr_ls2 : in std_logic_vector(1 downto 0);  -- least sign. 2 addr. bits
        --
	-- reg. address outputs
	--
	rs1_out : out dlx_reg_addr;
	rs2_out : out dlx_reg_addr;	  
	rd_out  : out dlx_reg_addr;	  
        --
	-- control outputs
	--
	s1_enab    : out std_logic_vector(0 to 6);    -- select s1 source
	s2_enab    : out std_logic_vector(0 to 4);    -- select s2_source
	dest_enab  : out std_logic_vector(0 to 6);    -- select destination
	alu_op_sel : out std_logic_vector(0 to 3);    -- alu operation
	const_sel  : out std_logic_vector(0 to 1);    -- select const for s1
	rf_op_sel  : out std_logic_vector(0 to 1);    -- select reg file operation
	immed_sel  : out std_logic_vector(0 to 1);    -- select immediate from ir
	exc_enab   : out std_logic_vector(0 to 7);    -- enable set exception bit
	mem_ctrl   : out std_logic_vector(0 to 10);   -- memory control lines
	mdr_sh_ctrl : out std_logic_vector(0 to 2));  -- mdr shift control
  end component;

  component datapath 
  port(signal data_in      : in  dlx_word;           -- data in from pads
       signal data_out     : out dlx_word;           -- data_out to pads
       signal addr_out     : out dlx_word;           -- addr. out to pads
       signal instr_out    : out dlx_word;           -- instr. reg. content
       signal alu_zero     : out std_logic;                     
       signal alu_negative : out std_logic;
       signal alu_overflow : out std_logic;
       signal icr_intrpt   : out std_logic;          -- at least one masked intrpt.
       signal icr_super    : out std_logic;          -- supervisor bit in icr
       signal mar_adr_ls2  : out std_logic_vector(0 to 1); -- least 2 addr. bits
       signal mar_adr_msb  : out std_logic;          --  msb of mar
       --
       -- reg. addr. inputs
       --
       signal reg_addr_rs1 : in dlx_reg_addr;
       signal reg_addr_rs2 : in dlx_reg_addr;
       signal reg_addr_rd  : in dlx_reg_addr;
       --
       -- control inputs
       --
       signal a_latch_en  : in std_logic;         -- low active
       signal a_out_en    : in std_logic;         -- low active
       signal b_latch_en  : in std_logic;         -- low active
       signal b_out_en    : in std_logic;         -- low active
       signal c_latch_en  : in std_logic;         -- low active
       signal regf_wr_en  : in std_logic;         -- low active

       signal mar_latch_en : in std_logic;        -- low active
       signal mar_out1_en  : in std_logic;        -- low active
       signal mdr_latch_en : in std_logic;        -- low active
       signal mdr_out1_en  : in std_logic;        -- low active
       signal mdr_sh_ctrl  : in std_logic_vector(0 to 2); -- enab signal for shift op.

       signal pc_latch_en  : in std_logic;        -- low active
       signal pc_out_en    : in std_logic;        -- low active
       signal iar_latch_en : in std_logic;        -- low active
       signal iar_out_en   : in std_logic;        -- low active
       signal tbr_latch_en : in std_logic;        -- low active
       signal tbr_out_en   : in std_logic;        -- low active

       signal icr_out_en   : in std_logic;        -- low active
       signal icr_latch_en : in std_logic;        -- low active
       signal icr_s_en     : in std_logic;        -- enab set s bit (low act.)
       signal icr_ioc_en   : in std_logic;        -- enab set ioc bit (low act.)
       signal icr_irra_en  : in std_logic;        -- enab set irra bit(low act.)
       signal icr_iav_en   : in std_logic;        -- enab set iav bit (low act.)
       signal icr_dav_en   : in std_logic;        -- enab set dav bit (low act.)
       signal icr_ovad_en   : in std_logic;       -- enab set ovad bit(low act.)
       signal icr_ovar_en   : in std_logic;       -- enab set ovar bit(low act.)
       signal icr_priv_en   : in std_logic;       -- enab set priv bit (low act.)
       signal icr_intrpt_in : in dlx_nibble;      -- intrpt. input from pads
       signal icr_intrpt_en : in std_logic;       -- enable setting of intrpt bits 

       signal ir_latch_en    : in std_logic;      -- low active
       signal ir_immed_o1_en : in std_logic;      -- low active
       signal ir_immed_o2_en : in std_logic;      -- low active
       signal ir_immed_size  : in std_logic;      -- '0'-> 16 bit / '1'-> 26 bit
       signal ir_immed_sign  : in std_logic;      -- '0'-> unsign./ '1'-> sign.  

       signal alu_latch_en : in std_logic;        -- low active
       signal alu_func     : in dlx_nibble;

       signal addr_mux_sel : in std_logic;        -- '0' : pc->out/ '1' : mar->out
       signal mdr_mux_sel  : in std_logic;        -- '0' : dest->mdr/ '1' : mem->mdr
       
       signal const_o1_en : in std_logic;         -- enable constant for s1 (low act.)
       signal const_o2_en : in std_logic;         -- enable constant for s2 (low act.)
       signal const_sel   : in std_logic_vector(0 to 1));  -- select constant
  end component;

  --
  -- signals to connect
  --
  signal neg_phi1_in : std_logic;              -- negated phi1
  
  signal dp_instr_out  : dlx_word; 
  signal dp_alu_zero   : std_logic;                     
  signal dp_alu_neg    : std_logic;
  signal dp_alu_ovfl   : std_logic;
  signal dp_intrpt_out : std_logic;
  signal dp_super_out  : std_logic;
  signal dp_adr_msb_out : std_logic;
  signal dp_adr_ls2_out : std_logic_vector(0 to 1);

  signal c_s1_enab    : std_logic_vector(0 to 6);    -- select s1 source
  signal c_s2_enab    : std_logic_vector(0 to 4);    -- select s2_source
  signal c_dest_enab  : std_logic_vector(0 to 6);    -- select destination
  signal c_alu_op_sel : std_logic_vector(0 to 3);    -- alu operation
  signal c_const_sel  : std_logic_vector(0 to 1);    -- select const for s1
  signal c_rf_op_sel  : std_logic_vector(0 to 1);    -- select reg file operation
  signal c_immed_sel  : std_logic_vector(0 to 1);    -- select immediate from ir
  signal c_exc_enab   : std_logic_vector(0 to 7);    -- enable set exception bit
  signal c_mem_ctrl   : std_logic_vector(0 to 10);   -- memory control lines
  signal c_mdr_sh_ctrl: std_logic_vector(0 to 2);   -- mdr shift control

  signal c_rs1_out    : dlx_reg_addr;
  signal c_rs2_out    : dlx_reg_addr;  
  signal c_rd_out     : dlx_reg_addr;
  
begin       

  the_controller : controller
    port map (phi1 => phi1,
              phi2 => phi2,
 	      halt => halt,
 	      reset => reset,
	      ready => ready,
 	      instr_in => dp_instr_out,
              alu_zero => dp_alu_zero,
	      alu_neg => dp_alu_neg,
 	      alu_ovfl => dp_alu_ovfl,
	      adr_msb => dp_adr_msb_out,
	      adr_ls2 => dp_adr_ls2_out,
 	      intrpt => dp_intrpt_out,
	      super => dp_super_out,
	      rs1_out => c_rs1_out,
	      rs2_out => c_rs2_out,
	      rd_out => c_rd_out,
	      s1_enab => c_s1_enab,
 	      s2_enab => c_s2_enab,
  	      dest_enab => c_dest_enab,
 	      alu_op_sel => c_alu_op_sel,
	      const_sel => c_const_sel,
 	      rf_op_sel => c_rf_op_sel,
	      immed_sel => c_immed_sel,
 	      exc_enab => c_exc_enab,
 	      mem_ctrl => c_mem_ctrl,
	      mdr_sh_ctrl => c_mdr_sh_ctrl);
	      
  the_datapath : datapath
    port map (data_in => d_bus_in,
              --
	      data_out => d_bus_out,
	      addr_out => a_bus,
	      instr_out => dp_instr_out,
	      alu_zero => dp_alu_zero,
	      alu_negative => dp_alu_neg,
	      alu_overflow => dp_alu_ovfl,
	      icr_intrpt => dp_intrpt_out,
	      icr_super => dp_super_out,
	      mar_adr_msb => dp_adr_msb_out,
	      mar_adr_ls2 => dp_adr_ls2_out,
	      --
	      -- reg. addr. inputs
	      --
	      reg_addr_rs1 => c_rs1_out,
	      reg_addr_rs2 => c_rs2_out,
	      reg_addr_rd  => c_rd_out,
	      --
	      -- s1_enab(0 to 6) : select source for s1 bus
	      -- |a|icr|iar|tbr|mdr|immed|consta| (b/pc/mar not connected to s1 bus)
	      --
	      a_out_en => c_s1_enab(0),
	      icr_out_en => c_s1_enab(1),
	      iar_out_en => c_s1_enab(2),
	      tbr_out_en => c_s1_enab(3),
	      mdr_out1_en => c_s1_enab(4),
	      ir_immed_o1_en => c_s1_enab(5),
	      const_o1_en => c_s1_enab(6),
	      --
	      -- s2_enab(0 to 4) : select source for s2 bus
	      -- |b|immed|pc|mar|constb|  (a/mdr/icr/tbr/iar not connected to s2 bus)
	      --
	      b_out_en => c_s2_enab(0),
	      ir_immed_o2_en => c_s2_enab(1),  
	      pc_out_en => c_s2_enab(2),
	      mar_out1_en => c_s2_enab(3),
	      const_o2_en => c_s2_enab(4),       
	      --	      
	      -- dest_enab(0 to 6) : select destination from dest bus
	      -- |c|icr|iar|tbr|pc|mar|mdr|
	      --
	      c_latch_en => c_dest_enab(0),
	      icr_latch_en => c_dest_enab(1),
	      iar_latch_en => c_dest_enab(2),
	      tbr_latch_en => c_dest_enab(3),
	      pc_latch_en => c_dest_enab(4),
	      mar_latch_en => c_dest_enab(5),
	      mdr_latch_en => c_dest_enab(6),
	      --
	      -- alu_op_sel(0 to 3) : select alu operation
	      --
	      alu_func => c_alu_op_sel,
              alu_latch_en => neg_phi1_in,
	      --
	      -- const_sel(0 to 2) (select constant for s2 bus)
  	      --
	      const_sel => c_const_sel,
	      --
	      -- immed_sel(0 to 1) : select immediate type 
	      -- |size_imm|sign_imm|
	      --
 	      ir_immed_size => c_immed_sel(0),
	      ir_immed_sign => c_immed_sel(1),
	      --
	      -- rf_op_sel(0 to 1) : register-file operation
	      -- |ab<-rf|rf<-c|mux_sel|set31||
	      --
	      a_latch_en => c_rf_op_sel(0),
	      b_latch_en => c_rf_op_sel(0),
	      regf_wr_en => c_rf_op_sel(1), 
	      --
	      -- exc_enab(0 to 7) : switch input / enable setting of exception bits
	      -- |s|ioc|irra|iav|dav|ovad|ovar|priv|
	      --
	      icr_s_en => c_exc_enab(0),
	      icr_ioc_en => c_exc_enab(1),
	      icr_irra_en => c_exc_enab(2),
	      icr_iav_en => c_exc_enab(3),
	      icr_dav_en => c_exc_enab(4),
	      icr_ovad_en => c_exc_enab(5),
	      icr_ovar_en => c_exc_enab(6),
	      icr_priv_en => c_exc_enab(7),
	      icr_intrpt_in => intrpt,
	      icr_intrpt_en => phi2,
	      --
	      -- mem_ctrl(0 to 10) : all control signal for memory interface etc.
	      -- |ir_en|addr_mux|mdr_mux|enable(0 to 3)|rw|pad_d_inout|pad_out_en|error|
	      --
	      ir_latch_en => c_mem_ctrl(0),
 	      addr_mux_sel => c_mem_ctrl(1),
	      mdr_mux_sel => c_mem_ctrl(2),
	      --
	      -- mdr shift control inputs
	      --
	      mdr_sh_ctrl => c_mdr_sh_ctrl);

  --
  -- negate phi1 signal
  --
  neg_phi1_in <= not phi1;

  --
  -- connect pad control lines
  --
  enable <= c_mem_ctrl(3 to 6);
  rw <= c_mem_ctrl(7);
  pad_io_sw <= c_mem_ctrl(8);
  pad_out_en <= c_mem_ctrl(9);
  error <= c_mem_ctrl(10);

end structural;

