--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  *** SYNOPSYS synthesizable code (ver. 3.2.a) ***
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  Structural architecture of the controller
--  (decoder and instantiated state machine)
--
--  file controller-dataflow.vhd
--------------------------------------------------------------------------

architecture dataflow of controller is

  component fsm
  port (phi1 : in std_logic;                          -- phase one of clock
      	reset : in std_logic;                         -- reset input
	halt  : in std_logic;                         -- halt input
	ready : in std_logic;                         -- mem. ready input
	alu_zero : in std_logic;                      -- alu zero bit
	alu_neg  : in std_logic;                      -- alu negative bit
	alu_ovfl : in std_logic;                      -- alu overflow bit
	intrpt   : in std_logic;                      -- interrupt active
	super    : in std_logic;                      -- supervisor bit input
	priv_ins : in std_logic;                      -- priviledged instr flag
	dec_1_in : in fsm_states;                     -- input from instr. dec. 1
	dec_2_in : in fsm_states;                     -- input from instr. dec. 2
	adr_msb  : in std_logic;                      -- most significant adr. bit 
	adr_ls2  : in std_logic_vector(1 downto 0);   -- least sign. 2 adr. bits
	s1_enab    : out std_logic_vector(0 to 7);    -- select s1 source
	s2_enab    : out std_logic_vector(0 to 5);    -- select s2_source
	dest_enab  : out std_logic_vector(0 to 7);    -- select destination
	alu_op_sel : out std_logic_vector(0 to 3);    -- alu operation
	const_sel  : out std_logic_vector(0 to 1);    -- select const for s1
	rf_op_sel  : out std_logic_vector(0 to 2);    -- select reg file op.
	immed_sel  : out std_logic_vector(0 to 1);    -- select immed. from ir
	exc_enab   : out std_logic_vector(0 to 7);    -- enable set exc. bit
	mem_ctrl   : out std_logic_vector(0 to 8));   -- memory control lines
  end component;

  component ir_decode_1
  port (instr_in : in dlx_word;
	dec1_out : out fsm_states);
  end component;	

  component ir_decode_2
  port (instr_in : in dlx_word;
	dec2_out : out fsm_states;
	mdr_ctrl_out : out std_logic_vector(0 to 2));
  end component;	
  
  component ir_decode_3 
  port (instr_in : in dlx_word;
        s1_sel        : out std_logic;
	rd_s2_adr_sel : out std_logic;
	rd_adr_set31  : out std_logic;
	priv_ins_out  : out std_logic);
  end component;
	
  signal decode_1_out : fsm_states;
  signal decode_2_out : fsm_states;
  
  signal intrn_s1_enab   : std_logic_vector(0 to 7);
  signal intrn_s2_enab   : std_logic_vector(0 to 5);
  signal intrn_dest_enab : std_logic_vector(0 to 7);
  signal gate_dest_enab  : std_logic_vector(0 to 6);
  signal intrn_exc_enab  : std_logic_vector(0 to 7);  
  signal intrn_rfop_sel  : std_logic_vector(0 to 2);
  signal intrn_mem_ctrl  : std_logic_vector(0 to 8);

  signal rd_adr_set31  : std_logic;
  signal rd_s2_adr_sel : std_logic;
  signal s1_sel     : std_logic;
  signal priv_instr : std_logic;
  
begin
  --
  -- instantiation of the fsm
  --
  state_machine : fsm
    port map (phi1 => phi1, reset => reset, halt => halt, ready => ready,
              alu_zero => alu_zero, alu_neg => alu_neg, alu_ovfl => alu_ovfl,
 	      intrpt => intrpt, super => super, priv_ins => priv_instr,
              dec_1_in => decode_1_out, dec_2_in => decode_2_out,
	      adr_msb => adr_msb, adr_ls2 => adr_ls2,
 	      s1_enab => intrn_s1_enab, s2_enab => intrn_s2_enab,
	      dest_enab => intrn_dest_enab, exc_enab => intrn_exc_enab,
 	      alu_op_sel => alu_op_sel, const_sel => const_sel,
	      rf_op_sel => intrn_rfop_sel, immed_sel => immed_sel,
 	      mem_ctrl => intrn_mem_ctrl);

  --
  -- instruction decoder 1
  -- decodes most instructions except load/store
  --
  decode_1: ir_decode_1
    port map (instr_in => instr_in, dec1_out => decode_1_out); 
  --
  -- instruction decoder 2
  -- decodes exact load/store instruction
  --
  decode_2: ir_decode_2
    port map (instr_in => instr_in, dec2_out => decode_2_out,
              mdr_ctrl_out => mdr_sh_ctrl); 
  --
  -- instruction decoder 3
  -- set signals for the generation of rd, s1(X), s2(Y) 
  --
  decode_3: ir_decode_3
    port map (instr_in => instr_in, s1_sel => s1_sel,
              rd_s2_adr_sel => rd_s2_adr_sel, rd_adr_set31 => rd_adr_set31,
	      priv_ins_out => priv_instr); 
  --
  -- generate register file addresses
  --
  rs1_out <= instr_in(6 to 10);
  rs2_out <= instr_in(16 to 20)                                     -- switched when store Rtype
	       when rd_s2_adr_sel = '1' and intrn_rfop_sel(1) ='0' else
             instr_in(11 to 15);                                    -- normal case
  rd_out <= instr_in(16 to 20)
              when rd_s2_adr_sel = '1' and rd_adr_set31 = '0' else  -- Rtype dest.
            instr_in(11 to 15)
 	      when rd_s2_adr_sel = '0' and rd_adr_set31 = '0' else  -- Itype dest.
	    (others => '1');                                        -- link dest. 
  --
  -- generate s1_icr, s1_iar, and s1_tbr out_en signals (low active)
  -- connect other s1 out_en signals (low active)
  --
  s1_enab(0) <= intrn_s1_enab(0);                                 -- |a|
  s1_enab(1) <= intrn_s1_enab(1) and
                ((s1_sel nand instr_in(10)) or intrn_s1_enab(4)); -- |icr|
  s1_enab(2) <= intrn_s1_enab(2) and
                ((s1_sel nand instr_in(9)) or intrn_s1_enab(4));  -- |iar|
  s1_enab(3) <= intrn_s1_enab(3) and
                ((s1_sel nand instr_in(8)) or intrn_s1_enab(4));  -- |tbr|
  s1_enab(4 to 6) <= intrn_s1_enab(5 to 7);                       -- |mdr|immed|consta|
  --
  -- generate s2_b and s2_immed out_en signals (low active)
  -- connect other s2 out_en signals (low active)
  --
  s2_enab(0) <= intrn_s2_enab(0) and
                (not rd_s2_adr_sel or intrn_s2_enab(2));      -- |b|
  s2_enab(1) <= intrn_s2_enab(1) and
                (rd_s2_adr_sel or intrn_s2_enab(2));          -- |immed|
  s2_enab(2 to 4) <= intrn_s2_enab(3 to 5);                   -- |pc|mar|constb|
  --
  -- generate dest_enab signals icr, iar, tbr (low active)
  -- connect other dest_enab signals (low active)
  -- gate all dest_enab signals (low active) with phi2 
  --       
  gate_dest_enab(0) <= intrn_dest_enab(0);                                 -- |c|
  gate_dest_enab(1) <= intrn_dest_enab(1) and  
                       ((s1_sel nand instr_in(20)) or intrn_dest_enab(4)); -- |icr|
  gate_dest_enab(2) <= intrn_dest_enab(2) and 
                       ((s1_sel nand instr_in(19)) or intrn_dest_enab(4)); -- |icr|
  gate_dest_enab(3) <= intrn_dest_enab(3) and 
                       ((s1_sel nand instr_in(18)) or intrn_dest_enab(4)); -- |icr|
  gate_dest_enab(4 to 6) <= intrn_dest_enab(5 to 7);                       -- |pc|mar|mdr|

  gate_dest : for i in 0 to 6 generate
    dest_enab(i) <= gate_dest_enab(i) or not phi2;
  end generate gate_dest;
  --
  -- gate intrn_exc_enab (low_active) with phi2) 
  --       
  gate_exc : for i in 0 to 7 generate
    exc_enab(i) <= intrn_exc_enab(i) or not phi2;
  end generate gate_exc;
  --
  -- gate mem_ctrl(0) = ir_en (low active) with phi2
  -- decode enable signals (enable(0) = mem_ctrl(3) ... enable(3) = mem_ctrl(6))
  -- connect all others
  --       
  mem_ctrl(0) <= intrn_mem_ctrl(0) or not phi2;
  mem_ctrl(1 to 2) <= intrn_mem_ctrl(1 to 2);
  mem_ctrl(3 to 6) <= "1111" when intrn_mem_ctrl(3 to 4) = "11" else
                      "1100" when intrn_mem_ctrl(3 to 4) = "10" and
		                  adr_ls2(1) = '0' else
		      "0011" when intrn_mem_ctrl(3 to 4) = "10" and
		                  adr_ls2(1) = '1' else
                      "1000" when intrn_mem_ctrl(3 to 4) = "01" and
		                  adr_ls2(1 downto 0) = "00" else
    		      "0100" when intrn_mem_ctrl(3 to 4) = "01" and
		                  adr_ls2(1 downto 0) = "01" else
		      "0010" when intrn_mem_ctrl(3 to 4) = "01" and
		                  adr_ls2(1 downto 0) = "10" else
		      "0001" when intrn_mem_ctrl(3 to 4) = "01" and
 		                  adr_ls2(1 downto 0) = "11" else
		      "0000";
  mem_ctrl(7 to 10) <= intrn_mem_ctrl(5 to 8);
  --
  -- gate rf_op_sel(1) = regf_write_enable (low active) with phi2
  -- gate rf_op_sel(0) = ab_latch_enable (low active) with phi2
  -- (intrn_rf_op_sel(1) is used only in the controller)
  --
  rf_op_sel(1) <= intrn_rfop_sel(2) or not phi2; -- |rf_c|
  rf_op_sel(0) <= intrn_rfop_sel(0) or not phi2; -- |ab_rf|

end dataflow;

