--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  Package body for the procedures used in the behavioural
--  descriptions of DLX processor
--  (for the architecture behaviour_4)
--  
--  (file behav_procedures-body.vhd)
--------------------------------------------------------------------------

PACKAGE BODY behav_procedures_4 IS

  -----------------------------------------------------------------
  -- procedure for reading from the data bus 
  -----------------------------------------------------------------
    PROCEDURE bus_read (address    : IN dlx_address;
                        data_width : IN mem_ACCESS_TYPE;
			data       : OUT dlx_word;
			--
			INT           : INOUT dlx_nibble;
			signal intrpt : IN dlx_nibble;
			--
			signal phi1, phi2 : IN std_logic;
                        signal addr_bus   : OUT dlx_address;
			signal data_bus   : INOUT dlx_word;
			signal reset      : IN std_logic;
			signal ready      : IN std_logic;
			signal rw         : OUT std_logic;
			signal enable     : OUT dlx_nibble;
			--
			tpd_behav : IN time) IS
    BEGIN
      --
      -- place the address on the address bus after the rising edge of phi2
      --
      WAIT UNTIL phi2 = '1';
      IF reset = '1' THEN
 	RETURN;
      END IF;      
      intrpt_check(phi2, intrpt, INT);           -- check for interrupts
      addr_bus <= address AFTER tpd_behav; 
      --
      -- set ENABLE and RW after the rising edge of phi1
      --
      WAIT UNTIL phi1 = '1';
      IF reset = '1' THEN 
        RETURN;
      END IF;
      rw <= '1' AFTER tpd_behav;
      case data_width is
	when word   => enable <= "1111" after tpd_behav;
	when half   =>
	  case address(1) is
	    when '0' => enable <= "1100" after tpd_behav;
	    when others => enable <= "0011" after tpd_behav;
 	  end case;
	when others =>
 	  case address(1 downto 0) is
	    when "00" => enable <= "1000" after tpd_behav;
	    when "01" => enable <= "0100" after tpd_behav;
	    when "10" => enable <= "0010" after tpd_behav;
	    when others => enable <= "0001" after tpd_behav;
  	  end case;
      end case ;
      --
      -- wait until data ready while checking for interrupts and reset
      --
      LOOP 
	wait until phi2 = '1';                   -- wait until rising edge of phi2
	intrpt_check(phi2, intrpt, INT);         -- check for interrupts
        WAIT UNTIL phi2 = '0';                   -- wait until falling edge of phi2
        EXIT WHEN ready = '1' OR reset = '1';
      END LOOP;
      data := TO_UX01(data_bus);                 -- register content can't be 'Z'
      --
      -- inactivate lines after rising edge of phi1
      --
      WAIT UNTIL phi1 = '1';
      IF reset = '1' THEN 
        RETURN;
      END IF;
      data_bus <= (OTHERS => 'Z') AFTER tpd_behav;
      enable <= "0000" after tpd_behav;
      rw <= '0' AFTER tpd_behav;
    END bus_read;

  -----------------------------------------------------------------
  -- procedure for writing on the data bus 
  -----------------------------------------------------------------
    procedure bus_write (address    : IN  dlx_address;
                         data_width : IN  mem_access_type;
                         data       : IN  dlx_word;
			 --
			 INT           : INOUT dlx_nibble;
			 signal intrpt : IN dlx_nibble;
			 --
			 signal phi1, phi2 : IN std_logic;
                         signal addr_bus   : OUT dlx_address;
			 signal data_bus   : INOUT dlx_word;
			 signal reset      : IN std_logic;
			 signal ready      : IN std_logic;
			 signal rw         : OUT std_logic;
			 signal enable     : OUT dlx_nibble;
			 --
			 tpd_behav : IN time) IS
    begin
      --
      -- place the address on the address bus after the rising edge of phi2
      --
      wait until phi2 = '1';
      if reset = '1' then
 	return;
      end if;      
      intrpt_check(phi2, intrpt, INT);           -- check for interrupts
      addr_bus <= address after tpd_behav; 
      --
      -- place data on the data bus after rising edge of phi2 
      --
      wait until phi2 = '1';
      if reset = '1' then
 	return;
      end if;
      intrpt_check(phi2, intrpt, INT);           -- check for interrupts
      data_bus <= data after tpd_behav;
      --
      -- set ENABLE and RW after the rising edge of phi1 & check for interrupts
      --
      wait until phi1 = '1';
      if reset = '1' then 
        return;
      end if;
      rw <= '0' after tpd_behav;
      case data_width is
	when word   => enable <= "1111" after tpd_behav;
	when half   =>
	  case address(1) is
	    when '0' => enable <= "1100" after tpd_behav;
	    when others => enable <= "0011" after tpd_behav;
 	  end case;
	when others =>
 	  case address(1 downto 0) is
	    when "00" => enable <= "1000" after tpd_behav;
	    when "01" => enable <= "0100" after tpd_behav;
	    when "10" => enable <= "0010" after tpd_behav;
	    when others => enable <= "0001" after tpd_behav;
  	  end case;
      end case ;
      --
      -- wait until data accepted while checking for interrupts and reset
      --
      LOOP 
	wait until phi2 = '1';                   -- wait until rising edge of phi2
	intrpt_check(phi2, intrpt, INT);         -- check for interrupts
        WAIT UNTIL phi2 = '0';                   -- wait until falling edge of phi2
        EXIT WHEN ready = '1' OR reset = '1';
      END LOOP;
      --
      -- inactivate lines after rising edge of phi1
      --
      wait until phi1 = '1';
      if reset = '1' then 
        return;
      end if;
      data_bus <= (others => 'Z') after tpd_behav;
      enable <= "0000" after tpd_behav;
      rw <= '0' after tpd_behav;
    end bus_write;

  -----------------------------------------------------------------
  -- procedure for checking the interrupt lines 
  -----------------------------------------------------------------
    procedure intrpt_check (signal phi2   : IN std_logic;
                            signal intrpt : IN dlx_nibble;
                            INT : INOUT dlx_nibble) IS
    begin
      --
      -- check for active interrupt line(s)
      --
      if phi2 = '1' then
        INT := INT or intrpt;
      end if;
    end intrpt_check;
    
  -----------------------------------------------------------------
  -- procedure for checking the halt signal and freezing the processor 
  -----------------------------------------------------------------
    procedure check_halt (signal halt     : IN std_logic;
                          signal addr_bus : OUT dlx_address;
			  signal data_bus : INOUT dlx_word;
			  signal rw       : OUT std_logic;
			  signal enable   : OUT dlx_nibble;
			  --
			  DEBUG     : IN boolean;
			  tpd_behav : IN time) IS
      variable L : line;
    begin
      if halt = '1' then
        if DEBUG then
          write(L, string'(">>> halt detected, freezing processor state..."));
          writeline(output, L);
        end if;
	--
        -- all outputs to high impedance
        --
        addr_bus <= (others => 'Z') after tpd_behav;
        data_bus <= (others => 'Z') after tpd_behav;
        enable <= "ZZZZ" after tpd_behav;
        rw <= 'Z' after tpd_behav;
        --
        -- freeze processsor
        --
        wait until halt = '0';
	
        if DEBUG then
          write(L, string'(">>> halt disactivated, resuming ..."));
          writeline(output, L);
        end if;
      end if;
    end check_halt;
    
    -----------------------------------------------------------------
    -- procedure for checking errors and exceptions 
    -----------------------------------------------------------------
    procedure exc_check (to_check : in std_logic;
                         p_exc : inout boolean) is
    begin
      --
      -- check for errors and exceptions
      --
      if to_check = '1' then               -- exception bit set to 1 ?
	p_exc := true;                   -- -> exception
      end if;
    end exc_check;

    -----------------------------------------------------------------
    -- procedure for checking priviledge violations 
    -----------------------------------------------------------------
    procedure priv_check (adr_msb : in std_logic;
                          s_bit : in std_logic;
                          PRIV : inout std_logic;
                          p_exc : inout boolean) is
    begin
      --
      -- check priviledge violation
      --
      if adr_msb = '0' and s_bit = '0' then  -- msb of address = '0' ?
	PRIV := '1';                         --  and not in supervisor mode ?
        p_exc := true;
      end if;
    end priv_check;
    
    -----------------------------------------------------------------
    -- procedure for checking address alignment violations 
    -----------------------------------------------------------------
    procedure adr_check (adr_ls2 : in std_logic_vector(1 downto 0);
                         data_width : mem_access_type;
                         to_set : inout std_logic;
                         p_exc : inout boolean) is
    begin
      case data_width is
	when word =>
	  --
          -- check the last two bits for being zero
	  --
	  if adr_ls2 /= "00" then
	    to_set := '1';                 -- IAV or DAV
	    p_exc := true;               -- -> exception 
	  end if; 
        when half =>
	  --
          -- check the last bit for being zero
	  --
	  if adr_ls2(0) /= '0' then
	    to_set := '1';                 -- IAV or DAV
	    p_exc := true;               -- -> exception 
	  end if;  
	when byte =>
	  --
	  -- do nothing
	  --
	  null;
      end case;
    end adr_check;

END behav_procedures_4;




